#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# This widget has to remain somewhat flexible, so that we can still use
# API elements such as:
#
#   ::Gtk::MiniClock.widget {{
#     make_bold:        true,
#     use_this_colour: :olive
#   }}
#
# =========================================================================== #    
# require 'gtk_paradise/widgets/shared_code/mini_clock/mini_clock_module.rb'
# include ::Gtk::MiniClockModule
# =========================================================================== #
# For ruby-gtk3 use the following require line:
#
#   require 'gtk_paradise/widgets/gtk3/mini_clock/mini_clock.rb'
#
# =========================================================================== #
module Gtk

module MiniClockModule

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === MAIN_FONT_SIZE
  # ========================================================================= #
  MAIN_FONT_SIZE = 18

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH  = 140

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 50

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'A simple mini-clock'

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      i           = nil,
      run_already = true,
      &block
    )
    super(:vertical)
    reset
    # ======================================================================= #
    # === Handle blocks given next
    # ======================================================================= #
    if block_given?
      yielded = yield
      # ===================================================================== #
      # === Handle Hashes first
      # ===================================================================== #
      if yielded.is_a? Hash
        # =================================================================== #
        # === :make_bold
        # =================================================================== #
        if yielded.has_key? :make_bold
          set_make_bold(yielded.delete(:make_bold))
        end
        # =================================================================== #
        # === :use_this_colour
        # =================================================================== #
        if yielded.has_key? :use_this_colour
          set_use_this_colour(yielded.delete(:use_this_colour))
        end
      else
        # =================================================================== #
        # Handle Symbols past this point here
        # =================================================================== #
        case yielded
        # =================================================================== #
        # === :do_not_show_the_seconds
        # =================================================================== #
        when :do_not_show_the_seconds
          @show_the_seconds = false
        # =================================================================== #
        # === :make_bold
        # =================================================================== #
        when :make_bold,
             :bold
          set_make_bold(true)
        end
      end
    end
    run if run_already
  end

  # ========================================================================= #
  # === set_make_bold
  # ========================================================================= #
  def set_make_bold(i = true)
    @make_bold = i
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, infer_the_namespace]
    title_width_height(TITLE, WIDTH, HEIGHT)
    # ======================================================================= #
    # === @make_bold
    #
    # Whether to have the clock be bold.
    # ======================================================================= #
    @make_bold = true
    # ======================================================================= #
    # === @use_this_colour
    # ======================================================================= #
    @use_this_colour = :steelblue
    # ======================================================================= #
    # === @show_the_seconds
    #
    # If this is true then we will also display the number of seconds.
    #
    # By default we will show the number of seconds.
    # ======================================================================= #
    @show_the_seconds = true
    use_gtk_paradise_project_css_file
    infer_the_size_automatically
  end

  # ========================================================================= #
  # === set_use_this_colour
  # ========================================================================= #
  def set_use_this_colour(i)
    @use_this_colour = i.to_sym
    if @label
      apply_the_colours_to_the_main_label # And do this since as of August 2022.
    end
  end; alias use_this_colour= set_use_this_colour # === use_this_colour=

  # ========================================================================= #
  # === create_the_main_label
  # ========================================================================= #
  def create_the_main_label
    # ======================================================================= #
    # === @label
    # ======================================================================= #
    @label = label
    @label.hint = 'This small widget will show the current (local) time.'
  end

  # ========================================================================= #
  # === create_skeleton                                          (create tag)
  # ========================================================================= #
  def create_skeleton
    create_the_main_label
    update_current_time # Set the time at once.
    determine_the_main_font_size
    @alignment = gtk_alignment(0.25, 0.25, 0.8, 0.2)
    @alignment.add(@label)
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    maximal(@alignment, 1)
  end

  # ========================================================================= #
  # === apply_the_colours_to_the_main_label
  # ========================================================================= #
  def apply_the_colours_to_the_main_label
    @label.set_use_this_colour(@use_this_colour)
  end

  # ========================================================================= #
  # === determine_the_main_font_size
  # ========================================================================= #
  def determine_the_main_font_size(i = MAIN_FONT_SIZE)
    @label.set_font_size(i)
  end

  # ========================================================================= #
  # === set_the_font_of_the_main_label
  # ========================================================================= #
  def set_the_font_of_the_main_label(i)
    @label.use_this_font = i
  end

  # ========================================================================= #
  # === update_label_continually
  #
  # This method will update the current time constantly.
  # ========================================================================= #
  def update_label_continually
    Thread.new {
      loop {
        sleep 1
        update_current_time
      }
    }
  end

  # ========================================================================= #
  # === update_current_time
  # ========================================================================= #
  def update_current_time
    if @show_the_seconds
      the_current_time = return_current_time
    else
      the_current_time = return_current_time_hh_mm_only
    end
    if @make_bold
      @label.set_markup(
        "<span weight=\"bold\">#{the_current_time}</span>" 
      )
    else
      @label.set_text(the_current_time)
    end
    apply_the_colours_to_the_main_label
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    create_skeleton_then_connect_skeleton
    update_label_continually
  end

  # ========================================================================= #
  # === Gtk::MiniClock[]
  # ========================================================================= #
  def self.[](i = ARGV)
    new(i)
  end

  # ========================================================================= #
  # === Gtk::MiniClockModule.widget
  # ========================================================================= #
  def self.widget(&block)
    ::Gtk::MiniClock.new(&block)
  end; self.instance_eval { alias return_clock_widget widget } # === Gtk.return_clock_widget

  # ========================================================================= #
  # === Gtk::MiniClockModule.run
  # ========================================================================= #
  def self.run
    require 'gtk_paradise/run'
    _ = Gtk::MiniClock.new(ARGV)
    r = ::Gtk.run
    r.set_size_request(_.width?, _.height?)
    r.signal_connect(:delete_event) { |_widget| Gtk.main_quit }
    r << _
    r.automatic_title
    r.top_left_then_run
  end

end

# =========================================================================== #
# === Gtk.return_simple_clock_widget
# =========================================================================== #
def self.return_simple_clock_widget(&block)
  ::Gtk::MiniClockModule.widget(&block)
end; self.instance_eval { alias clock                      return_simple_clock_widget } # === Gtk.clock
     self.instance_eval { alias return_clock               return_simple_clock_widget } # === Gtk.return_clock
     self.instance_eval { alias return_simple_clock_widget return_simple_clock_widget } # === Gtk.return_simple_clock_widget
     self.instance_eval { alias return_clock_widget        return_simple_clock_widget } # === Gtk.return_clock_widget
     self.instance_eval { alias mini_clock                 return_simple_clock_widget } # === Gtk.mini_clock
  
end