#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::Tabby::TabbyModule
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/tabby/tabby_module.rb'
# include ::Gtk::Tabby::TabbyModule
# =========================================================================== #
module Gtk

class Tabby

module TabbyModule

  include ::Gtk::BaseModule

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Gtk::Tabby'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = 1400

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 900

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize
    super(:horizontal)
    reset
    hbox = gtk_hbox
    hbox.spacing = 24 # add 24 pixel
    # Das GTK Frame. Das hat unsere table.
    info_frame = gtk_frame(' File info ')
    info_frame.set_width_request(180)

    table = ::Gtk::Table.new( 2, 2, false)
    table.set_border_width(12)
    table.set_row_spacings(5)
    table.set_column_spacings(10)
    add_quit_button(table)
    info_frame.add(table) # add table to our frame

    @size_group = gtk_size_group(:horizontal)
    # add_row is defined below.
    add_row(table, 0, @size_group, 'Foreground')
    hbox.pack_minimal(info_frame, 10)

    @vbox = gtk_vbox
    @notebook = gtk_notebook
    @vbox.pack_maximal( @notebook, 0)
    hbox.pack_maximal( @vbox, 0)
    add(hbox)
    # Gtk::Tabby::View is a special class.
    ::Gtk::Tabby::View.new(@notebook, '/')
    show_all
  end

  # ========================================================================= #
  # === reset
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    width_height(WIDTH, HEIGHT)
    use_gtk_paradise_project_css_file
    infer_the_size_automatically
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    2
  end

  # ========================================================================= #
  # === add_row
  #
  # Add to our size group?
  # ========================================================================= #
  def add_row(
      table, row, size_group, label_text
    )
    label = gtk_label # (label_text, true)
    label.align_left
    markup = '<span style="italic" weight="bold">'+label_text+'</span>'
    label.set_markup(markup)
    if use_gtk2?
      table.attach(
        label, 0, 1, row, row + 1, Gtk::EXPAND | Gtk::FILL, 0, 0, 0
      )
    elsif use_gtk3?
      table.attach(
        label, 0, 1, row, row + 1, Gtk::AttachOptions::EXPAND | Gtk::AttachOptions::FILL, 0, 0, 0
      )
    end

    label = gtk_label
    label.set_markup('<small>Gtk Label als markup</small>')
    table.attach(label,1, 2, row, row + 1,0, 0, 0, 0)
  end

  # ========================================================================= #
  # === add_quit_button
  # ========================================================================= #
  def add_quit_button(table)
    # Der quit button.
    @button_quit = gtk_button('_Quit')
    @button_quit.on_clicked { Gtk.main_quit }
    table.attach_defaults(@button_quit, 0, 1, 0, 1)
  end

  # ========================================================================= #
  # === tree_select
  # ========================================================================= #
  def tree_select( tree, move )
    tree.move_cursor(::Gtk::MOVEMENT_DISPLAY_LINES, move)
    tree.selection.select_path(tree.cursor[0])
  end

  # ========================================================================= #
  # === tree_edit
  #
  # We use xterm here.
  # ========================================================================= #
  def tree_edit(this_file)
    `xterm -e vim #{this_file}`
  end

  # ========================================================================= #
  # === tree_scroll
  # ========================================================================= #
  def tree_scroll( tree, move )
    tree.move_cursor(::Gtk::MOVEMENT_PAGES, move)
    tree.selection.select_path(tree.cursor[0])
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    12
  end

  # ========================================================================= #
  # === favicon?
  # ========================================================================= #
  def favicon?
    :tabble
  end

  # ========================================================================= #
  # === Gtk::TabbyModule.run
  # ========================================================================= #
  def self.run
    require 'gtk_paradise/run'
    _ = ::Gtk::Tabby.new
    r = ::Gtk.run
    r << _
    r.automatic_size
    r.top_left_then_run
  end

end; end; end