#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::Tabby::ViewModule
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/tabby/view_module.rb'
# include ::Gtk::Tabby::ViewModule
# =========================================================================== #
module Gtk

class Tabby

module ViewModule

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === NAMESPACE
  # ========================================================================= #
  NAMESPACE = inspect

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Gtk::ViewModule'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = 1200

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 1000

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(notebook, path)
    set_path(path)
    reset
    @notebook = notebook
    create_skeleton
    tree_view_column = ::Gtk::TreeViewColumn.new( 
                       'FileListing', @render, {text: 0 } )
    @tree_view.append_column(tree_view_column)
    create_scrolled_window
    hbox = gtk_hbox
    # Die @combo_box zeigt hin zum base dir.
    hbox.minimal(@combo_box, 0)
    vbox = gtk_vbox
    vbox.minimal(hbox, 0)
    vbox.maximal(@scrolled_window, 0)
    vbox.show_all
    # ======================================================================= #
    # Add the vbox to the notebook. Title: FirstRow
    # ======================================================================= #
    @notebook.append_page(vbox, gtk_label( '_FirstRow',  true))
    load_file_path
  end

  # ========================================================================= #
  # === reset
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, NAMESPACE]
    set_use_this_font(:dejavu_condensed_25)
    # ======================================================================= #
    # === @title
    # ======================================================================= #
    @title = 'tabby - a file browser'
    # ======================================================================= #
    # === @selected
    # ======================================================================= #
    @selected   = nil
    # ======================================================================= #
    # === @dir
    # ======================================================================= #
    @dir        = nil
    # ======================================================================= #
    # === @is_loading
    # ======================================================================= #
    @is_loading = false
    # ======================================================================= #
    # === @width
    # ======================================================================= #
    set_width(WIDTH)
    # ======================================================================= #
    # === @height
    # ======================================================================= #
    set_height(HEIGHT)
    enable_css if respond_to? :enable_css
    increase_font_size
  end

  # ========================================================================= #
  # === set_path
  # ========================================================================= #
  def set_path(i)
    @path = i
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    2
  end

  # ========================================================================= #
  # === create_tree_view
  # ========================================================================= #
  def create_tree_view
    @tree_view = gtk_tree_view(::Gtk::ListStore.new(String))
    @tree_view.enable_search = false
    @tree_view.selection.mode = Gtk::SELECTION_SINGLE
    # mouse button release
    @tree_view.signal_connect(:button_release_event) { |widget, event|
      if event.event_type == Gdk::Event::BUTTON_RELEASE and event.button == 2 #
        e 'Mouse putton release.'
        View.new( @notebook, @path+get_selected().to_s )
        @notebook.set_page(-1)
      end
    }
    # mouse button press event - double click
    @tree_view.on_button_press_event { |widget, event|
      if event.event_type == Gdk::Event::BUTTON2_PRESS and event.button == 1
        e 'MouseButton: Doppelklick'
        load_selected # We will now get what we selected 
      end
    }
    @tree_view.on_key_press_event { |widget, event|
      e 'key press event ....'
      if Gdk::Window::ModifierType::CONTROL_MASK == event.state & Gdk::Window::CONTROL_MASK
        case event.keyval
        when Gdk::Keyval::GDK_u # ctrl-u, page up
          tree_scroll(@tree_view, -1)
        when Gdk::Keyval::GDK_d # ctrl-d, page down
          tree_scroll(@tree_view,  1)
        end
      else
        case event.keyval
        when Gdk::Keyval::GDK_Return,
             Gdk::Keyval::GDK_3270_Enter, 
             Gdk::Keyval::GDK_ISO_Enter, 
             Gdk::Keyval::GDK_KP_Enter
          load_selected
        when Gdk::Keyval::GDK_e # e
          begin
            file = @path+'/'+widget.selection.selected[0]
          rescue
            file = nil
          end
          tree_edit(file) unless file.nil?
        when Gdk::Keyval::GDK_j, Gdk::Keyval::GDK_J then # j
          tree_select( @tree_view, 1)
        when Gdk::Keyval::GDK_k, Gdk::Keyval::GDK_K then # k
          tree_select( @tree_view, -1)
        when Gdk::Keyval::GDK_q # q
          Gtk.main_quit
        end
      end
    }
  end

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
    create_combo_box
    create_renderer
    create_tree_view # Create a tree view next.
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    Thread.abort_on_exception = true
    add(@box_xyz)
  end

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    create_skeleton
    connect_skeleton
  end

  # ========================================================================= #
  # === create_renderer
  #
  # This will colourize the cell-renderer a bit.
  # ========================================================================= #
  def create_renderer
    @render = gtk_cell_renderer_text
    @render.background = 'lemonchiffon'
    @render.set_property('foreground', 'black')
  end

  # ========================================================================= #
  # === create_combo_box
  # ========================================================================= #
  def create_combo_box
    @combo_box = gtk_combo_box
    @combo_box.active = 0
    @combo_box.on_changed {
      # puts 'The combobox was changed'
      @path = @combo_box.active_iter[0] unless @combo_box.active_iter.nil?
      load_file_path unless @is_loading
    }
  end

  # ========================================================================= #
  # === create_scrolled_window
  # ========================================================================= #
  def create_scrolled_window
    @scrolled_window = gtk_scrolled_window(@tree_view) { :automatic }
  end

  # ========================================================================= #
  # === escape
  # ========================================================================= # 
  def escape(v)
    GLib.locale_to_utf8(v)
  end

  # ========================================================================= #
  # === load_selected
  #
  # Lets find out what we selected. via the below method 
  # get_selected
  # ========================================================================= #
  def load_selected
    unless @tree_view.selection.selected.nil?
      @path += get_selected
      load_file_path
    end
  end

  # ========================================================================= #
  # === get_selected
  # ========================================================================= #
  def get_selected
    return @tree_view.selection.selected[0] unless @tree_view.selection.selected.nil?
  end

  # ========================================================================= #
  # === load_file_path
  #
  # Sets @is_loading to true, clears combo box and tree view, then 
  # adds information to itself.
  # ========================================================================= #
  def load_file_path
    @is_loading = true # is loading is now true until we are finished here
    @combo_box.model.clear # clear all combo boxes
    @tree_view.model.clear # clear all tree view models

    case @path
    when '/' 
      @combo_box.prepend_text('/') # prependa / for root tag 
    else
      here = @path.split( /\// ) 
      _ = ''.dup
      here.each {|v|
        _ << v+'/'
        @combo_box.prepend_text(_)
      }
    end
    # select first, always
    @combo_box.active = 0    
    array_files = []
    path = GLib.locale_from_utf8(@path)
    begin
      dir = Dir.new(path)
    rescue Exception => error
      e error
    end    
    # sort the dirs in natural order
    entries = dir.entries
    if entries
      entries.sort_by {|entry|
        entry.downcase
      }.each { |v|
        # sym = FileTest.symlink?( path + "/" + v) ? '@' : ''
        sym = ''.dup
        next if v == '.' || v == '..' # skip . and ..
        _ = path + '/' + v
        if File.directory?(_)
          @tree_view.model.set_value( @tree_view.model.append,
            0, escape(sym + v + '/')) 
        else
          array_files << sym + v
        end
      }
      dir.close
      array_files.each { |new_value| 
        @tree_view.model.set_value(
          @tree_view.model.append, 0, escape(new_value)
        )
      }
      @is_loading = false
    end
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    10
  end

  # ========================================================================= #
  # === favicon?
  # ========================================================================= #
  def favicon?
    '/home/x/DATA/images/HOUSE_ITEMS/WECKER.jpg'
  end

  # ========================================================================= #
  # === Gtk::ViewModule.run
  # ========================================================================= #
  def self.run
    require 'gtk_paradise/run'
    _ = ::Gtk::Tabby::View.new
    r = ::Gtk.run
    r << _
    r.automatic_size
    r.top_left_then_run
  end

end; end; end