#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::UserModule
#
# Note that this class depends on the gem called save_file.
#
# Usage example:
#
#   Gtk::User.new(ARGV)
#
# =========================================================================== #
# Todos:
#
#   - Add support for german at least.
#
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/user/user_module.rb'
# include ::Gtk::UserModule
# =========================================================================== #
module Gtk

module UserModule

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === NAMESPACE
  # ========================================================================= #
  NAMESPACE = inspect

  # ========================================================================= #
  # === FONT_TO_USE
  # ========================================================================= #
  FONT_TO_USE = :hack_20

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = ' User '

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = 620

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 240

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      i           = nil,
      run_already = true
    )
    super(TITLE)
    reset
    set_this_widget(
      create_hbox(
        image_accessories_text_editor,
        gtk_label(TITLE)
      )
    )
    exit_on_destroy_signal
    set_commandline_arguments(i)
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    title_width_height(TITLE, WIDTH, HEIGHT)
    # ======================================================================= #
    # === @use_this_language
    #
    # This variable keeps track of the language in use for this widget.
    # ======================================================================= #
    @use_this_language = :german
    # ======================================================================= #
    # === @save_into_this_yaml_file
    # ======================================================================= #
    @save_into_this_yaml_file = 'user.yml'.dup
    if File.directory?('/Depot/j/') # Prepend this, even though it is a bit hackish.
      @save_into_this_yaml_file.prepend('/Depot/j/')
    end
    set_font(FONT_TO_USE)
    use_gtk_paradise_project_css_file
    infer_the_size_automatically
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    2
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    2
  end
  
  # ========================================================================= #
  # === create_buttons                                          (buttons tag)
  # ========================================================================= #
  def create_buttons
    # ======================================================================= #
    # === @button_save
    # ======================================================================= #
    @button_save = button('_Save the dataset')
    @button_save.clear_background
    @button_save.set_name('button1')
    @button_save.on_hover_lightblue
    @button_save.hint =
      'Click on this button to <b>save the dataset into a local .yml '\
      'file</b> (called <b>'+use_which_yaml_file?+'</b>)'
    @button_save.on_clicked {
      do_save_the_dataset
    }
  end

  # ========================================================================= #
  # === use_which_yaml_file?
  # ========================================================================= #
  def use_which_yaml_file?
    @save_into_this_yaml_file
  end

  # ========================================================================= #
  # === do_save_the_dataset                                        (save tag)
  #
  # This is the method that will save the dataset into a .yml file.
  # ========================================================================= #
  def do_save_the_dataset
    begin
      require 'save_file'
    rescue LoadError; end
    hash = {}
    first_name = @entry_first_name.text.to_s
    e first_name
    family_name = @entry_family_name.text.to_s
    e family_name
    telephone_number = @entry_telephone_number.text.to_s
    e telephone_number
    hash[:first_name]       = first_name
    hash[:family_name]      = family_name
    hash[:telephone_number] = telephone_number
    # ======================================================================= #
    # Determine where to save the .yml file next.
    # ======================================================================= #
    into = use_which_yaml_file?
    e "Saving into `#{into}`."
    SaveFile.write_what_into(YAML.dump(hash), into)
  end

  # ========================================================================= #
  # === create_the_entries                           (entries tag, entry tag)
  # ========================================================================= #
  def create_the_entries
    # ======================================================================= #
    # === @entry_family_name
    # ======================================================================= #
    @entry_family_name = entry
    @entry_family_name.select_all_upon_mouse_button_click_event
    @entry_family_name.bblack1
    @entry_family_name.pad8px
    # ======================================================================= #
    # === @entry_telephone_number
    # ======================================================================= #
    @entry_telephone_number = entry
    @entry_telephone_number.allow_only_numbers_as_input
    @entry_telephone_number.select_all_upon_mouse_button_click_event
    @entry_telephone_number.bblack1
    @entry_telephone_number.pad8px
    # ======================================================================= #
    # === @entry_first_name
    # ======================================================================= #
    @entry_first_name = entry
    @entry_first_name.select_all_upon_mouse_button_click_event
    @entry_first_name.bblack1
    @entry_first_name.pad8px
    [
      @entry_family_name,
      @entry_telephone_number,
      @entry_first_name
    ].each {|this_entry|
      this_entry.set_size_request(180, 40)
    }
  end

  # ========================================================================= #
  # === create_the_grid                                            (grid tag)
  # ========================================================================= #
  def create_the_grid
    # ======================================================================= #
    # === @grid
    # ======================================================================= #
    @grid = default_grid
    @grid.set_column_spacing(8)
    @grid.set_row_spacing(8)
  end

  # ========================================================================= #
  # === create_skeleton                                          (create tag)
  # ========================================================================= #
  def create_skeleton
    create_buttons
    create_the_entries
    create_the_grid
    array_all_labels = []
    case @use_this_language
    # ========================================================================= #
    # === :english
    # ========================================================================= #
    when :english
      @label_first_name = left_aligned_label('First name:')
    # ========================================================================= #
    # === :german
    # ========================================================================= #
    when :german
      @label_first_name = left_aligned_label('Vorname:')
    end

    array_all_labels << @label_first_name
    case @use_this_language
    when :english
      @label_for_the_family_name = left_aligned_label('Family name:')
    when :german
      @label_for_the_family_name = left_aligned_label('Familienname:')
    end
    array_all_labels << @label_for_the_family_name

    @label_telephone_number = left_aligned_label('Telephone number:')

    array_all_labels << @label_telephone_number

    array_all_labels.each {|this_label|
      this_label.set_size_request(290, 40)
    }

    # ======================================================================= #
    # === @main_vbox
    # ======================================================================= #
    @main_vbox = create_vbox
    @alignment = ::Gtk::Alignment.new(0.12, 0.05, 0, 0)
  end

  # ========================================================================= #
  # === populate_the_grid
  # ========================================================================= #
  def populate_the_grid
    @grid.reset
    @grid.left(@label_first_name)
    @grid.right(@entry_first_name)
    @grid.new_row
    @grid.left(@label_for_the_family_name)
    @grid.right(@entry_family_name)
    @grid.new_row
    @grid.left(@label_telephone_number)
    @grid.right(@entry_telephone_number)
    @grid.new_row
    @grid.full_row(@button_save)
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    abort_on_exception
    populate_the_grid
    @alignment.add(@grid)
    add(@alignment)
  end

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    super()
  end

  # ========================================================================= #
  # === Gtk::User[]
  # ========================================================================= #
  def self.[](i = ARGV)
    new(i)
  end

  # ========================================================================= #
  # === Gtk::UserModule.run
  # ========================================================================= #
  def self.run(
      i = ARGV
    )
    require 'gtk_paradise/run'
    _ = ::Gtk::User.new(i)
    r = ::Gtk.run
    r << _
    r.set_size_request(_.width?, _.height?)
    r.infer_the_title
    r.top_left_then_run
  end

end; end