#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::WgetWrapperModule
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/wget_wrapper/wget_wrapper_module.rb'
# include ::Gtk::WgetWrapperModule
# =========================================================================== #
module Gtk

module WgetWrapperModule

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'WgetWrapper'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = 1200

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 340

  # ========================================================================= #
  # === TEXT_TO_USE_FOR_THE_DOWNLOAD_THE_REMOTE_URL_BUTTON
  #
  # Here we also use different unicode symbols for the download text.
  #
  # Useful ones may include: ⭳ ⇩ ↓
  # ========================================================================= #
  TEXT_TO_USE_FOR_THE_DOWNLOAD_THE_REMOTE_URL_BUTTON = 
    '⭳ Download the remote URL ⭳'

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      commandline_arguments = nil,
      run_already           = true
    )
    super(:vertical)
    reset
    set_commandline_arguments(
      commandline_arguments
    )
    run if run_already
  end

  # ========================================================================= #
  # === reset
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, infer_the_namespace]
    title_width_height_font(TITLE, WIDTH, HEIGHT, :dejavu_condensed_22)
    use_gtk_project_css_file
    infer_the_size_automatically
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    2
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    4
  end

  # ========================================================================= #
  # === create_the_main_frame
  # ========================================================================= #
  def create_the_main_frame
    @main_frame = create_frame(@gtk_button_download_the_remote_url)
  end

  # ========================================================================= #
  # === create_entries                                          (entries tag)
  # ========================================================================= #
  def create_entries
    # ======================================================================= #
    # === entry tag
    # ======================================================================= #
    @gtk_entry_holding_the_remote_url = create_entry
    @gtk_entry_holding_the_remote_url.hint =
      "Input the remote URL here. This is the URL that\n<b>wget</b> "\
      "will attempt to download."
    @gtk_entry_holding_the_remote_url.set_size_request(500, 50)
    @gtk_entry_holding_the_remote_url.inner_padding = 15
    @gtk_entry_holding_the_remote_url.pad5px
    @gtk_entry_holding_the_remote_url.mar5px
    @gtk_entry_holding_the_remote_url.ghost_text('Input the remote URL here ...')
  end

  # ========================================================================= #
  # === create_buttons
  # ========================================================================= #
  def create_buttons
    @gtk_button_download_the_remote_url = gtk_bold_button(
      TEXT_TO_USE_FOR_THE_DOWNLOAD_THE_REMOTE_URL_BUTTON
    )
    @gtk_button_download_the_remote_url.fancy_tooltips = 'Click this button '\
      'to download the remote archive.'
    @gtk_button_download_the_remote_url.set_border_width(2)
    @gtk_button_download_the_remote_url.set_name('button1')
    @gtk_button_download_the_remote_url.modify_background(:normal, :royalblue)
    @gtk_button_download_the_remote_url.on_clicked {
      the_download_button_has_been_clicked
    }
    @gtk_button_download_the_remote_url.set_size_request(150, 58)
  end

  # ========================================================================= #
  # === connect_skeleton
  # ========================================================================= #
  def connect_skeleton
    higher_hbox = create_hbox
    higher_hbox.maximal(@gtk_entry_holding_the_remote_url, 2)
    lower_vbox = create_vbox
    lower_vbox.maximal(@main_frame, 2)
    lower_vbox.minimal(@done_message, 0)
    _ = horizontal_pane(higher_hbox, lower_vbox)
    _.set_position(100)
    maximal(_)
  end

  # ========================================================================= #
  # === reset_done_message
  # ========================================================================= #
  def reset_done_message
    @done_message.set_text('')
  end

  # ========================================================================= #
  # === update_the_status_bar_with_this_text
  # ========================================================================= #
  def update_the_status_bar_with_this_text(i)
    @done_message.set_text(i)
  end

  # ========================================================================= #
  # === show_done_message
  # ========================================================================= #
  def show_done_message
    result = File.basename(url?)
    if result.nil? or result.empty?
      # In this case make no change.
    else
      text = 'Downloading of '+result+
             ' has finished into '+
             @base_directory+
             File.basename(result)+
             '.' 
      update_the_status_bar_with_this_text(text)
    end
  end

  # ========================================================================= #
  # === cd_into_a_useful_base_directory
  #
  # This method is to be used to cd to a "safe" directory.
  # ========================================================================= #
  def cd_into_a_useful_base_directory
    if File.directory? '/tmp/'
      cd '/tmp/'
    end
    @base_directory = "#{Dir.pwd}/"
  end

  # ========================================================================= #
  # === url?
  # ========================================================================= #
  def url?
    _ = @gtk_entry_holding_the_remote_url.text
    if _.include? '#'
      _.delete!('#')
    end
    _.strip
  end

  # ========================================================================= #
  # === the_download_button_has_been_clicked
  # ========================================================================= #
  def the_download_button_has_been_clicked(url = url?)
    if url.empty?
      update_the_status_bar_with_this_text('No remote URL was provided.')
    else
      reset_done_message
      esystem "wget #{url}"
      show_done_message
    end
  end

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
    @done_message = create_label
    create_buttons
    create_entries
    create_the_main_frame # Must come after the button was created.
  end

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    cd_into_a_useful_base_directory
    create_skeleton
    connect_skeleton
    show_all
  end

  # ========================================================================= #
  # === Gtk::WgetWrapperModule.run
  # ========================================================================= #
  def self.run
    require 'gtk_paradise/run'
    _ = ::Gtk::WgetWrapper.new
    r = ::Gtk.run
    r << _
    r.add_shortcut(1, 'focus(:left)', :alt)
    r.set_size_request(_.width?, _.height?)
    r.top_left_then_run
  end

  # ========================================================================= #
  # === Gtk::WgetWrapper[]
  # ========================================================================= #
  def self.[](i = ARGV)
    new(i)
  end

end; end