#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# Although this is a class, the functionality must be made available
# on the toplevel module called "Colours" as well. So the class is
# just the wrapper over that toplevel functionality.
# =========================================================================== #
# require 'colours/commandline/commandline.rb'
# Colours::Commandline.new
# =========================================================================== #
require 'colours/base/base.rb'

module Colours

class Commandline < Colours::Base # === Colours::Commandline

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      commandline_arguments = ARGV,
      run_already          = true
    )
    set_commandline_arguments(
      commandline_arguments
    )
    # ======================================================================= #
    # Also store it on the toplevel since as of November 2023.
    # ======================================================================= #
    ::Colours.set_commandline_arguments(
      commandline_arguments
    )
    run if run_already
  end

  # ========================================================================= #
  # === menu                                                       (menu tag)
  # ========================================================================= #
  def menu(
      i = ARGV
    )
    if i.is_a? Array
      i.each {|entry| menu(entry) }
    else
      case i # case tag
      # ===================================================================== #
      # === colours --autogenerate_html_colours_methods
      # ===================================================================== #
      when /^-?-?autogenerate(-|_)?html(-|_)?colours(-|_)?methods$/i,
           /^-?-?autogenerate$/i
        ::Colours.autogenerate_html_colours_methods
      # ===================================================================== #
      # === colours --autogenerate_256_colours_methods
      # ===================================================================== #
      when /^-?-?autogenerate(-|_)?256(-|_)?colours(-|_)?methods$/i,
           /^-?-?256$/
        require 'colours/module_256_colours/module_256_colours.rb'
        ::Colours.autogenerate_256_colours_methods
      # ===================================================================== #
      # === colours --generate_shell_file_containing_the_html_colours
      #
      # This entry point will autogenerate the shell-file that contains
      # the HTML colours (RGB colours).
      # ===================================================================== #
      when /^-?-?generate(-|_)?shell(-|_)?file(-|_)?containing(-|_)?the(-|_)?html(-|_)?colours$/i,
           /^-?-?create(-|_)?shell(-|_)?file$/i # === colours --create-shell-file
        ::Colours.generate_shell_file_containing_the_html_colours
      # ===================================================================== #
      # === colours --show-html_colours
      # ===================================================================== #
      when /^-?-?show(-|_)?html(-|_)?colours$/i,
           /^-?-?show(-|_)?all(-|_)?html(-|_)?colours$/i,
           /^-?-?show(-|_)?all(-|_)?colours$/i,
           /^-?-?html(-|_)?colours$/i,
           'hcolours',
           'scolours',
           'all',
           'ALL',
           '1',
           'ALL_COLOURS',
           'colours?',
           'all_colours',
           'COLOURS',
           'html',
           '' # Show all colours here
        ::Colours.show_html_colours
        exit
      # ===================================================================== #
      # === colours --basics
      # ===================================================================== #
      when /^-?-?basics?$/i
        ::Colours.show_basic_colour_palette
        exit
      # ===================================================================== #
      # === colours --open
      # ===================================================================== #
      when /^-?-?open$/i
        open_this_file :then_exit
      # ===================================================================== #
      # === colours --file?
      # ===================================================================== #
      when /^-?-?file\??$/i
        cliner {
          e 'We will next read in from the default file, from:'
          e
          e "  #{sfile(TEST_FILE)}"
          e
          read_and_display_this_file
        }
      # ===================================================================== #
      # === colours --show-colour-palette
      # === colours --palette
      #
      # This entry point will show the basic colour palette.
      # ===================================================================== #
      when /^-?-?show(-|_)?colou?r(-|_)?palette$/i,
           /^-?-?show(-|_)?palette$/i,
           /^-?-?palette$/i
        show_colour_palette
      # ===================================================================== #
      # === colours --help
      # ===================================================================== #
      when /^-?-?help$/i
        show_help
      # ===================================================================== #
      # === colours --project_base_directory?
      # ===================================================================== #
      when /^-?-?project(_|-)?base(_|-)?dir\??$/i,
           /^-?-?project(_|-)?base(_|-)?directory\??$/i,
           /^-?-?base(_|-)?dir\??$/i
        e Colours.project_base_directory?
      # ===================================================================== #
      # === ruby commandline.rb --version
      # ===================================================================== #
      when /-?-?version/i
        Colours.show_the_version
      # ===================================================================== #
      # === colours --version
      #
      # Show the current version of the colours gem.
      # ===================================================================== #
      when /^-?-?versions?\??$/i
        show_the_version
      end
    end
  end; alias check_against_menu menu # === check_against_menu

  # ========================================================================= #
  # === show_help
  # ========================================================================= #
  def show_help
    ::Colours.show_help
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    menu
  end

end

# =========================================================================== #
# === Colours.set_commandline_arguments
# =========================================================================== #
def self.set_commandline_arguments(i = ARGV)
  @commandline_arguments = [i].flatten.compact
end

# =========================================================================== #
# === Colours.commandline_arguments?
# =========================================================================== #
def self.commandline_arguments?
  @commandline_arguments
end; self.instance_eval { alias commandline? commandline_arguments? } # === Colours.commandline?

# =========================================================================== #
# === Colours.menu                                                 (menu tag)
#
#
# To invoke this method, try:
#
#   colours --show_html_colours
#   colours file?
#
# =========================================================================== #
def self.menu(
    i = ARGV
  )
  _ = ::Colours::Commandline.new(i, :do_not_run_yet)
  _.menu(i) # Directly pass it into the menu-method here.
end; self.instance_eval { alias check_against_menu menu } # === Colours.check_against_menu

end

if __FILE__ == $PROGRAM_NAME
  alias e puts
  Colours::Commandline.new(ARGV)
end