#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::Box
#
# Note that this is for Gtk::Box, which is new since as of ruby-gtk3.
# =========================================================================== #
# require 'gtk_paradise/core_classes/box.rb'
# =========================================================================== #
require 'gtk_paradise/toplevel_methods/determine_which_gtk_version_is_in_use.rb'

# =========================================================================== #
# We assume only ruby-gtk3 and ruby-gtk4 to be here past this point.
# =========================================================================== #
module Gtk

class Box # This is specifically for gtk3 and gtk4.

  if respond_to? :set_spacing
    alias set_padding set_spacing # === set_padding
  end

  # ========================================================================= #
  # === text
  # ========================================================================= #
  def text(
      i, padding_to_use = 2
    )
    minimal(::Gtk::Label.new(i), padding_to_use)
  end

  # ========================================================================= #
  # === left_aligned_text
  # ========================================================================= #
  def left_aligned_text(
      i, padding_to_use = 2
    )
    text = ::Gtk::Label.new(i)
    text.left_align
    minimal(text, padding_to_use)
  end

  # ========================================================================= #
  # === add_space
  #
  # This is, ultimately for now, just a vertical "spacer". At a later
  # time we may have to query the orientation of the box, but for
  # now it just sufficed for my use case to define it as vertical.
  # ========================================================================= #
  def add_space
    ::Gtk.vertical_spacer
  end

end

# =========================================================================== #
# === Gtk.create_vbox
#
# This method can be used to create a vbox, a "vertical box".
#
# In the past, up until October 2022, the default parameters for this
# method were "i = ARGV", but this can create issues when the user
# passes a String, since this method will then erroneously assume
# that these Strings constitute widgets to be added - which is not
# correct. Thus, the parameters were changed towards "*i". 
# =========================================================================== #
def self.create_vbox(*i)
  box = ::Gtk::Box.new(:vertical)
  [i].flatten.compact.each {|widget_to_be_added|
    box.add_or_append(widget_to_be_added)
  }
  return box
end; self.instance_eval { alias vbox        create_vbox } # === Gtk.vbox
     self.instance_eval { alias return_vbox create_vbox } # === Gtk.return_vbox

# =========================================================================== #
# === Gtk.hbox
#
# This method accepts a variable amount of input, which will be
# the widgets that we will put into this widget.
#
# Keep in mind that input to this method may be like this:
#
#   hbox = gtk_hbox(
#      bold_label: 'Merge these images together:',
#      entry: @gtk_entry_for_merging_images
#    )
#
# Note that since as of September 2022 this is permissive, even for
# ruby-gtk3, as ruby-gtk2 will specifically only pull in vbox.rb
# and hbox.rb.
# =========================================================================== #
def self.hbox(
    *boxes, &block
  )
  append_mode = :minimal
  _ = ::Gtk::Box.new(:horizontal)
  # ========================================================================= #
  # Handle blocks next
  # ========================================================================= #
  if block_given?
    yielded = yield
    case yielded
    when :maximal
      append_mode = :maximal
    end
    if boxes.empty?
      boxes = [yielded]
    end
  end
  # ========================================================================= #
  # Handle the "boxes" argument next.
  # ========================================================================= #
  boxes.flatten.each {|box|
    if box.is_a? String
      box = ::Gtk.text(box)
    end
    # ======================================================================= #
    # === Handle Hashes next
    # ======================================================================= #
    if box.is_a? Hash
      # ===================================================================== #
      # We must iterate through this Hash now, via .each_pair, to retain
      # the original structure. Otherwise the gtk-entry may appear on
      # the left side, whereas it was specified last in that Hash.
      # ===================================================================== #
      box.each_pair {|key, value|
        case key
        # =================================================================== #
        # === hbox = gtk_hbox {{ spacing: 10 }}
        # =================================================================== #
        when :spacing
          _.set_spacing(value)
        # =================================================================== #
        # === entry
        #
        # This entry point assumes that the user provided a Gtk::Entry
        # to this method.
        # =================================================================== #
        when :entry
          case append_mode
          when :maximal
            _.maximal(value) # Add the widget here at once.
          else
            _.minimal(value) # Add the widget here at once.
          end
        # =================================================================== #
        # === :bold_label
        # =================================================================== #
        when :bold_label
          case append_mode
          when :maximal
            _.maximal(gtk_label(value).make_bold)
          else
            _.minimal(gtk_label(value).make_bold)
          end
        else
          # ================================================================= #
          # At this point we should check whether the key is unhandled.
          # ================================================================= #
          puts 'Unhandled keys in the Hash: '+key.to_s
        end
      }
    else
      if append_mode == :maximal
        _.maximal(box)
      else
        _.minimal(box)
      end
    end
  }
  _ # And return our widget here.
end

end