#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::Entry
#
# The Gtk::Entry holds an input-widget, into which the user can input
# some text. A few aliases exist to this, such as Gtk.input_field().
#
# Notable methods added to this file here include:
#
#   .append_text()
#   .on_click_event()
#
# Remote documentation for Gtk::Entry can be found here:
#
#   https://ruby-gnome2.osdn.jp/hiki.cgi?Gtk%3A%3AEntry
#
# Note that the code in this file works for both ruby-gtk2 and ruby-gtk3.
#
# However had, it has to be remembered that this file will re-define
# Gtk::Entry, also known as duck-patching, which may cause issues.
# This is why we also disable the warnings briefly - see for the
# code doing so below this comment.  
# =========================================================================== #
# require 'gtk_paradise/core_classes/entry.rb'
# input_field = ::Gtk.input_field
# =========================================================================== #
module Gtk

class Entry # === Gtk::Entry

  require 'gtk_paradise/toplevel_methods/determine_which_gtk_version_is_in_use.rb'
  # require 'gtk_paradise/toplevel_methods/toplevel_methods.rb' # Makes no sense to pull in all toplevel methods here.

  # Gtk.disable_warnings # ← Explained on top of this file.

  # ========================================================================= #
  # === initialize
  #
  # Extend Gtk::Entry to accept default arguments.
  #
  # The first argument is any default entry to be used. The default
  # entry will be an empty '' String.
  # ========================================================================= #
  #def initialize(
  #    default_argument = :default
  #  )
  #  super()
  #  case default_argument
  #  # ======================================================================= #
  #  # === :default
  #  # ======================================================================= #
  #  when :default
  #    default_argument = ''
  #  end
  #  set_text(default_argument)
  #end
  # Gtk.enable_warnings # And re-enable the warnings here.

  # ========================================================================= #
  # === text?
  # ========================================================================= #
  def text?
    text
  end

  if ::Gtk.use_gtk4?
    def set_focus(i = false)
      # ===================================================================== #
      # Currently does nothing. set_focus() is possible in ruby-gtk3
      # but was disabled in ruby-gtk4.
      # ===================================================================== #
    end
  end

  # ========================================================================= #
  # === <<
  #
  # This method name is a slight misnomer, but I want to be able to use
  # << onto a gtk-entry as well.
  # ========================================================================= #
  def <<(i)
    set_text(i.to_s)
  end

  # ========================================================================= #
  # === strip!
  # ========================================================================= #
  def strip!
    set_text(text?.strip)
  end

  # ========================================================================= #
  # === set_maxsize
  # ========================================================================= #
  def set_maxsize(i)
    set_max_length(i)
  end

  # ========================================================================= #
  # === make_scrollable
  #
  # This method allows us to tell a gtk-entry widget to respond to
  # mouse-scroll events.
  # ========================================================================= #
  def make_scrollable
    add_events(:scroll_mask)
  end; alias respond_to_scroll_events make_scrollable # === respond_to_scroll_events

  # ========================================================================= #
  # === increment_by
  # ========================================================================= #
  def increment_by(i = 1)
    old_value = text?.to_i
    new_value = old_value + i
    set_text(new_value.to_s)
  end

  # ========================================================================= #
  # === ghost_text
  #
  # This is essentially a placeholder text, as a hint, which was added
  # in gtk3.
  # ========================================================================= #
  if ::Gtk.use_gtk3?
    alias ghost_text   set_placeholder_text # === ghost_text
    alias ghost_text=  set_placeholder_text # === ghost_text=
    alias shadow_hint= set_placeholder_text # === shadow_hint=
    alias shadow_text= set_placeholder_text # === shadow_text=
    alias ghost=       set_placeholder_text # === shadow_text=
  else
    # ======================================================================= #
    # === ghost_text
    # ======================================================================= #
    def ghost_text(i)
      set_text(i)
    end
  end

  # ========================================================================= #
  # === move_cursor_to_the_most_right_position
  # ========================================================================= #
  def move_cursor_to_the_most_right_position
    self.position = -1
  end; alias cursor_is_to_the_right move_cursor_to_the_most_right_position # === cursor_is_to_the_right
       alias cursor_is_right        move_cursor_to_the_most_right_position # === cursor_is_right
       alias cursor_is_on_the_end   move_cursor_to_the_most_right_position # === cursor_is_on_the_end

  # ========================================================================= #
  # === align_to_the_right
  # ========================================================================= #
  def align_to_the_right
    self.xalign = 1
  end

  # ========================================================================= #
  # === default_styling
  #
  # Simply enable the default-entry CSS styling.
  # ========================================================================= #
  def default_styling
    css_class('default_entry')
  end; alias default_entry  default_styling # === default_entry
       alias default_values default_styling # === default_values

  # ========================================================================= #
  # === try_to_remove_the_last_character
  #
  # This method will try to remove the last character, if there is
  # any text. Minimum text will be ''.
  # ========================================================================= #
  def try_to_remove_the_last_character(
      _ = text?
    )
    if _.size > 0
      set_text(_[0 .. -2].to_s)
    end
  end

  # ========================================================================= #
  # === enable_this_action_on_enter_key
  #
  # This method only works for actions associated with the entry-widget.
  # ========================================================================= #
  def enable_this_action_on_enter_key(this_method)
    on_key_press_event { |widget, event|
      send(this_method)
    }
  end; alias enable_action_on_enter_key enable_this_action_on_enter_key # === enable_action_on_enter_key

  # ========================================================================= #
  # === treat_the_last_comment_as_synonymous_for_on_enter_event
  #
  # Pass a block to this method.
  #
  # Usage example:
  #
  #   @entry = entry
  #   @entry.treat_the_last_comment_as_synonymous_for_on_enter_event {
  #     run_this_method(@entry)
  #   }
  #
  # ========================================================================= #
  def treat_the_last_comment_as_synonymous_for_on_enter_event(&block)
    on_changed {|widget, event| # event is probably nil.
      new_text = widget.text?
      if new_text and (new_text[-1, 1] == '#')
        if block_given?
          yield
        end
        # =================================================================== #
        # Next we must remove the last character:
        # =================================================================== #
        widget.set_text(new_text[0 .. -2])
        true
      else
        false
      end
    }
  end

  # ========================================================================= #
  # === on_enter_event_select_this
  # ========================================================================= #
  def on_enter_event_select_this
    on_button_press_event { |widget, event|
      set_focus(true)
      select_region(0, -1)
    }
  end

  # ========================================================================= #
  # === on_click_colour_change_to_this_colour
  #
  # The first argument is the colour to which you want to change the entry
  # to.
  #
  # The second argument (a Hash) may contain the key :duration, which is
  # used to denote when to revert to the old default.
  #
  # Usage example:
  #
  #   on_click_colour_change_to_this_colour(:steelblue, { duration: 3 })
  #
  # ========================================================================= #
  def on_click_colour_change_to_this_colour(
      use_this_colour = :mintcream,
      hash            = {}
    )
    signal_connect(:event) {|widget, event|
      if ::Gtk.event_left_mouse_button_click?(event)
        old_name = name # Keep track of the original colour.
        set_name(use_this_colour.to_s)
        Thread.new {
          duration = hash[:duration] # Assume n seconds here.
          sleep duration
          set_name(old_name)
        }.join
      end
    }
  end

  # ========================================================================= #
  # === deselect_everything
  # ========================================================================= #
  def deselect_everything
    set_focus(false)
    select_region(0, 0)
  end

  # ========================================================================= #
  # === allow_only_numbers
  #
  # This method accepts one main argument, which should be your Gtk widget.
  # Usually this will be a Gtk::Entry by default.
  #
  # This widget will then only react to NUMBERS input given, such as 1, 3
  # or 8. Other input will be rejected.
  #
  # That way you can make "input-fields" that will only have numbers, and
  # no other characters allowed. The user then is assisted to provide
  # only# valid input, and avoid faulty, unwanted input.
  #
  # Do note that a specialized Widget also exists, called
  # NumbersOnly (in the file aptly named numbers_only.rb).
  # ========================================================================= #
  def allow_only_numbers(
      be_not_too_strict = true
    )
    case be_not_too_strict
    # ========================================================================= #
    # === :default
    # ========================================================================= #
    when :default
      be_not_too_strict = true
    end
    signal_connect(:key_press_event) { |widget, event|
      keyval = Gdk::Keyval.to_name(event.keyval)
      case keyval
      when /\d/
        false
      else
        if be_not_too_strict
          # ================================================================= #
          # === Only escape BackSpace in this case
          # ================================================================= #
          case keyval
          when 'BackSpace'
            false
          else
            true
          end
        else
          true
        end
      end 
    }
  end; alias allow_only_numbers_as_input  allow_only_numbers # === allow_only_numbers_as_input
       alias accept_only_numbers_as_input allow_only_numbers # === accept_only_numbers_as_input

  # ========================================================================= #
  # === simple_completion
  #
  # To make use of this method do:
  #
  #   entry = gtk_entry
  #   entry.simple_completion = %w( abc def ghi )
  #
  # ========================================================================= #
  def simple_completion(array)
    entry_completion = ::Gtk::EntryCompletion.new
    entry_completion.populate_with_this_array(array)
    self.completion = entry_completion # Attach the Gtk::EntryCompletion here.
  end

  # ========================================================================= #
  # === set_read_only
  #
  # This method will disallow changes to the Gtk::Entry instance at hand.
  # ========================================================================= #
  def set_read_only
    self.editable = false
  end; alias may_not_be_modified                set_read_only # === may_not_be_modified
       alias can_not_be_modified                set_read_only # === can_not_be_modified
       alias disable_user_input                 set_read_only # === disable_user_input
       alias disallow_changes                   set_read_only # === disallow_changes
       alias the_user_may_not_modify_this_entry set_read_only # === the_user_may_not_modify_this_entry
       alias is_read_only                       set_read_only # === is_read_only
       alias read_only                          set_read_only # === read_only
       alias readonly                           set_read_only # === readonly

  # ========================================================================= #
  # === append_text
  #
  # This "alias" can be used to append text onto the gtk-entry at hand.
  #
  # Usage example:
  #
  #   append_text('Hello world!')
  #
  # ========================================================================= #
  def append_text(this_text)
    set_text("#{self.text}#{this_text}")
  end

  # ========================================================================= #
  # === do_upcase_the_first_character
  # ========================================================================= #
  def do_upcase_the_first_character
    _ = text?.to_s
    _[0,1] = _[0,1].upcase
    set_text(_)
  end

  # ========================================================================= #
  # === do_upcase
  #
  # This method will upcase the text of the entry at hand.
  # ========================================================================= #
  def do_upcase
    set_text(text?.to_s.upcase)
  end

  # ========================================================================= #
  # === on_click_highlight_all
  #
  # If we hit the Gtk::Entry with a click event from our mouse, then we 
  # will highlight the whole line through this method.
  #
  # This method simplifies this for us. Also note the aliases to
  # this method.
  # ========================================================================= #
  def on_click_highlight_all
    on_click_event { :highlight_text }
  end; alias on_select_highlight_this                 on_click_highlight_all # === on_select_highlight_this
       alias on_click_select_all                      on_click_highlight_all # === on_click_select_all
       alias on_click_select_everything               on_click_highlight_all # === on_click_select_everything
       alias select_everything_on_click               on_click_highlight_all # === select_everything_on_click
       alias on_mouse_click_select_everything         on_click_highlight_all # === on_mouse_click_select_everything
       alias select_all_upon_mouse_button_click_event on_click_highlight_all # === select_all_upon_mouse_button_click_event
       alias select_all_on_click_event                on_click_highlight_all # === select_all_on_click_event
       alias highlight_on_click_event                 on_click_highlight_all # === highlight_on_click_event
       alias focus_on_click_event                     on_click_highlight_all # === focus_on_click_event
       alias highlight_on_focus                       on_click_highlight_all # === focus_on_click_event

  # ========================================================================= #
  # === enable_search_icon
  # ========================================================================= #
  def enable_search_icon(
      icon_name = 'system-search-symbolic'
    )
    set_icon_from_icon_name(
      ::Gtk::EntryIconPosition::PRIMARY,
      icon_name
    )
  end

  # ========================================================================= #
  # === enable_search_icon_via_image
  # ========================================================================= #
  def enable_search_icon_via_image(i)
    set_icon_from_pixbuf(i)
  end

  # ========================================================================= #
  # === on_clicked
  # ========================================================================= #
  def on_clicked(&block)
    on_click_event(&block)
  end

  # ========================================================================= #
  # === do_disappear
  # ========================================================================= #
  def do_disappear
    set_visibility(false)
  end; alias dissappear do_disappear # === dissappear

  # ========================================================================= #
  # === deselect_all_on_click_event
  # ========================================================================= #
  def deselect_all_on_click_event
    on_click_event { :deselect_text }
  end

  # ========================================================================= #
  # === border
  #
  # This is a simplified wrapper over .inner_border=
  #
  # Upstream documentation for .inner_border= can be read here:
  #
  #   https://developer.gnome.org/gtk3/stable/GtkEntry.html#gtk-entry-get-inner-border
  #
  # Note that gtk_entry_get_inner_border() has been deprecated since as 
  # of GTK version 3.4.
  #
  # As an alternative, the standard border and padding CSS properties
  # are recommended to be used.
  #
  # Because we use the wrapper method border(), we can continue to use
  # it though; once we have a work-around for .inner_border= not working
  # on more recent gtk-versions, we can adjust the code here.
  # ========================================================================= #
  def border(*i)
    if i.is_a?(Array) and (i.size < 4)
      # ===================================================================== #
      # In this case we have too few entries. Thus, we fill the missing
      # ones up.
      # ===================================================================== #
      first_element = i.first
      old_array = i.dup
      (4 - i.size).times {
        old_array << first_element
      }
      i = old_array
    end
    self.inner_border = ::Gtk::Border.new(i[0], i[1], i[2], i[3])
  end; alias inner_padding= border # === inner_padding=

  # ========================================================================= #
  # === return_input
  # ========================================================================= #
  def return_input
    self.text
  end; alias content return_input # === content

  # ========================================================================= #
  # === empty?
  # ========================================================================= #
  def empty?
    return_input.empty?
  end

  # ========================================================================= #
  # === default=
  #
  # This wrapper-method is used primarily so that we can convert the
  # given input into a String object.
  # ========================================================================= #
  def default=(i)
    set_text(i.to_s)
  end; alias set_content default= # === set_content

  # ========================================================================= #
  # === make_bold
  #
  # This will make the entry appear "bold", aka with bold text.
  # ========================================================================= #
  def make_bold
    font = Pango::FontDescription.new('bold')
    if ::Gtk.use_gtk2?
      modify_font(font)
    else
      override_font(font)
    end
  end

  # ========================================================================= #
  # === clear
  #
  # This method will "empty" the content, aka set the text to ''.
  # ========================================================================= #
  def clear
    set_text('')
  end

  # ========================================================================= #
  # === on_key_release
  # ========================================================================= #
  def on_key_release(&block)
    signal_connect(:key_press_event, &block)
  end

  # ========================================================================= #
  # === do_select_everything
  #
  # This method can be used to simply select everything - the whole content
  # of a gtk-entry, as if the user did this via the mouse.
  # ========================================================================= #
  def do_select_everything
    select_region(0, -1)
  end; alias select_everything do_select_everything # === select_everything

  # ========================================================================= #
  # === allow_enter
  # ========================================================================= #
  def allow_enter
    set_activates_default(true)
  end; alias enable_enter_event allow_enter # === enable_enter_event

end

# =========================================================================== #
# === Gtk.input_field
#
# Convenience method to instantiate a new Gtk::Entry object.
#
# Some aliases exist, such as Gtk.entry() and similar.
#
# Usage example:
#
#   input_field = ::Gtk.input_field
#
# The method must be able to respond to input like this one here:
#
#   input_field = gtk_input_field {{ max_length: 50 }}
#
# =========================================================================== #
def self.input_field(
    i = :default, &block
  )
  entry = Gtk::Entry.new
  if i
    case i
    # ======================================================================= #
    # === :default
    # ======================================================================= #
    when :default
      i = '' 
    end
    entry.set_text(i)
  end
  if block_given?
    yielded = yield
    if yielded.is_a? Hash
      # ===================================================================== #
      # === :max_length
      # ===================================================================== #
      if yielded.has_key? :max_length
        entry.set_max_length(yielded.delete(:max_length))
      end
    end
  end
  return entry
end; self.instance_eval { alias entry        input_field } # === Gtk.entry
     self.instance_eval { alias input        input_field } # === Gtk.input
     self.instance_eval { alias return_entry input_field } # === Gtk.return_entry
     self.instance_eval { alias gtk_input    input_field } # === Gtk.gtk_input

end