#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::Label
#
# All gtk-label related code is stored here in this .rb file.
# =========================================================================== #
# require 'gtk_paradise/core_classes/label.rb'
# =========================================================================== #
module Gtk

class Label # === Gtk::Label

  require 'gtk_paradise/toplevel_methods/determine_which_gtk_version_is_in_use.rb'

  # ========================================================================= #
  # === make_selectable
  #
  # This will make the given Gtk::Label selectable.
  # ========================================================================= #
  def make_selectable
    self.selectable = true
  end; alias enable_on_click_actions make_selectable # === enable_on_click_actions
       alias is_selectable           make_selectable # === is_selectable

  # ========================================================================= #
  # === reposition_the_cursor_to_the_right
  # ========================================================================= #
  def reposition_the_cursor_to_the_right
    #_ = text?
    # set_cursor_position(_.size - 1)
    # do_move_cursor(step, count, extend_selection)
  end

  # ========================================================================= #
  # === cursor_is_on_the_right
  #
  # This does not work. For some selectable labels we should be able to
  # position the cursor programmatically.
  # ========================================================================= #
  # def cursor_is_on_the_right
  #   self.position = -1
  # end

  # ========================================================================= #
  # === set_bold_text
  # ========================================================================= #
  def set_bold_text(i)
    set_text(i.to_s)
    make_bold
  end

  # ========================================================================= #
  # === make_bold
  #
  # This method will make the whole Gtk::Label instance bold.
  #
  # The second argument will determine whether we will use a mnemonic or 
  # whether we will not.
  # ========================================================================= #  
  def make_bold(
      old_text     = text,
      use_mnemonic = :default
    )
    case old_text
    # ======================================================================= #
    # === :default
    #
    # This entry point here will simply use the original text.
    # ======================================================================= #
    when :default,
         :use_the_default_text
      old_text = text
    # ======================================================================= #
    # === :retain_mnemonic
    # ======================================================================= #
    when :retain_mnemonic,
         :use_mnemonic
      old_text = text
      use_mnemonic = true
    end
    case use_mnemonic
    # ======================================================================= #
    # === :default
    # ======================================================================= #
    when :default
      use_mnemonic = false
    end
    old_text = old_text.dup if old_text.frozen?
    if old_text.include? '&'
      # ===================================================================== #
      # Oddly enough, ruby-gtk2 complains about this character.
      # The warning was: "Gtk-WARNING **: Failed to set text from
      # markup due to error parsing markup" [...] Entity did not 
      # end with a semicolon; most likely you used an ampersand 
      # character without intending to start an entity - escape 
      # ampersand as &amp;"
      # So we do this ^^^.
      # ===================================================================== #
      old_text.gsub!(/&/, '&amp;')
    end
    # ======================================================================= #
    # === Handle ruby-gtk2 next
    #
    # Next, we must differ between gtk2 and gtk3.
    # ======================================================================= #
    # if Gtk.are_we_using_gtk2?
    #   set_markup(
    #     %Q(<span weight="bold">#{old_text}</span>),
    #     use_mnemonic
    #   )
    # ======================================================================= #
    # === Check for ruby-gtk4 next:
    #
    # We may have to use a CSS rule here such as "font-weight: 500;".
    # ======================================================================= #
    if Gtk.are_we_using_gtk4?
      #e 'Not yet implemented .make_bold() for ruby-gtk4.'
      css_class('bold_font')
    else # else we use gtk3.
      # ===================================================================== #
      # === Handle ruby-gtk3 next
      #
      # set_markup(
      #   %Q(<span weight="bold">#{old_text}</span>),
      #   use_underline: use_mnemonic
      # )
      # ===================================================================== #
      # An alternative may be to do this here:
      #
      #   override_font(Pango::FontDescription.new('bold'))
      #
      # In fact, we are trying this as of December 2020 for ruby-gtk3.
      # ===================================================================== #
      override_font(
        Pango::FontDescription.new('bold')
      )
    end
  end; alias bold      make_bold # === bold
       alias text_bold make_bold # === text_bold

  # ========================================================================= #
  # === align_left
  #
  # Note that this could also be .set_alignment(0,0). For whatever the
  # reason, I picked (0, 1) as the default here, even before November
  # 2020. So we'll stick to that for the time being. 
  # ========================================================================= #
  def align_left
    set_alignment(0, 1) if respond_to?(:set_alignment) # ruby-gtk4 does not have this.
  end; alias left_align                        align_left # === left_align
       alias left                              align_left # === left
       alias align_to_the_left                 align_left # === align_to_the_left
       alias to_the_left                       align_left # === to_the_left
       alias the_text_appears_on_the_left_side align_left # === the_text_appears_on_the_left_side
       alias align_towards_the_left            align_left # === align_towards_the_left

  # ========================================================================= #
  # === left_rotate
  #
  # The next alias is to allow "tilted", aka slanted, text. It currently
  # only works for ruby-gtk3.
  # ========================================================================= #
  if Gtk.do_we_use_gtk3?
    alias left_rotate angle= # === left_rotate
  end

  # ========================================================================= #
  # === parse_hyperlinks
  #
  # This method can be used to "parse" hyperlinks - simply enabling
  # markup on the given label.
  #
  # The idea of this method was initially to custom-parse the answers
  # in the studium-gem, but this has not yet been implemented here.
  # ========================================================================= #
  def parse_hyperlinks
    enable_markup
  end; alias enable_hyperlinks parse_hyperlinks # === enable_hyperlinks
       alias fancy_parse       parse_hyperlinks # === fancy_parse

  # ========================================================================= #
  # === set_markup_text
  #
  # This is typically used for Gtk::Label, hence the internal check.
  # ========================================================================= #
  def set_markup_text(i)
    set_text(i)
    do_markify
  end

  # ========================================================================= #
  # === text?
  # ========================================================================= #
  def text?
    text.to_s
  end

  # ========================================================================= #
  # === align_left_and_center_in_the_middle
  # ========================================================================= #
  def align_left_and_center_in_the_middle
    set_alignment(1, 0.5)
  end

  # ========================================================================= #
  # === align_right
  #
  # The two values to the alignment are xalign and yalign.
  #
  # The values for xalign are:
  #
  #   0   is left
  #   0.5 is centered
  #   1   is right
  #
  # ========================================================================= #
  def align_right
    set_alignment(1, 0.5)
  end; alias right              align_right # === right
       alias right_align        align_right # === right_align
       alias align_to_the_right align_right # === align_to_the_right

  require 'gtk_paradise/emojis/emojis.rb'
  # ========================================================================= #
  # === set_emoji
  # ========================================================================= #
  def set_emoji(i = :happy_smiley)
    set_text(::Gtk::Emojis.map_emoji(i))
  end

  # ========================================================================= #
  # === center_and_make_bold
  # ========================================================================= #
  def center_and_make_bold
    center
    make_bold
  end

  # ========================================================================= #
  # === colourize
  #
  # The first argument to this method, as a String, should denote which
  # colour we will be using.
  # ========================================================================= #
  def colourize(
      use_this_colour = :slateblue
    )
    use_this_colour = Gdk::Color.parse(use_this_colour.to_s)
    new_value = %Q(<span foreground="#{use_this_colour}">#{self.text}</span>)
    set_markup(new_value)
  end

  # ========================================================================= #
  # === clear
  #
  # This "empties" a gtk-label.
  # ========================================================================= #
  def clear
    set_text('')
  end

  # ========================================================================= #
  # === append
  # ========================================================================= #
  def append(i = '')
    set_text("#{text}#{i}")
  end; alias << append # === <<

  # ========================================================================= #
  # === center
  # ========================================================================= #
  def center(
      i = :center
    )
    if Gtk.are_we_using_gtk2?
      i = ::Gtk::JUSTIFY_CENTER
    end
    set_justify(i)
  end

  # ========================================================================= #
  # === set_font_size
  #
  # This method can be used to specifically set a font-size.
  # ========================================================================= #
  def set_font_size(i = 25)
    i = Pango::FontDescription.new(i.to_s)
    if Gtk.are_we_using_gtk2?
      modify_font(i)
    else
      override_font(i)
    end
  end; alias font_size= set_font_size # === font_size=

end

  # ========================================================================= #
  # === Gtk.modify_label
  #
  # This method can be used for doing doing coloured labels.
  #
  # The second argument is a default value.
  #
  # Usage examples:
  #
  #   modify_label('test1','green')
  #   modify_label('URL: ','darkblue')
  #   modify_label('links ','darkblue', :left)
  #
  # ========================================================================= #
  def self.modify_label(
      name,
      colour  = 'darkblue',
      justify = nil
    )
    _ = Gtk.create_label(name)
    if Gtk.are_we_using_gtk2?
      _.set_markup(
        %Q[<markup><span foreground="#{colour}">#{name} </span></markup>], true
      )
    else # else use ruby-gtk3
      _.set_markup(
        %Q[<markup><span foreground="#{colour}">#{name} </span></markup>], 
        { use_underline: true }
      )
    end
    unless justify.nil?
      left_justify = :left
      if use_gtk2?
        left_justify = ::Gtk::JUSTIFY_LEFT
      end
      case justify
      when :center
        _.set_justify(::Gtk::JUSTIFY_CENTER)
      when :left
        _.set_justify(left_justify)
      when :right
        _.set_justify(::Gtk::JUSTIFY_RIGHT)
      when :fill
        _.set_justify(::Gtk::JUSTIFY_FILL)
      else
        _.set_justify(::Gtk::JUSTIFY_LEFT)
      end
    end
    return _
  end

  # ========================================================================= #
  # === Gtk.modify_bold_label
  #
  # To make use of this code, do:
  #
  #   require 'gtk_paradise'; Gtk.modify_bold_label('Hello world!')
  #
  # Wrapper to easier modify the colours of a label, using bold label.
  # Can be quite convenient.
  #
  # The gist of it is the method set_markup(), as in:
  #
  #   set_markup(%Q[<markup><span weight="bold" foreground="#{colour}">Test</span></markup>], true)
  #
  # Specific usage examples:
  #
  #   Gtk.modify_bold_label('LE: ','red')
  #   Gtk.modify_bold_label('AttackMe: ','slateblue')
  #   Gtk.modify_bold_label('Ifconfig')
  #   Gtk.modify_bold_label('AttackMe: ','slateblue','left')
  #
  # ========================================================================= #
  def self.modify_bold_label(
      name,
      colour    = 'black', # This here is the foreground label.
      justify   = '',
      alignment = ''
    )
    colour = random_colour if colour == 'rand'
    _ = create_label(name) # Instantiate a new Gtk::Label widget here.
    _.set_markup(
      %Q[<markup><span weight='bold' foreground="#{colour}">#{name} </span></markup>],
      { use_underline: true }
    )
    justify_right = :right
    if use_gtk2?
      justify_right = ::Gtk::JUSTIFY_RIGHT
    end
    case justify
    # ======================================================================= #
    # === center
    # ======================================================================= #
    when :center, :c, 'center',''
      if ::Gtk.use_gtk2?
        _.set_justify(::Gtk::JUSTIFY_CENTER)
      else
        _.set_justify(:center)
      end
    # ======================================================================= #
    # === left
    # ======================================================================= #
    when :left,   :l, 'left'
      _.set_justify(::Gtk::JUSTIFY_LEFT)
    # ======================================================================= #
    # === right
    # ======================================================================= #
    when :right,  :r, 'right'
      _.set_justify(justify_right)
    # ======================================================================= #
    # === fill
    # ======================================================================= #
    when :fill,   :f, 'fill'
      _.set_justify(::Gtk::JUSTIFY_FILL)
    else # generic fallback (else tag)
      _.set_justify(::Gtk::JUSTIFY_LEFT) # The default.
    end
    if alignment.is_a? Numeric
      alignment = alignment.to_s
    end
    unless alignment.empty?
      case alignment
      # ===================================================================== #
      # === :center
      # ===================================================================== #
      when :center, :c, 'center'
        _.set_alignment(0.5,0.5)
      # ===================================================================== #
      # === :left
      # ===================================================================== #
      when :left,   :l, 'left', '1'
        _.set_alignment(0,1)
      # ===================================================================== #
      # === :right
      # ===================================================================== #
      when :right,  :r, 'right'
        _.set_alignment(1,0)
      # ===================================================================== #
      # === :fill
      # ===================================================================== #
      when :fill,   :f, 'fill'
        _.set_alignment(0,0)
      else # generic fallback
        _.set_alignment(0,1)
      end
    end
    return _ 
  end

  # ========================================================================= #
  # === Gtk.bold_label
  #
  # This is like Gtk.label() but it makes the font in use bold.
  # ========================================================================= #
  def self.bold_label(
      default_text  = '',
      use_mnemonics = false
    )
    _ = ::Gtk.label(default_text, use_mnemonics)
    _.make_bold
    return _
  end

  # ========================================================================= #
  # === Gtk.markup_label
  #
  # Use this if you want to modify an existing label. It uses an extremely
  # simplified model. Ah never mind ... it currently is TOTALLY broken ;)
  #
  # markup_label(label, blalalala)
  # ========================================================================= #
  def self.markup_label(
      label_to_modify,
      text,
      colour_to_use = 'slateblue'
    )

    super_string = "<markup>
      <span weight='bold' foreground='black'>
    
      #{text}
    </span>"
 
    label_to_modify.set_markup(%Q[#{super_string}
      <span weight='bold' foreground='#{colour_to_use}'>test bold?</span>
      </markup>],
      true
    )
  end

  # ========================================================================= #
  # === Gtk.simple_text
  #
  # This method has been specifically createdf or the gir-bindings for
  # ruby-gtk3.
  # ========================================================================= #
  def self.simple_text(i = '')
    label = Gtk::Label.new
    label.set_text(i) if i and label.respond_to?(:set_text)
    return label
  end

  # ========================================================================= #
  # === Gtk.create_label
  #
  # A simple shortcut method to create a new Gtk label (Gtk::Label).
  #
  # The first argument to this method should be a String, indicating
  # the text that we will use for this particular label.
  # ========================================================================= #
  def self.create_label(
      default_text  = '',
      make_bold     = false,
      use_mnemonics = false
    )
    case use_mnemonics
    # ======================================================================= #
    # === :do_use_mnemonics
    # ======================================================================= #
    when :do_use_mnemonics
      use_mnemonics = true
    end
    # ======================================================================= #
    # The next line is necessary due to deprecations between ruby-gtk2 and
    # ruby-gtk3.
    # ======================================================================= #
    if use_gtk3?
      if use_mnemonics.is_a?(String) or
         use_mnemonics.is_a?(TrueClass) or
         use_mnemonics.is_a?(FalseClass)
        use_mnemonics = { use_underline: use_mnemonics }
      end
    end
    if default_text.is_a? Symbol
      default_text = default_text(
        ::Gtk::Emojis.map_emoji(default_text)
      )
    end
    if ::Gtk.use_gtk2?
      label = ::Gtk::Label.new
      label.set_text(default_text.to_s) if label.respond_to?(:set_text)
      # label = ::Gtk::Label.new(
      #   default_text,
      #   use_mnemonics
      # )
    else # else for gtk3
      label = ::Gtk::Label.new(
        default_text,
        use_mnemonics: use_mnemonics
      )
    end
    # ======================================================================= #
    # === Handle "HTML strings" next
    # ======================================================================= #
    if default_text and
       default_text.respond_to?(:include?) and
      (default_text.include?('<b>') or
       default_text.include?('<i>'))
      label.set_markup(default_text)
    end
    # ======================================================================= #
    # === Handle Hashes next
    # ======================================================================= #
    if use_mnemonics.is_a? Hash
      # ===================================================================== #
      # === :colour
      #
      # Handle Hashes here, such as:
      #
      #   { colour: :white }
      #
      # ===================================================================== #
      if use_mnemonics.has_key? :colour
        label.set_colour(use_mnemonics[:colour])
      end
    end
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      # ===================================================================== #
      # === Handle Hashes next
      # ===================================================================== #
      if yielded.is_a? Hash
        # =================================================================== #
        # === :tooltip
        # =================================================================== #
        if yielded.has_key? :tooltip
          # ================================================================= #
          # In this case the user wants to quickly make use of a tooltip.
          # ================================================================= #
          use_this_text = yielded.delete(:tooltip)
          label.fancy_tooltips = use_this_text
        end
      end
    end
    if make_bold and label.respond_to?(:make_bold)
      label.make_bold
    end
    return label
  end; self.instance_eval { alias text         create_label } # === Gtk.text
       self.instance_eval { alias label        create_label } # === Gtk.label
       self.instance_eval { alias gtk_label    create_label } # === Gtk.gtk_label
       self.instance_eval { alias return_label create_label } # === Gtk.return_label

end