#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::SpinButton
#
# The upstream API for gtk3 can be found here:
#
#   https://lazka.github.io/pgi-docs/Gtk-3.0/classes/SpinButton.html
#   https://docs.gtk.org/gtk3/class.SpinButton.html
#
# =========================================================================== #
# require 'gtk_paradise/core_classes/spin_button.rb'
# =========================================================================== #
module Gtk

class SpinButton # === Gtk::SpinButton

  # ========================================================================= #
  # === set_max_value
  # ========================================================================= #
  def set_max_value(max)
    min = range.first
    set_range(min, max)
  end

  # ========================================================================= #
  # === range_span
  # ========================================================================= #
  def range_span(
      start_value = 1960,
      end_value   = 2030,
      increment   =    1
    )
    set_range(start_value, end_value)
    set_start_point(start_value)
    set_climb_rate(increment)
  end

  # ========================================================================= #
  # === text?
  # ========================================================================= #
  def text?
    text.to_s
  end

  # ========================================================================= #
  # === on_value_changed
  # ========================================================================= #
  def on_value_changed(&block)
    signal_connect(:value_changed, &block)
  end

  # ========================================================================= #
  # === start_value=
  #
  # This method can be used to designate the start value used and 
  # displayed by a spin-button.
  # ========================================================================= #
  def start_value=(i)
    set_value(i.to_i)
  end; alias set_start_point start_value= # === set_start_point
       alias start_position= start_value= # === start_position=

end

# =========================================================================== #
# === Gtk.spin_button
#
# Note that the values that are given to Gtk::SpinButton specify
# min, max and range.
#
# Usage examples:
#
#   gtk_spin_button(0, 300, 1)
#   spinner_button = gtk_spinner_button {{ start_point: 1 }}
#
# =========================================================================== #
def self.spin_button(
    start_position = 0,
    end_position   = 100,
    increment_size = 1,
    &block
  )
  _ = ::Gtk::SpinButton.new(start_position, end_position, increment_size)
  # ========================================================================= #
  # === Handle blocks given to this method next
  # ========================================================================= #
  if block_given?
    yielded = yield
    if yielded.is_a? Hash
      # ===================================================================== #
      # === :start_point
      # ===================================================================== #
      if yielded.has_key? :start_point
        _.set_start_point(yielded[:start_point])
      end
      # ===================================================================== #
      # === :start_value
      # ===================================================================== #
      if yielded.has_key? :start_value
        _.set_value(yielded[:start_value].to_s)
      end
    end
  end
  return _
end; self.instance_eval { alias spinbutton         spin_button } # === Gtk.spinbutton
     self.instance_eval { alias gtk_spinner_button spin_button } # === Gtk.gtk_spinner_button
     self.instance_eval { alias gtk_spinbutton     spin_button } # === Gtk.gtk_spinbutton

end