#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::Widget
# =========================================================================== #
# require 'gtk_paradise/core_classes/widget.rb'
# =========================================================================== #
module Gtk

class Widget # === Gtk::Widget

  require 'gtk_paradise/toplevel_methods/determine_which_gtk_version_is_in_use.rb'
  require 'gtk_paradise/fonts/fonts.rb'

  # ========================================================================= #
  # === add_hsep
  # ========================================================================= #
  def add_hsep
    _ = ::Gtk.horizontal_separator
    add(_)
  end

  # ========================================================================= #
  # Next, call in gdk-specific modifications.
  # ========================================================================= #
  require 'gtk_paradise/gdk/event_crossing.rb'

  if Gtk.do_we_use_gtk4?
    # ======================================================================= #
    # === set_border_width
    # ======================================================================= #
    def set_border_width(i = 5)
      css_class("pad#{i}px")
    end
  end

  # ========================================================================= #
  # === on_hover                                               (on_hover tag)
  #
  # This method can be used to modify the colour of a given widget, on
  # a "on-hover" event.
  #
  # Usage examples:
  #
  #   on_hover(:lightblue)
  #   on_hover(:tomato)
  #
  # ========================================================================= #
  def on_hover(
      this_colour = :lightgreen,
      &block
    )
    # ======================================================================= #
    # Next we have to sanitize the input a little bit:
    # ======================================================================= #
    if this_colour.to_s.include? 'on_hover_'
      this_colour = this_colour.to_s.delete_prefix('on_hover_').to_sym
    end
    modify_background(
      :on_hover,
      this_colour,
      &block
    )
  end; alias on_hover_colour on_hover # === on_hover_colour
       alias hover           on_hover # === hover

  # ========================================================================= #
  # === modify_background
  #
  # This method will set the background colour for a widget in a
  # particular state. All other style values are left untouched.
  # See also Gtk::Widget#modify_style.
  #
  # Since as of November 2023, the gtk2-specific variant has been
  # removed. For reference purposes, the old variant ::Gtk::STATE_NORMAL
  # available in ruby-gtk2 was replaced via :normal in ruby-gtk3.
  # Prelight would be ::GTK::STATE_PRELIGHT in ruby-gtk2, but is
  # now :prelight in ruby-gtk3.
  #
  # For ruby-gtk2 we also would then call modify_bg(state, colour).
  #
  # The arguments to this method are:
  #
  #   state:  the state(GtkStateType) for which to set the 
  #           background color.
  #
  #   colour: the Gdk::Color to assign (does not need to 
  #           be allocated).
  #
  # ========================================================================= #
  def modify_background(
      state,
      colour = nil,
      &block
    )
    # ======================================================================= #
    # === Handle Hashes first, for the variable state
    # ======================================================================= #
    if colour.is_a? Hash
      if colour.has_key? :colour
        colour = colour.delete(:colour)
      end
    end
    case state
    # ======================================================================= #
    # === :prelight
    #
    # This is synonymous with :on_hover.
    # ======================================================================= #
    when :prelight,
         :hover,
         :on_hover
      state = :prelight
    # ======================================================================= #
    # === :normal
    #
    # This entry point is the default entry point.
    # ======================================================================= #
    when :normal,
         :default
      state = :normal
    # ======================================================================= #
    # === :active
    # ======================================================================= #
    when :active
      state = :active
    end
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      if yielded.is_a?(Symbol)
        colour = yielded
      end
    end
    if colour.is_a? Symbol
      # ===================================================================== #
      # Convert it into a Gtk::Colours representation. We could check if it
      # is a valid html-colour, but for the time being we do not do this.
      # ===================================================================== #
      if Gtk.do_we_use_gtk2? # This is retained for now, until we change it to ruby-gtk3.
        colour = ::Gtk.colour_parse(colour)
        # =================================================================== #
        # gtk3 now uses RGBA rather than Gdk.parse().
        # RGBA stands for red, green, blue and alpha.
        # The values go from 0.0 to 1.0. However had, it seems as if
        # ruby-gtk3 also supports HTML colours directly, so no need
        # to convert them.
        # =================================================================== #
      end
    end
    _ = "hover_effect_#{colour}"
    css_class(_)
    # else # else assume ruby-gtk3 here.
    #   override_background_color(state, colour)
    # end
  end

  # ========================================================================= #
  # === yellow_background
  #
  # This method can be used to designate a light yellow background
  # colour tone. It depends on support for CSS being available -
  # thus, ruby-gtk2 does not work with this method.
  # ========================================================================= #
  def yellow_background
    add_css_class('BG_light_yellow')
  end; alias light_yellow_background    yellow_background # === light_yellow_background
       alias light_yellowish_background yellow_background # === light_yellowish_background
       alias yellowish_background       yellow_background # === yellowish_background

  # ========================================================================= #
  # === very_light_yellowish_background
  #
  # This is similar to yellow_background() defined above, but the difference
  # will be that the yellow colour here will be very faint - even fainter
  # than on the method defined above.
  #
  # This depends on CSS, so it will not work on ruby-gtk2.
  # ========================================================================= #
  def very_light_yellowish_background
    add_css_class('BG_very_light_yellowish') 
  end; alias yellowish_background2               very_light_yellowish_background # === yellowish_background2
       alias background_yellow2                  very_light_yellowish_background # === background_yellow2
       alias very_light_yellow_background_colour very_light_yellowish_background # === very_light_yellow_background_colour
       alias very_light_yellow_background        very_light_yellowish_background # === very_light_yellow_background
       alias BG_yellow                           very_light_yellowish_background # === very_light_yellow_background_colour
       alias yello                               very_light_yellowish_background # === yello

  # ========================================================================= #
  # === on_hover_lightblue
  # ========================================================================= #
  def on_hover_lightblue
    on_hover(__method__)
  end

  # ========================================================================= #
  # === on_hover_steelblue
  # ========================================================================= #
  def on_hover_steelblue
    on_hover(__method__)
  end

  # ========================================================================= #
  # === on_hover_tomato
  # ========================================================================= #
  def on_hover_tomato
    on_hover(__method__)
  end

  # ========================================================================= #
  # === on_hover_lightgreen
  # ========================================================================= #
  def on_hover_lightgreen
    on_hover(__method__)
  end

  # ========================================================================= #
  # === on_hover_paleturquoise
  # ========================================================================= #
  def on_hover_paleturquoise
    on_hover(__method__)
  end

  # ========================================================================= #
  # === on_hover_skyblue
  # ========================================================================= #
  def skyblue
    on_hover(__method__)
  end

  # ========================================================================= #
  # === on_hover_lightpink
  # ========================================================================= #
  def lightpink
    on_hover(__method__)
  end

  # ========================================================================= #
  # === on_hover_darkmagenta
  # ========================================================================= #
  def on_hover_darkmagenta
    on_hover(__method__)
  end

  # ========================================================================= #
  # === on_hover_limegreen
  # ========================================================================= #
  def on_hover_limegreen
    on_hover(__method__)
  end; alias mediumaquamarine on_hover_limegreen # === mediumaquamarine
       alias khaki            on_hover_limegreen # === khaki

  # ========================================================================= #
  # === lightskyblue_background
  # ========================================================================= #
  def lightskyblue_background
    add_css_class('BG_lightskyblue')
  end; alias bg_lightblue          lightskyblue_background # === bg_lightblue
       alias lightblue_background  lightskyblue_background # === lightblue_background
       alias light_blue_background lightskyblue_background # === light_blue_background

  # ========================================================================= #
  # === light_pink_background
  # ========================================================================= #
  def light_pink_background
    add_css_class('BG_lightpink')
  end; alias lightpink_background light_pink_background # === lightpink_background

  # ========================================================================= #
  # === modify_to_this_font
  #
  # This will modify the font for that particular widget at hand.
  #
  # Usage example:
  #
  #   modify_this_font(:liberation)
  #
  # ========================================================================= #
  def modify_to_this_font(
      this_font = 'Sans 20'
    )
    this_font = ::Gtk.expand_this_font(this_font) # This is part of gtk_paradise/fonts/fonts.rb.
    _ = Pango::FontDescription.new(this_font)
    # if Gtk.are_we_using_gtk2?
    #  modify_font(_)
    #else
      override_font(_)
    #end
  end; alias modify_this_font modify_to_this_font # === modify_this_font
       alias font_to_use=     modify_to_this_font # === font_to_use=

  # ========================================================================= #
  # === use_these_css_classes
  #
  # This method allows for some convenience in regards to using CSS
  # classes. You can pass several CSS classes into this method.
  #
  # It is important to separate these CSS classes via a
  # ' ' character specifically.
  #
  # Specific usage example:
  #
  #   @frame.use_these_css_classes('no_box_shadow bblack2 BG_black pad0px mar0px')
  #
  # ========================================================================= #
  def use_these_css_classes(i)
    # ======================================================================= #
    # The input has to have a ' ' character, if this method is to work.
    # ======================================================================= #
    if i and i.include?(' ')
      i.split(' ').each {|this_css_class|
        css_class(this_css_class)
      }
    else
      e 'The given input (`'+i.to_s+'`) does not have a " " character.'
    end
  end; alias use_these_CSS_classes use_these_css_classes # === use_these_CSS_classes
       alias css_classes           use_these_css_classes # === css_classes

  # ========================================================================= #
  # === add_to_the_CSS_style_context
  #
  # This method must be guaranteed to work on style_context. Thus, it can
  # only work properly if style_context() actually returns something useful.
  # ========================================================================= #
  def add_to_the_CSS_style_context(i)
    style_context.add_class(i)
  end

  # ========================================================================= #
  # === deselect
  #
  # Note that this method will NOT work on every gtk-widget; the primary
  # use case has been for gtk-entry, but I figured that other gtk-widgets
  # may benefit from this method as well, so it was added onto class
  # Gtk::Widget.
  # ========================================================================= #
  def deselect
    select_region(0, 0)
  end; alias deselect_everything deselect # === deselect_everything
       alias defocus             deselect # === defocus

  # ========================================================================= #
  # === remove_widgets
  #
  # This method will remove all child-widgets from a container.
  # ========================================================================= #
  def remove_widgets(
      i = children
    )
    i.each {|this_child_widget|
      remove(this_child_widget)
    }
  end

  # ========================================================================= #
  # === set_background_colour
  #
  # This method can be used to designate a specific background colour
  # for a Gtk::Button.
  # ========================================================================= #
  def set_background_colour(
      i = :whitesmoke
    )
    # if ::Gtk.use_gtk2?
    #   modify_bg(
    #     ::Gtk::STATE_NORMAL,
    #     Gdk::Color.parse(i.to_s)
    #   )
    # else
    if i.is_a?(String) and i.start_with?('#') # Assume a hex-code in this event.
      unless Object.const_defined?(:Colours)
        begin
          require 'colours'
        rescue LoadError; end
      end
      i = ::Colours.convert_hex_code_to_RGBA_array(i)
      i = Gdk::RGBA.new(*i)
    elsif i.is_a? Gdk::Color
      # =================================================================== #
      # Convert it into RGB values next.
      # =================================================================== #
      array = [
        (i.red   * 100.0 / 65535) / 100,
        (i.green * 100.0 / 65535) / 100,
        (i.blue  * 100.0 / 65535) / 100,
        1.0
      ]
      i = Gdk::RGBA.new(*array)
    end
    override_background_color(:normal, i)
    #end
    @background_colour = i
  end; alias set_background_color        set_background_colour # === set_background_color
       alias bg_colour                   set_background_colour # === bg_colour
       alias bg_colour=                  set_background_colour # === bg_colour=
       alias bg_color                    set_background_colour # === bg_colour
       alias background                  set_background_colour # === background
       alias background=                 set_background_colour # === background=
       alias background_color            set_background_colour # === background_color
       alias background_colour           set_background_colour # === background_colour
       alias background_colour=          set_background_colour # === background_colour=
       alias background_color=           set_background_colour # === background_color=
       alias set_background              set_background_colour # === set_background
       alias change_background_to        set_background_colour # === change_background_to
       alias change_background_colour_to set_background_colour # === change_background_colour_to
       alias change_background_color_to  set_background_colour # === change_background_color_to

  # ========================================================================= #
  # === add_css_class
  #
  # This method can be used to quickly add a CSS class to a particular
  # gtk-widget.
  #
  # Usage example how to use it via the gtk_paradise gem:
  #
  #   label = create_label('This is a test')
  #   label.add_css_class('BG_black')
  #   label.add_css_class('yellow')
  #   label.css_class('bblack2') # This variant is shorter.
  #
  # ========================================================================= #
  def add_css_class(
      i
    )
    if respond_to? :style_context
      i = i.to_s if i.is_a? Symbol
      add_to_the_CSS_style_context(i)
      # ===================================================================== #
      # Always register it after it was added.
      # No, this can not be done - we need to add CSS rules elsewhere.
      # ===================================================================== #
      # ::Gtk.append_these_css_rules(i)
    end
  end; alias set_class          add_css_class # === set_class
       alias set_css_class      add_css_class # === set_css_class
       alias css_class          add_css_class # === css_class
       alias css                add_css_class # === css
       alias add_this_css_class add_css_class # === add_this_css_class
       alias add_css            add_css_class # === add_this_css_class
       alias add_class          add_css_class # === add_class

  # ========================================================================= #
  # === add_or_append
  # ========================================================================= #
  def add_or_append(i)
    if Gtk.do_we_use_gtk3? # For gtk3.
      add(i)
    elsif Gtk.do_we_use_gtk4? # For gtk4.
      append(i)
    end
  end

  # ========================================================================= #
  # === enable_scroll_events
  #
  # This method will enable scroll-events for a particular widget, such
  # as a Gtk::Entry instance.
  #
  # Note that set_events() is less flexible as it can not be used on
  # widgets that have already been realized.
  # ========================================================================= #
  def enable_scroll_events
    # if ::Gtk.use_gtk2? # assume ruby-gtk2 here.
    #   add_events(Gdk::EventMask::SCROLL_MASK)
    # else
      add_events(:scroll_mask)
    # end
  end; alias add_scroll_event enable_scroll_events # === add_scroll_event

  # ========================================================================= #
  # === set_scroll_event
  # ========================================================================= #
  def set_scroll_event
    set_events(Gdk::EventMask::SCROLL_MASK) # Enable scroll-events here.
  end

  # ========================================================================= #
  # === rounded_border
  #
  # This method can be used to "assign" a rounded border to a gtk-widget.
  #
  # This functionality depends on CSS rules.
  #
  # Usage examples:
  #
  #   .rounded_border('12px')
  #   .rounded_border('12px', :default)
  #   .rounded_border('12px', :default, '10px solid darkgreen')
  #
  # ========================================================================= #
  def rounded_border(
      i                 = 12,
      use_this_CSS_name = :default,
      regular_border    = '1px solid steelblue' # This has to be supplied by the user.
    )
    case use_this_CSS_name
    when :default
      use_this_CSS_name = '.border'
    end
    if use_this_CSS_name.frozen?
      use_this_CSS_name = use_this_CSS_name.dup
    end
    css_rules_to_apply = "\n#{use_this_CSS_name}#{i.to_s.delete('px')} {\n".dup
    css_rules_to_apply << '  border: '+regular_border.to_s.delete(';')+';'+"\n"
    css_rules_to_apply << '  border-radius: '+i.to_s.delete('px')+'px;'+"\n"
    css_rules_to_apply << "}\n"
    ::Gtk.add_to_the_toplevel_CSS_string(css_rules_to_apply)
    _ = "#{use_this_CSS_name}#{i.to_s.delete('px')}".delete('.')
    css_class(_)
    return css_rules_to_apply # And return it as well, just in case.
  end

  # ========================================================================= #
  # === add_hspacer
  # ========================================================================= #
  def add_hspacer
    add(horizontal_line)
  end; alias add_horizontal_spacer add_hspacer # === add_horizontal_spacer

  # ========================================================================= #
  # === horizontal_line
  # ========================================================================= #
  def horizontal_line
    return ::Gtk.horizontal_spacer
  end

  # ========================================================================= #
  # === align_to_the_middle
  # ========================================================================= #
  def align_to_the_middle
    if respond_to? :set_alignment # Must safeguard it.
      set_alignment(0.5)
    end
  end

  # ========================================================================= #
  # === set_markify_text
  #
  # This method combines two different methods in one, aka .set_text()
  # and then a call to .do_markify().
  # ========================================================================= #
  def set_markify_text(i)
    set_text(i.to_s)
    do_markify
  end

  # ========================================================================= #
  # === focus_in_event
  # ========================================================================= #
  def focus_in_event(&block)
    signal_connect(:focus_in_event, &block)
  end; alias on_focus_in_event focus_in_event # === on_focus_in_event

  # ========================================================================= #
  # === grey_out
  #
  # This is typically used for gtk-button, to "grey" out the button, thus
  # make it no longer sensitive to user-input change.
  # ========================================================================= #
  def grey_out
    set_sensitive(false)
  end; alias not_sensitive  grey_out # === not_sensitive
       alias is_insensitive grey_out # === is_insensitive

  require 'gtk_paradise/toplevel_methods/determine_which_gtk_version_is_in_use.rb'
  # ========================================================================= #
  # === grey_in
  # ========================================================================= #
  def grey_in
    set_sensitive(true)
  end; alias is_sensitive_again grey_in # === is_sensitive_again
       unless respond_to?(:is_sensitive) or ::Gtk.use_gtk2? # This check is required for ruby-gtk2.
         alias is_sensitive grey_in # === is_sensitive
       end

  # ========================================================================= #
  # === children?
  # ========================================================================= #
  def children?
    children
  end

  # ========================================================================= #
  # === transition_change_background_to
  #
  # The first argument should be the target colour.
  # ========================================================================= #
  def transition_change_background_to(
      to_this_colour     = :lightgreen,
      delay_in_n_seconds = 3
    )
    old_name = name.to_s
    set_name("BG_#{to_this_colour}")
    Thread.new {
      sleep(delay_in_n_seconds) # Reinstate the old colour after n seconds.
      set_name(old_name)
    }
  end

  # ========================================================================= #
  # === align_into_the_center
  # ========================================================================= #
  def align_into_the_center
    set_halign(::Gtk::Align::CENTER)
  end

  # ========================================================================= #
  # === override_background_colour
  # ========================================================================= #
  def override_background_colour(i)
    override_background_color(:normal, i)
  end

  # ========================================================================= #
  # === do_not_expand
  #
  # This method can be used to force a button to stay at its original
  # size.
  # ========================================================================= #
  def do_not_expand
    set_hexpand(false) if respond_to? :set_hexpand
    set_vexpand(false) if respond_to? :set_vexpand
  end; alias dont_expand              do_not_expand # === dont_expand
       alias retain_the_original_size do_not_expand # === retain_the_original_size
       alias disallow_resizing        do_not_expand # === disallow_resizing
       alias do_not_resize            do_not_expand # === do_not_resize

  # ========================================================================= #
  # === rpack
  #
  # rpack stands for "right pack". This is ultimately just a wrapper over
  # .pack_end().
  # ========================================================================= #
  def rpack(
      widget,
      padding_to_use = 0
    )
    pack_end(
      widget,
      fill: true,
      expand: true,
      padding: padding_to_use
    )
  end

  # ========================================================================= #
  # === add_button_press_events
  # ========================================================================= #
  def add_button_press_events
    # if ::Gtk.use_gtk2?
    #   add_events(Gdk::Event::BUTTON_PRESS_MASK)
    if ::Gtk.use_gtk3? # else for ruby-gtk3
      add_events(Gdk::EventMask::BUTTON_PRESS_MASK)
    elsif ::Gtk.use_gtk4?
      e '.add_events() has to be changed for ruby-gtk4. '\
        'It is currently not working.'
    end
  end; alias enable_context_menu_on_button_click_events add_button_press_events # === enable_context_menu_on_button_click_events
       alias enable_context_menu                        add_button_press_events # === enable_context_menu

  # ========================================================================= #
  # === remove_css_class
  #
  # This method can be used to remove a particular CSS class from a 
  # widget.
  # ========================================================================= #
  def remove_css_class(
      this_CSS_class
    )
    context = style_context
    context.remove_class(this_CSS_class) # Remove the CSS class here.
  end; alias remove_this_css_class remove_css_class # === remove_this_css_class
       alias remove_CSS_class      remove_css_class # === remove_CSS_class

  # ========================================================================= #
  # === clear_background
  #
  # This method will clear the background.
  # ========================================================================= #
  def clear_background
    add_css_class('clear_background')
  end; alias clear_border      clear_background # === clear_border
       alias remove_background clear_background # === remove_background

  # ========================================================================= #
  # === enable_markup
  # ========================================================================= #
  def enable_markup
    set_use_markup(true)
  end; alias markify      enable_markup # === markify
       alias do_markify   enable_markup # === do_markify
       alias parse_markup enable_markup # === parse_markup

  # ========================================================================= #
  # === same_minimal
  #
  # This method can be used to group together several calls to
  # .minimal(), along with proper padding. In fact, the same padding
  # will be applied to each invocation of minimal(), which explains
  # the name chosen for this method.
  # ========================================================================= #
  def same_minimal(
      padding_to_use = 0,
      *these_widgets
    )
    these_widgets.flatten.each {|this_widget|
      minimal(this_widget, padding_to_use)
    }
  end

  # ========================================================================= #
  # === align_to_center
  #
  # This method will align towards the center. Note that not all widgets
  # respond to this naturally; it is best to use this for gtk-entry
  # widgets and perhaps gtk-label and similar widgets.
  #
  # Since as of May 2023 another approach is used: if the widget does
  # not respond to .xalign= then another method will be called
  # automatically. Unfortunately this does not work, so it had to be
  # reverted ...
  # ========================================================================= #
  def align_to_center
    # if respond_to?(:xaling=)
      self.xalign = 0.5
    # else
    #   self.halign = ::Gtk::Alignment.new(0.5, 0.5, 1.0, 1.0)
    # end
  end; alias align_to_the_center    align_to_center # === align_to_the_center
       alias center                 align_to_center # === center
       alias do_center              align_to_center # === do_center
       alias to_center              align_to_center # === to_center
       alias to_the_middle          align_to_center # === to_the_middle
       alias align_into_the_middle  align_to_center # === align_into_the_middle
       alias center_align           align_to_center # === center_align

  # ========================================================================= #
  # === make_bold
  #
  # This method can be used to make a widget bold. In ruby-gtk4 this has
  # changed, though. I am not 100% positive yet, but I think it has
  # changed.
  # ========================================================================= #
  def make_bold
    # if use_gtk3?
      override_font(
        Pango::FontDescription.new('bold')
      )
    #end
  end; alias bold make_bold # === bold

  # ========================================================================= #
  # === on_key_press_event
  # ========================================================================= #
  def on_key_press_event(&block)
    signal_connect(:key_press_event, &block)
  end

  # ========================================================================= #
  # === tooltip=
  # ========================================================================= #
  def tooltip=(i)
    ::Gtk.tooltip_for(self, i)
  end

  # ========================================================================= #
  # === on_value_changed
  # ========================================================================= #
  def on_value_changed(&block)
    signal_connect(:value_changed, &block)
  end

  # ========================================================================= #
  # === tooltip
  #
  # Attach a small tooltip to the button at hand.
  #
  # The argument to this method should be a String, denoting which
  # text should be displayed.
  # ========================================================================= #
  def tooltip(
      text_for_the_tooltip = ''
    )
    # if ::Gtk.use_gtk2?
    #   ::Gtk::Tooltips.new.set_tip(self, text_for_the_tooltip, nil)
    # else # else assume ruby-gtk3 or ruby-gtk4.
      if text_for_the_tooltip.is_a? String
        text_for_the_tooltip = { text: text_for_the_tooltip }
      end
      ::Gtk::Tooltip.new(text_for_the_tooltip)
    # end 
  end; alias use_this_tooltip tooltip # === use_this_tooltip
       # alias tooltip=         tooltip # === tooltip=

  # ========================================================================= #
  # === read_only
  # ========================================================================= #
  def read_only
    set_property('editable', false)
  end; alias it_can_not_be_edited read_only # === it_can_not_be_edited

  # ========================================================================= #
  # === set_width_request
  # ========================================================================= #
  def set_width_request(i = 85)
    set_property('width-request', i)
  end

  # ========================================================================= #
  # === set_height_request
  # ========================================================================= #
  def set_height_request(i = 85)
    set_property('height-request', i)
  end

  # ========================================================================= #
  # === red
  # ========================================================================= #
  def red
    modify_foreground(__method__)
  end

  # ========================================================================= #
  # === is_padded
  #
  # This method uses a hardcoded value for the "padding" in use.
  # ========================================================================= #
  def is_padded(i = 8)
    set_border_width(i)
  end

  # ========================================================================= #
  # === on_button_press_event
  # ========================================================================= #
  def on_button_press_event(&block)
    signal_connect(:button_press_event, &block)
  end

  # ========================================================================= #
  # === on_changed
  #
  # This method allows code such as the following example:
  #
  #   @combo_box.on_changed { sync_combo_box_to_main_gtk_entry }
  #
  # ========================================================================= #
  def on_changed(&block)
    signal_connect(:changed, &block)
  end; alias on_changed_event on_changed # === on_changed_event

  # ========================================================================= #
  # === on_switch_page
  #
  # This method is specifically for use in a Gtk::Notebook widget.
  # ========================================================================= #
  def on_switch_page(&block)
    signal_connect(:switch_page, &block)
  end; alias on_tab_switch   on_switch_page # === on_tab_switch
       alias on_switched_tab on_switch_page # === on_switched_tab

  # ========================================================================= #
  # === pad0px
  # ========================================================================= #
  def pad0px
    css_class('pad0px')
  end

  # ========================================================================= #
  # === pad1px
  # ========================================================================= #
  def pad1px
    css_class('pad1px')
  end

  # ========================================================================= #
  # === pad2px
  # ========================================================================= #
  def pad2px
    css_class('pad2px')
  end

  # ========================================================================= #
  # === pad3px
  # ========================================================================= #
  def pad3px
    css_class('pad3px')
  end

  # ========================================================================= #
  # === pad4px
  # ========================================================================= #
  def pad4px
    css_class('pad4px')
  end

  # ========================================================================= #
  # === pad5px
  # ========================================================================= #
  def pad5px
    css_class('pad5px')
  end

  # ========================================================================= #
  # === pad6px
  # ========================================================================= #
  def pad6px
    css_class('pad6px')
  end

  # ========================================================================= #
  # === pad7px
  # ========================================================================= #
  def pad7px
    css_class('pad7px')
  end

  # ========================================================================= #
  # === pad8px
  # ========================================================================= #
  def pad8px
    css_class('pad8px')
  end

  # ========================================================================= #
  # === pad9px
  # ========================================================================= #
  def pad9px
    css_class('pad9px')
  end

  # ========================================================================= #
  # === pad10px
  # ========================================================================= #
  def pad10px
    css_class('pad10px')
  end

  # ========================================================================= #
  # === pad11px
  # ========================================================================= #
  def pad11px
    css_class('pad11px')
  end

  # ========================================================================= #
  # === pad12px
  # ========================================================================= #
  def pad12px
    css_class('pad12px')
  end

  # ========================================================================= #
  # === pad13px
  # ========================================================================= #
  def pad13px
    css_class('pad13px')
  end

  # ========================================================================= #
  # === pad14px
  # ========================================================================= #
  def pad14px
    css_class('pad14px')
  end

  # ========================================================================= #
  # === pad15px
  # ========================================================================= #
  def pad15px
    css_class('pad15px')
  end

  # ========================================================================= #
  # === pad16px
  # ========================================================================= #
  def pad16px
    css_class('pad16px')
  end

  # ========================================================================= #
  # === pad17px
  # ========================================================================= #
  def pad17px
    css_class('pad17px')
  end

  # ========================================================================= #
  # === pad18px
  # ========================================================================= #
  def pad18px
    css_class('pad18px')
  end

  # ========================================================================= #
  # === pad19px
  # ========================================================================= #
  def pad19px
    css_class('pad19px')
  end

  # ========================================================================= #
  # === pad20px
  # ========================================================================= #
  def pad20px
    css_class('pad20px')
  end

  # ========================================================================= #
  # === mar1px
  # ========================================================================= #
  def mar1px
    css_class('mar1px')
  end

  # ========================================================================= #
  # === mar2px
  # ========================================================================= #
  def mar2px
    css_class('mar2px')
  end

  # ========================================================================= #
  # === mar3px
  # ========================================================================= #
  def mar3px
    css_class('mar3px')
  end

  # ========================================================================= #
  # === mar4px
  # ========================================================================= #
  def mar4px
    css_class('mar4px')
  end

  # ========================================================================= #
  # === mar5px
  # ========================================================================= #
  def mar5px
    css_class('mar5px')
  end

  # ========================================================================= #
  # === mar6px
  # ========================================================================= #
  def mar6px
    css_class('mar6px')
  end

  # ========================================================================= #
  # === mar7px
  # ========================================================================= #
  def mar7px
    css_class('mar7px')
  end

  # ========================================================================= #
  # === mar8px
  # ========================================================================= #
  def mar8px
    css_class('mar8px')
  end

  # ========================================================================= #
  # === mar9px
  # ========================================================================= #
  def mar9px
    css_class('mar9px')
  end

  # ========================================================================= #
  # === mar10px
  # ========================================================================= #
  def mar10px
    css_class('mar10px')
  end

  # ========================================================================= #
  # === width_height
  #
  # This method is just a little wrapper over .set_size_request().
  # ========================================================================= #
  def width_height(
      width  = 560,
      height = 160
    )
    set_size_request(width, height)
  end

  # ========================================================================= #
  # === pretty_tooltip
  # ========================================================================= #
  def pretty_tooltip(i)
    if i and i.include?('&')
      i = i.dup if i.frozen?
      # ===================================================================== #
      # Oddly enough, ruby-gtk2 complains about this character.
      #
      # The warning was: "Gtk-WARNING **: Failed to set text from
      # markup due to error parsing markup" [...] Entity did not 
      # end with a semicolon; most likely you used an ampersand 
      # character without intending to start an entity - escape 
      # ampersand as &amp;"
      # So we do this ^^^.
      # ===================================================================== #
      i.gsub!(/&/, '&amp;')
    end
    set_tooltip_markup(i)
  end; alias pretty_tooltip= pretty_tooltip # === pretty_tooltip=
       alias fancy_tooltip=  pretty_tooltip # === fancy_tooltip=
       alias fancy_tooltips= pretty_tooltip # === fancy_tooltips=
       alias hint=           pretty_tooltip # === hint=
       alias info            pretty_tooltip # === info
       alias popup_hint      pretty_tooltip # === popup_hint
       alias fancy_tip       pretty_tooltip # === fancy_tip
       alias fancy_hint      pretty_tooltip # === fancy_hint
       alias fancy_hint=     pretty_tooltip # === fancy_hint=
       alias set_hovertip    pretty_tooltip # === set_hovertip

  # ========================================================================= #
  # === on_scroll_event
  #
  # This is the event that occurs when the mouse-scroll-wheel is clicked.
  # ========================================================================= #
  def on_scroll_event(&block)
    signal_connect(:scroll_event, &block)
  end; alias on_scrolled           on_scroll_event # === on_scrolled
       alias on_scrolling          on_scroll_event # === on_scrolling
       alias on_scrolling_event    on_scroll_event # === on_scrolling_event
       alias on_mouse_scroll_event on_scroll_event # === on_mouse_scroll_event

  # ========================================================================= #
  # === make_bold_font
  #
  # This makes use of CSS rules, so it will not work for ruby-gtk2 by
  # default.
  # ========================================================================= #
  def make_bold_font
    add_css_class 'bold_font'
  end

  # ========================================================================= #
  # Next we will define some colour-CSS clases:
  # ========================================================================= #
  %w(
    bindigo1
    bdarkorchid1
    blightgreen1
    bnavy1
    bsteelblue1
    darkblue
    darkgreen
    royalblue
    slateblue
    steelblue
    tomato
  ).each {|this_colour_method|
    define_method(this_colour_method.to_sym) {
      add_css_class(__method__.to_s)
    }
  }

  # ========================================================================= #
  # === khaki_background
  # ========================================================================= #
  def khaki_background
    add_css_class('BG_khaki')
  end

  # ========================================================================= #
  # === light_green_background
  # ========================================================================= #
  def light_green_background
    add_css_class('BG_light_green')
  end; alias lightgreen_background light_green_background # === lightgreen_background

  # ========================================================================= #
  # === azure_background
  # ========================================================================= #
  def azure_background
    add_css_class('BG_azure')
  end; alias bg_azure azure_background # === bg_azure

  # ========================================================================= #
  # === bblack1px
  # ========================================================================= #
  def bblack1px
    css_class('bblack1')
  end; alias bblack1 bblack1px # === bblack1

  # ========================================================================= #
  # === bblack2px
  # ========================================================================= #
  def bblack2px
    css_class('bblack2')
  end; alias bblack2 bblack2px # === bblack2

  # ========================================================================= #
  # === bblack3px
  # ========================================================================= #
  def bblack3px
    css_class('bblack3')
  end; alias bblack3 bblack3px # === bblack3

  # ========================================================================= #
  # === bblack4px
  # ========================================================================= #
  def bblack4px
    css_class('bblack4')
  end; alias bblack4 bblack4px # === bblack4

  # ========================================================================= #
  # === bblack5px
  # ========================================================================= #
  def bblack5px
    css_class('bblack5')
  end; alias bblack5 bblack5px # === bblack5

  # ========================================================================= #
  # === hcenter
  #
  # The name "hcenter" is an abbreviation for "horizontal center" position.
  #
  # This variant has been confirmed to work for the gtk-hbox widget, in
  # July 2022.
  # ========================================================================= #
  def hcenter(
      i = :center
    )
    case i
    # ======================================================================= #
    # === :center
    # ======================================================================= #
    when :center
      i = ::Gtk::Align::CENTER
    end
    set_halign(i) # Or Gtk::Align::CENTER
  end; alias horizontal_center      hcenter # === horizontal_center
       alias halign_center          hcenter # === halign_center
       alias hjustify               hcenter # === hjustify
       alias center_this_widget     hcenter # === center_this_widget
       alias clever_center          hcenter # === clever_center
       alias center_into_the_middle hcenter # === center_into_the_middle
       alias centered_container     hcenter # === centered_container

  # ========================================================================= #
  # === do_focus
  #
  # This variant is a slightly nicer-looking variant compared to the
  # .set_focus(true) variant.
  # ========================================================================= #
  def do_focus
    set_focus(true)
  end; alias do_focus_on_it do_focus # === do_focus_on_it
       alias do_select      do_focus # === do_select

  # ========================================================================= #
  # === on_click_event
  #
  # This is a shortcut to respond to an on-click event, which is the
  # left mouse click normally.
  #
  # You can enable this in your code by doing:
  #
  #   .on_click_event { :highlight_text }
  #   .on_click_event { :deselect_text }
  #
  # ========================================================================= #
  def on_click_event
    if block_given?
      yielded = yield
      case yielded
      # ===================================================================== #
      # === :highlight_text
      # ===================================================================== #
      when :highlight_text,
           :select_everything,
           :select_text
        on_button_press_event { |widget, event|
          set_focus(true)
          select_region(0, -1)
        }
      # ===================================================================== #
      # === :deselect_text
      #
      # This will deselect any selection on a gtk-entry. It is not as
      # useful as the :highlight_text functionality, but I wanted to
      # provide both, simply to quickly show how it is done.
      # ===================================================================== #
      when :deselect_text
        on_button_press_event { |widget, event|
          set_focus(true)
          select_region(0, 0)
        }
      end 
    end
  end

  # ========================================================================= #
  # === shadow_type_etched_in
  # ========================================================================= #
  def shadow_type_etched_in
    set_shadow_type(::Gtk::SHADOW_ETCHED_IN)
  end

  # ========================================================================= #
  # === align_to_right
  # ========================================================================= #
  def align_to_right
    set_halign(:end) # halign=Gtk.Align.END
  end

  # ========================================================================= #
  # === enable_all_events
  #
  # This can be used for quick debugging, when you have a widget that is
  # not responding to some events, to simply turn them all on.
  #
  # This currently only works for ruby-gtk3. Not sure if we should enable
  # it for ruby-gtk2 as well.
  #
  # Note that the method-alias called "enable_enter_event" is, strictly
  # speaking, a misnomer. The reason why this was added still in August
  # 2021 was because it is simpler to "implement".
  # ========================================================================= #
  def enable_all_events(
      i = Gdk::EventMask::ALL_EVENTS_MASK
    )
    set_events(i) 
  end; alias all_events             enable_all_events # === all_events
       alias enable_events          enable_all_events # === enable_events
       alias enable_enter_event     enable_all_events # === enable_enter_event
       alias respond_to_enter_event enable_all_events # === respond_to_enter_event

  # ========================================================================= #
  # === on_mouse_button_clicked
  # ========================================================================= #
  def on_mouse_button_clicked(&block)
    signal_connect(:event) {|widget, event|
      if event.is_a?(Gdk::EventButton) and
        (event.event_type.name == 'GDK_BUTTON_PRESS')# and
        case event.button
        when 1,2,3 # These are all left-mouse-button events.
          yield if block_given?
        end
      end
    }
  end; alias on_mouse_click on_mouse_button_clicked # === on_mouse_click

  # ========================================================================= #
  # === sync_connect
  #
  # This method can be used to sync-connect two different widgets. This
  # is usually done to connect a combo-box to an entry, so whenever the
  # user changes the combo-box, the entry is also updated.
  #
  # Make sure that you pass the correct widgets to this method - I
  # accidentally passed the same widget into both arguments, which
  # of course leads to a failure.
  #
  # This method may have to be extended in the future, if we need
  # to combine other widgets more easily so.
  # ========================================================================= #
  def sync_connect(
      widget1,
      widget2
    )
    widget1.on_changed {
      text = widget1.text?.to_s
      widget2.set_text(text)
    }
  end

  # ========================================================================= #
  # === on_key_pressed
  #
  # When the user hits the enter key, we will do an action supplied
  # in via the yield method.
  # ========================================================================= #
  def on_key_pressed(
      this_key = :enter
    )
    signal_connect(:key_press_event) {|widget, event|
      if ::Gtk.enter_key?(event) # Handle the enter-key here.
        yield if block_given?
      end
    }
  end; alias click_event_when_enter_key_was_pressed on_key_pressed # === click_event_when_enter_key_was_pressed
       alias on_enter_key                           on_key_pressed # === on_enter_key
       alias on_enter_key_pressed                   on_key_pressed # === on_enter_key_pressed
       alias on_enter                               on_key_pressed # === on_enter
       alias on_enter_key_clicked                   on_key_pressed # === on_enter_key_clicked
       alias on_the_enter_key_was_pressed           on_key_pressed # === on_the_enter_key_was_pressed

  # ========================================================================= #
  # === font?
  #
  # This adds experimental support for simply querying the font in use.
  # ========================================================================= #
  def font?
    pango_context = self.pango_context
    font_description = pango_context.font_description
    font_in_use = font_description.to_s
    return font_in_use
  end

  # ========================================================================= #
  # === set_font
  #
  # This is a slightly simpler way to set the font. We also allow for
  # Symbols here, such as :hack_25. This refers to the font called "Hack",
  # with a size of 25. Not all symbols are enabled, mind you - I only use
  # some, but may extend more over the coming months and years.
  # ========================================================================= #
  def set_font(
      i = :dejavu_condensed_50
    )
    if i.is_a? Symbol
      i = ::Gtk.expand_this_font(i)
    end
    i = Pango::FontDescription.new(i)
    # if ::Gtk.use_gtk2?
    #   modify_font(i)
    if ::Gtk.use_gtk3?
      override_font(i)
    elsif ::Gtk.use_gtk4?
      # ===================================================================== #
      # Add it as a CSS rule next:
      # ===================================================================== #
      use_this_new_CSS_rule = '.font_'+i.to_s.downcase.tr(' ','_')+' {
  font-family: '+i.to_s+';
}
'
      css_class(use_this_new_CSS_rule)
    else
      e 'Unknown engine in use.'
    end
  end; alias font set_font # === font

  # ========================================================================= #
  # === increase_the_font_size_to
  #
  # The first argument has to be a percentage value for now. At a later
  # time this may change. The percentage value means a scaling factor,
  # such as "150%" meaning to upscale to 1.5 the former size.
  # ========================================================================= #
  def increase_the_font_size_to(to = '150%')
    # ======================================================================= #
    # First, obtain the old font size that is currently in use by the
    # widget at hand.
    # ======================================================================= #
    #old_font = ::Gtk::Settings.default.gtk_font_name
    pango_context = self.pango_context
    font_description = pango_context.font_description
    old_font = font_description.to_s
    # ======================================================================= #
    # Next, scale it up. The assumed value in old_font should look like
    # this:
    #
    #   "Noto Sans Bold 10"
    #
    # ======================================================================= #
    if old_font.include?(' ')
      splitted = old_font.split(' ')
      last_element = splitted.last # This should represent the font size at hand.
      scaled_up_last_element = last_element.to_f * (to.to_s.delete('%').to_f / 100.0)
      splitted[-1] = scaled_up_last_element.to_s
      new_font = splitted.join(' ')
      new_pango_font = Pango::FontDescription.new(new_font)
      # if ::Gtk.use_gtk2?
      #   modify_font(new_pango_font)
      # else # else assume gtk3
        override_font(new_pango_font)
      # end
    else
      e "#{old_font} does not contain any \" \" tokens."
    end
  end

  # ========================================================================= #
  # === on_button_release_event
  # ========================================================================= #
  def on_button_release_event(&block)
    signal_connect(:button_release_event, &block)
  end

  # ========================================================================= #
  # === on_activate
  #
  # This is the shortcut variant over the :activate signal. This signal
  # is typically triggered when the user clicks on the entry, via the
  # mouse cursor.
  # ========================================================================= #
  def on_activate(&block)
    signal_connect(:activate, &block)
  end; alias is_activated? on_activate # === is_activated?
    #  alias activate on_activate # === on_activate
    #  ^^^ this alias would lead to the following error: "rbgobj_signal.c:939: warning: previous definition of activate was here"

  # ========================================================================= #
  # === do_focus
  # ========================================================================= #
  def do_focus
    set_focus(true)
  end

  # ========================================================================= #
  # === background_colour?
  # ========================================================================= #
  def background_colour?
    return @background_colour
  end

  # ========================================================================= #
  # === simple_markup
  # ========================================================================= #
  def simple_markup(*i)
    _ = ''.dup
    i.flatten.each_slice(2).each {|text_to_display, colour_to_use|
      if colour_to_use
        _ << '<span weight="bold" foreground="'+colour_to_use.to_s+'">'+
             text_to_display.to_s+'</span>'
      else
        _ << text_to_display.to_s
      end
    }
    set_markup(_)
  end

  # ========================================================================= #
  # === pack_and_expand
  #
  # This method simplified the .pack_start() method.
  # ========================================================================= #
  def pack_and_expand(
      widget,
      padding_to_use = 0
    )
    # if ::Gtk.use_gtk2?
    #  pack_start(widget, true, true, padding_to_use)
    if ::Gtk.use_gtk4?
      append(widget)
      if padding_to_use
        css_class(
          'pad'+padding_to_use.to_s+'px'
        )
      end
    else
      pack_start(widget, expand: true, fill: true, padding: padding_to_use)
    end
  end; alias pack_with_expand pack_and_expand # === pack_with_expand
       alias pack_maximal     pack_and_expand # === pack_maximal
       alias maximal          pack_and_expand # === maximal

  # ========================================================================= #
  # === on_click_colour_change_to_this_colour
  # ========================================================================= #
  def on_click_colour_change_to_this_colour(
      use_this_colour = :mintcream,
      hash            = {}
    )
    signal_connect(:event) {|widget, event|
      if event_left_mouse_button_click?(event)
        old_name = name # Keep track of the original colour.
        set_name(use_this_colour.to_s)
        Thread.new {
          duration = hash[:duration]
          sleep duration
          set_name(old_name)
        }
      end
    }
  end

  # ========================================================================= #
  # === try_to_add_default_CSS_rules
  #
  # This is mostly an ad-hoc method to enable the default CSS files to
  # work on toplevel-created widgets. It was added in September 2021.
  # ========================================================================= #
  def try_to_add_default_CSS_rules
    ::Gtk.initialize_default_css_provider
  end; alias enable_free_form_css try_to_add_default_CSS_rules # === enable_free_form_css

  # ========================================================================= #
  # === on_mouse_double_click_event
  #
  # This method allows you to easily respond to mouse-double-click events.
  # ========================================================================= #
  def on_mouse_double_click_event(&block)
    signal_connect(:button_press_event) {|widget, event|
      if is_a_mouse_double_click_event?(event)
        yield
      end
    }
  end

  # ========================================================================= #
  # === is_a_mouse_double_click_event?
  # ========================================================================= #
  def is_a_mouse_double_click_event?(event)
    if ::Gtk.use_gtk3?
      case event.event_type.name
      when 'GDK_2BUTTON_PRESS' # Double-click event here.
        return true
      when 'GDK_BUTTON_PRESS'
        if event.button == 1
          return false # Could be true, though.
        else
          return false
        end
      end
    else
      false
    end
  end

  # ========================================================================= #
  # === on_right_mouse_button_clicked
  #
  # This is the event when the right mouse button is clicked.
  # ========================================================================= #
  def on_right_mouse_button_clicked(&block)
    signal_connect(:event) {|widget, event|
      if event.is_a?(Gdk::EventButton) and
        (event.event_type.name == 'GDK_BUTTON_PRESS') and
        (event.button == 3)
        yield if block_given?
      end
    }
  end; alias on_right_mouse_button_click       on_right_mouse_button_clicked # === on_right_mouse_button_click
       alias on_right_mouse_button_click_event on_right_mouse_button_clicked # === on_right_mouse_button_click_event

  # ========================================================================= #
  # === padding=
  # ========================================================================= #
  def padding=(i)
    set_border_width(i)
  end; alias padding padding= # === padding

  # ========================================================================= #
  # === connect_to
  #
  # Whenever the gtk-entry changes, we connect a change to another widget,
  # which is usually some spinner. This method can be used to achieve
  # precisely this.
  # ========================================================================= #
  def connect_to(this_widget)
    this_widget.enable_scroll_events
    signal_connect(:value_changed) {|widget, event|
      this_widget.set_text(text?)
    }
  end

  # ========================================================================= #
  # === on_toggled
  #
  # This method has been created primarily for class Gtk::CheckButton.
  # ========================================================================= #
  def on_toggled(&block)
    signal_connect(:toggled, &block)
  end

  # ========================================================================= #
  # === quit_on_child_exited_signal
  # ========================================================================= #
  def quit_on_child_exited_signal
    signal_connect(:child_exited) { gtk_main_quit }
  end

  # ========================================================================= #
  # === intelligent_exit
  # ========================================================================= #
  def intelligent_exit
    signal_connect(:destroy) { ::Gtk.main_quit }
  end

  # ========================================================================= #
  # === pack_without_expand                                     (minimal tag)
  # ========================================================================= #
  def pack_without_expand(
      widget,
      padding_to_use = 0
    )
    # ======================================================================= #
    # The second argument may be a Hash. If this is the case we check for
    # it's "type".
    # ======================================================================= #
    if padding_to_use.is_a? Hash
      work_on_a_copy = padding_to_use.dup # To avoid side-effects.
      # ===================================================================== #
      # === :padding
      # ===================================================================== #
      if work_on_a_copy.has_key? :padding
        padding_to_use = work_on_a_copy[:padding]
      end
    end
    # ======================================================================= #
    # === Handle ruby-gtk2 first
    # ======================================================================= #
    # if ::Gtk.use_gtk2?
    #   begin
    #     pack_start(
    #       widget,
    #       false,
    #       false,
    #       padding_to_use
    #     )
    #   rescue ArgumentError => error
    #     puts 'An error happened in '+__FILE__+' (method: pack_without_expand())'
    #     pp error
    #     puts 'caller() was:'
    #     pp caller()
    #     exit
    #   end
    if ::Gtk.use_gtk4?
      # ===================================================================== #
      # === Handle ruby-gtk4 here
      # ===================================================================== #
      append(
        widget
      )
      if padding_to_use
        add_css_class(
          "pad#{padding_to_use}px"
        )
      end
    else # else handle ruby-gtk3
      pack_start(
        widget,
        expand:  false,
        fill:    false,
        padding: padding_to_use
      )
    end
  end; alias pack_minimal pack_without_expand # === pack_minimal
       alias minimal      pack_without_expand # === minimal
    # ======================================================================= #
    # In the past we added an alias called append(). This is not
    # possible in ruby gtk4.
    # ======================================================================= #
    unless ::Gtk.use_gtk4?
       alias append       pack_without_expand # === append
    end

  # ========================================================================= #
  # === scroll_up?
  #
  # This method returns true if the scroll-event (of the middle mouse
  # button) is going upwards.
  # ========================================================================= #
  def scroll_up?(event)
    (event.event_type.name == 'GDK_SCROLL') and
    (event.direction == Gdk::ScrollDirection::UP)
  end; alias scroll_up_event? scroll_up? # === scroll_up_event?

  # ========================================================================= #
  # === scroll_down?
  #
  # This method returns true if the scroll-event (of the middle mouse
  # button) is going downwards.
  # ========================================================================= #
  def scroll_down?(event)
    (event.event_type.name == 'GDK_SCROLL') and
    (event.direction == Gdk::ScrollDirection::DOWN)
  end; alias scroll_down_event? scroll_down? # === scroll_down_event?

  # ========================================================================= #
  # === on_mouse_double_click_event
  #
  # This is currently (January 2021) unfinished. One day we want to
  # respond to double-click events.
  # ========================================================================= #
  def on_mouse_double_click_event_UNFINISHED_METHOD(
      event,
      &block
    )
    # ======================================================================= #
    # Assume ruby-gtk2 here:
    # ======================================================================= #
    #if ::Gtk.use_gtk2? and
    # (event.event_type == Gdk::Event::BUTTON2_PRESS) and
    # (event.button == 1)
    #  signal_connect(:button_press_event, &block)
    # ======================================================================= #
    # assume ruby-gtk3 here.
    # ======================================================================= #
    # elsif is_a_mouse_double_click_event?(event)
    if is_a_mouse_double_click_event?(event)
      signal_connect(:button_press_event, &block)
    end
  end; alias on_double_click on_mouse_double_click_event_UNFINISHED_METHOD # === on_double_click

  # ========================================================================= #
  # === complex_font
  # ========================================================================= #
  def complex_font(
      this_font = 'Calibre 32'
    )
    this_font = ::Gtk.expand_this_font(this_font)
    _ = Pango::FontDescription.new(this_font)
    override_font(_)
  end

  # ========================================================================= #
  # === use_this_font=
  # ========================================================================= #
  def use_this_font=(
      i = 'EmojiOne Color 22'
    )
    if i.is_a?(String) or i.is_a?(Symbol)
      i = ::Gtk.expand_this_font(i)
      i = Pango::FontDescription.new(i.to_s)
    end
    # if ::Gtk.use_gtk2?
    #  modify_font(i)
    if ::Gtk.use_gtk3?
      override_font(i)
    elsif ::Gtk.use_gtk4?
      puts 'override_font() has not yet been ported to ruby-gtk4  '\
           'and the gtk_paradise gem.'
    end
  end; alias use_this_font        use_this_font= # === use_this_font
       alias set_use_this_font    use_this_font= # === set_use_this_font
       alias try_to_use_this_font use_this_font= # === try_to_use_this_font
       alias fonty                use_this_font= # === fonty
       alias font=                use_this_font= # === font=

  require 'gtk_paradise/colours/parse.rb'
  # ========================================================================= #
  # === modify_foreground
  #
  # This method will modify (change) the foreground colour.
  # ========================================================================= #
  def modify_foreground(
      colour = :steelblue,
      state  = ::Gtk::StateType::NORMAL
    )
    if colour.is_a? Symbol
      # ===================================================================== #
      # Convert it into a Gtk::Colours representation.
      # ===================================================================== #
      colour = ::Gtk.colour_parse(colour)
    end
    # if ::Gtk.use_gtk2?
    #   modify_fg(
    #     state, colour # Flip it here. This is deliberately different to modify_background.
    #   )
    # else
    # ===================================================================== #
    # Else we assume ruby-gtk3 is in use.
    #
    # state: the state for which to set the color
    #
    # colour: the colour to assign
    # ===================================================================== #
    if state.is_a? ::Gtk::StateType
      # =================================================================== #
      # Must convert it into a StateFlag then.
      # For example, <Gtk::StateType normal> should be :NORMAL.
      #
      # The available StateFlags constants are:
      #
      #   [:ACTIVE, :INCONSISTENT, :PRELIGHT, :BACKDROP, :VISITED,
      #    :CHECKED, :LINK, :FOCUSED, :INSENSITIVE, :DROP_ACTIVE,
      #    :SELECTED, :NORMAL, :DIR_LTR, :DIR_RTL]
      #
      # =================================================================== #
      case state.inspect.to_s
      when /prelight/
        state = :PRELIGHT
      when /active/
        state = :ACTIVE
      when /normal/
        state = :NORMAL
      else
        puts 'Not yet registered state: '+state.inspect.to_s
      end
    end
    # ===================================================================== #
    # ruby-gtk3 wants Gdk::RGBA rather than Gdk::Color. So we must
    # convert this next.
    # ===================================================================== #
    if colour.is_a? Gdk::Color
      colour = ::Gtk.convert_gdk_colour_to_gdk_rgba_colour(colour)
      # =================================================================== #
      # Or:
      #   colour = Gdk::RGBA.new(*array)
      #   colour.parse(colour.to_s).to_s # (*array)
      # =================================================================== #
    end
    override_color(
      state, colour
    )
    # end
  end; alias set_colour          modify_foreground # === set_colour
       alias foreground_colour=  modify_foreground # === foreground_colour=
       alias foreground_color=   modify_foreground # === foreground_color=
       alias foreground_colour   modify_foreground # === foreground_colour
       alias foreground_color    modify_foreground # === foreground_color
       alias use_this_colour=    modify_foreground # === use_this_colour=
       alias set_use_this_colour modify_foreground # === set_use_this_colour
       alias change_colour_to    modify_foreground # === change_colour_to

end; end