#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === BasicCairoShapes
#
# This example shows how to use cairo to draw shapes.
#
# Usage example:
#
#   BasicCairoShapes.new
#
# =========================================================================== #
# require 'gtk_paradise/examples/basic_cairo_shapes.rb'
# =========================================================================== #
require 'gtk3'

class BasicCairoShapes < Gtk::Window

  # ========================================================================= #
  # === SIZE_FOR_THE_TRIANGLE
  # ========================================================================= #
  SIZE_FOR_THE_TRIANGLE = 50

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Basic shapes'

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      run_already = true
    )
    super()
    run if run_already
  end

  # ========================================================================= #
  # === initialize_the_drawing_area
  # ========================================================================= #
  def initialize_the_drawing_area
    @drawing_area = Gtk::DrawingArea.new  
    @drawing_area.signal_connect(:draw) { on_draw }
    add(@drawing_area)
  end

  # ========================================================================= #
  # === on_draw
  #
  # This is triggered by .initialize_the_drawing_area().
  # ========================================================================= #
  def on_draw
    cairo_context = @drawing_area.window.create_cairo_context  
    draw_shapes(cairo_context)
  end

  # ========================================================================= #
  # === draw_shapes
  #
  # This method will draw the various shapes.
  # ========================================================================= #
  def draw_shapes(cairo_context)
    # ======================================================================= #
    # Also draw a triangle:
    # ======================================================================= #
    triangle(cairo_context)
    # ======================================================================= #
    # Draw the 1st rectangle
    # ======================================================================= #
    cairo_context.set_source_rgb(0.6, 0.6, 0.6)
    cairo_context.rectangle(1, 20, 120, 280) # x, y, width, height 
    # ======================================================================= #
    # 1st square, which is essentially a rectangle
    # ======================================================================= #
    cairo_context.rectangle(180, 20, 80, 80)
    cairo_context.fill
    # ======================================================================= #
    # The 1st full circle
    # ======================================================================= #
    cairo_context.arc(330, 60, 40, 0, 2 * Math::PI)
    cairo_context.fill
    # ======================================================================= #
    # The 1st partial circle
    # ======================================================================= #
    cairo_context.arc(90, 160, 40, Math::PI/4, Math::PI)
    cairo_context.fill
    # ======================================================================= #
    # The 1st ellipsoid
    # ======================================================================= #
    cairo_context.translate 220, 180
    cairo_context.scale 1, 0.7 # Shrink the y-axis.
    cairo_context.arc 0, 0, 50, 0, 2 * Math::PI
    cairo_context.set_source_rgb(0.7, 0.7, 0.5) # Make it beige.
    cairo_context.fill
  end

  # ========================================================================= #
  # === triangle
  #
  # This triangle will be shown in a colour.
  # ========================================================================= #
  def triangle(i)
    i.move_to(400, 150) # The start-position, the tip of the triangle, on top.
    i.rel_line_to(SIZE_FOR_THE_TRIANGLE, 2 * SIZE_FOR_THE_TRIANGLE) # Draw to first point, to the right.
    i.rel_line_to(-2 * SIZE_FOR_THE_TRIANGLE, 0)   # Draw to the second point, to the left.
    i.close_path # This here will draw the missing third line actually.
    i.set_source_rgb(1, 0.3, 0.5) # Make it red.
    i.fill
  end

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    set_title TITLE
    signal_connect(:destroy) { Gtk.main_quit }
    initialize_the_drawing_area
    set_default_size(800, 500)
    set_window_position :center
    show_all
    move(0, 0)
  end

end

if __FILE__ == $PROGRAM_NAME
  BasicCairoShapes.new
  Gtk.main
end # rb basic_cairo_shapes.rb