#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
require 'gtk2'

module Pong

class CenteredItem

  attr_accessor :x
  attr_accessor :y

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      x, y, width, height
    )
    register_sigint
    @x = x
    @y = y
    @width = width
    @height = height
  end # start calculations

  # ========================================================================= #
  # === min_x
  # ========================================================================= #
  def min_x; @x - @width  / 2; end
  def max_x; @x + @width  / 2; end
  def min_y; @y - @height / 2; end
  def max_y; @y + @height / 2; end

  # ========================================================================= #
  # === height?
  # ========================================================================= #
  def height?
    @height
  end; alias height height? # === height

  # ========================================================================= #
  # === register_sigint
  # ========================================================================= #
  def register_sigint
    Signal.trap('SIGINT') { exit }
  end

  # ========================================================================= #
  # === width?
  # ========================================================================= #
  def width?
    @width
  end; alias width width? # === width

end; end

module Pong

# =========================================================================== #
# === Pong:: Window.new
# =========================================================================== #
class Window < Gtk::Window

  SPEED = 15

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(speed = SPEED)
    super()
    @game_pause = false
    @speed = speed

    self.title = 'Pong Demonstration'
    signal_connect(:destroy) { Gtk.main_quit }
    signal_connect(:key_press_event) { |widget, event|
      if event.state.control_mask? and event.keyval == Gdk::Keyval::GDK_q
        destroy
        true
      else
        false
      end
    }
    signal_connect(:key_press_event) { |widget, event|
      case event.event_type.name # event.event_type.name
      when 'GDK_KEY_PRESS'
        # a key was pressed:
        _ = Gdk::Keyval.to_name(event.keyval)
        case _
        when 'space'
          pause_game
        end
      else # puts event.event_type.name # pass thru for now 
      end
    }
    set_default_size(280, 240)
    @field = Field.new
    @drawing_area = Gtk::DrawingArea.new
    set_expose_event
    vb = Gtk::VBox.new(false, 7)
    vb.border_width = 12
    vb.pack_start(@drawing_area, true, true, 0)
    vb.show_all
    add(vb)
    Gtk.timeout_add(@speed) {
      @field.update unless @game_pause == true
      @drawing_area.queue_draw unless @drawing_area.destroyed?
    }
  end

  # ========================================================================= #
  # === pause_game
  # ========================================================================= #
  def pause_game # pause the game
    @game_pause ^= true
  end

  # ========================================================================= #
  # === set_expose_event
  # ========================================================================= #
  def set_expose_event
    @drawing_area.signal_connect(:expose_event) { |widget, event|
      cr = widget.window.create_cairo_context
      cr.scale(*widget.window.size)
      @field.draw(cr)
    }
  end

end; end

module Pong

class Field

  attr_accessor :width
  attr_accessor :height

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(margin = 0.05)
    @margin = margin
    reset
  end

  # ========================================================================= #
  # === reset
  # ========================================================================= #
  def reset
    @left_paddle = Paddle.new(self, @margin, 0.5)
    @right_paddle = Paddle.new(self, 1 - @margin, 0.7)
    @paddles = [ @left_paddle, @right_paddle ]
    @ball = Ball.new
  end

  # ========================================================================= #
  # === update
  # ========================================================================= #
  def update
    @paddles.each { |paddle| paddle.update(@ball) } # die schläger
    @ball.update
    @paddles.each { |paddle| paddle.update_ball(@ball) }
  end

  # ========================================================================= #
  # === draw
  # ========================================================================= #
  def draw(cr)
    cr.set_source_rgba(1, 1, 1)
    cr.rectangle(0, 0, 1, 1)
    cr.fill
    cr.save {
      cr.set_source_rgba(0.8, 0.8, 0.8, 0.8)
      cr.set_line_join(Cairo::LINE_JOIN_ROUND)
      @paddles.each { |paddle| cr.save { paddle.draw(cr) }}
    }
    cr.set_source_rgba(0, 0, 0)
    cr.save {@ball.draw(cr)}
  end

end; end


# =========================================================================== #
# The main namespace is module Pong.
# =========================================================================== #
module Pong

  # ========================================================================= #
  # === Pong::CenteredCircle
  # ========================================================================= #
  class CenteredCircle < CenteredItem

    # ======================================================================= #
    # === draw
    # ======================================================================= #
    def draw(cr)
      cr.translate(min_x, min_y)
      cr.scale(@width, @height)
      cr.arc(0.5, 0.5, 0.5, 0, 2 * Math::PI)
      cr.fill
    end

  end

  # ========================================================================= #
  #
  # ========================================================================= #
  class CenteredRect < CenteredItem

    def draw(cr)
      cr.translate(min_x, min_y)
      cr.scale(@width, @height)
      cr.rectangle(0, 0, 1, 1)
      cr.fill
    end

  end

  # ========================================================================= #
  # Ein Ball. Auch CenteredCircle.
  # Ball.new
  # ========================================================================= #
  class Ball < CenteredCircle

    attr_accessor :dx, :dy

    # ======================================================================= #
    # === initialize
    # ======================================================================= #
    def initialize(dx = 0.02, dy = 0.02)
      super(0.8, 0.5, 0.04, 0.04)
      @dx = dx
      @dy = dy
    end

    # ======================================================================= #
    # === update
    # ======================================================================= #
    def update
      @x += @dx
      @y += @dy
      # ball bouncing
      if max_y > 1
        @y = 1 - (max_y - 1)
        @dy *= -1
      elsif min_y < 0
        @y -= min_y
        @dy *= -1
      end
      if max_x > 1
        @x = 1 - (max_x - 1)
        @dx *= -1
      elsif min_x < 0
        @x -= min_x
        @dx *= -1
      end
    end
  end

  # ========================================================================= #
  # class Paddle.new
  # ========================================================================= #
  class Paddle < CenteredRect

    # ======================================================================= #
    # === initialize
    # ======================================================================= #
    def initialize(field, x, y)
      super(x, y, 0.05, 0.3)
      @field = field
    end

    # ======================================================================= #
    # === update
    # ======================================================================= #
    def update(ball)
      # is the ball coming towards us?
      if (ball.x < @x and ball.dx > 0) or
         (ball.x > @x and ball.dx < 0)
        # move to intercept it
        @y = ball.y #
      end
    end

    # ======================================================================= #
    # === ball_hit?
    # ======================================================================= #
    def ball_hit?(ball)
      ball.y > min_y and ball.y < max_y
    end

    # ======================================================================= #
    # === update_ball
    # ======================================================================= #
    def update_ball(ball)
      if ball_hit?(ball)
        if ball.min_x < @x and ball.max_x > min_x # hit our left side
          ball.x -= (ball.max_x - min_x)
          ball.dx = -ball.dx
        elsif ball.max_x > @x and ball.min_x < max_x # hit our right side
          ball.x += (max_x - ball.min_x)
          ball.dx = -ball.dx
        end
      end
    end
  end

end

Pong::Window.new.show_all
Gtk.main
# ruby $RUBY_GTK/GAM/PING_PONG/cairo-pong.rb