#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
require 'gtk2'

class ProgressBarWindow < Gtk::Window

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize
    super()
    set_title('ProgressBar Example')
    set_border_width(15)
    set_size_request(800, 350)

    vbox = Gtk::VBox.new(false, 6)
    add(vbox)

    # in a class, using variables with @ character, is instance variable
    # that means access it from everywhere in this class without 
    @progressbar = Gtk::ProgressBar.new
    vbox.pack_start(@progressbar, true, false, 0)

    button = Gtk::CheckButton.new('Show text')
    button.signal_connect(:toggled) {|button| on_show_text_toggled(button) }
    vbox.pack_start(button, true, true, 0)

    # The activity mode goes from left to right to left.
    button = Gtk::CheckButton.new('Activity mode')
    button.signal_connect(:toggled) {|button| on_activity_mode_toggled(button) }
    vbox.pack_start(button, true, true, 0)

    button = Gtk::CheckButton.new('Right to Left')
    button.signal_connect(:toggled) {|button| on_right_to_left_toggled(button) }
    vbox.pack_start(button, true, true, 0)

    @timeout_id = GLib::Timeout.add(50) {on_timeout}
    @activity_mode = false

    signal_connect(:destroy) { Gtk.main_quit }
    show_all
    move(0, 0)
  end

  # ========================================================================= #
  # === on_show_text_toggled
  # ========================================================================= #
  def on_show_text_toggled(button)
    show_text = button.active?
    if show_text
      text = 'some text'
    else
      text = nil
    end
    @progressbar.set_text(text)
    @progressbar.set_show_text(show_text)
  end

  # ========================================================================= #
  # === on_activity_mode_toggled
  # ========================================================================= #
  def on_activity_mode_toggled(button)
    @activity_mode = button.active?
    if @activity_mode
      # ===================================================================== #
      # This here is the pulsing-bar going left to right.
      # Via set_pulse_step() we can set the pulse-step size.
      # ===================================================================== #
      @progressbar.set_pulse_step(0.001)
      @progressbar.pulse
    else
     @progressbar.set_fraction(0.0)
    end
  end

  # ========================================================================= #
  # === on_right_to_left_toggled
  # ========================================================================= #
  def on_right_to_left_toggled(button)
    value = button.active?
    @progressbar.set_inverted(value) if @progressbar.respond_to? :set_inverted
  end

  # ========================================================================= #
  # === on_timeout
  # ========================================================================= #
  def on_timeout
    if @activity_mode
      @progressbar.pulse
    else
      new_value = @progressbar.fraction + 0.01
      new_value = 0 if new_value > 1
      @progressbar.set_fraction(new_value)
    end
    return true
  end

end

if __FILE__ == $PROGRAM_NAME
  ProgressBarWindow.new
  Gtk.main  
end