#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::ProgressBarWindow
# =========================================================================== #
# $GTK_PARADISE/examples/gtk3/007_progress_bar_example.rb
# =========================================================================== #
require 'gtk3'

module Gtk

class ProgressBarWindow < ::Gtk::Box

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'ProgressBar Example'

  # ========================================================================= #
  # === CSS_TO_USE
  #
  # Define all CSS rules here.
  # ========================================================================= #
  CSS_TO_USE = '

@define-color progress_bg_color #37C8AB;

progressbar {
  color: rgba(204, 215, 212, 0.7);
  font-feature-settings: "tnum";
}

#progressbar1 {
  background-color: snow;
  color: green;
  border-color: tomato;
  border-width: 35px;
  background-image:
    linear-gradient(
      to bottom,
      shade (@progress_bg_color, 1.3),
      shade (@progress_bg_color, 1.08)
    );
  border: 5px solid shade (@progress_bg_color, 0.9);
}

#progressbar1.horizontal > trough {
  min-width: 550px;
}

progressbar > trough {
  border: none;
  border-radius: 80px;
  background-color: #101415;
}

'

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize
    super(:vertical)
    set_title(TITLE) if respond_to?(:set_title)
    set_size_request(800, 350)

    # ======================================================================= #
    # Put it all onto the vertical box next.
    # ======================================================================= #
    @vbox = ::Gtk::Box.new(:vertical, 6)
    add(@vbox)

    # in a class, using variables with @ character, is instance variable
    # that means access it from everywhere in this class without 
    @progressbar = ::Gtk::ProgressBar.new
    @vbox.pack_start(@progressbar, expand: true, fill: false, padding: 0)

    button = ::Gtk::CheckButton.new('Show text')
    button.signal_connect(:toggled) {|button| on_show_text_toggled(button) }
    @vbox.pack_start(button, expand: true, fill: true, padding: 0)

    # The activity mode goes from left to right to left.
    button = ::Gtk::CheckButton.new('Activity mode')
    button.signal_connect(:toggled) {|button| on_activity_mode_toggled(button) }
    @vbox.pack_start(button, expand: true, fill: true, padding: 0)

    button = ::Gtk::CheckButton.new('Right to Left')
    button.signal_connect(:toggled) {|button| on_right_to_left_toggled(button) }
    @vbox.pack_start(button, expand: true, fill: true, padding: 0)

    # ======================================================================= #
    # Set the timeout next.
    # ======================================================================= #
    @timeout_id = GLib::Timeout.add(50) { on_timeout }
    @activity_mode = false
    create_and_add_another_progressbar
    signal_connect(:destroy) { ::Gtk.main_quit }
    run
  end

  # ========================================================================= #
  # === on_show_text_toggled
  # ========================================================================= #
  def on_show_text_toggled(button)
    show_text = button.active?
    if show_text
      text = 'some text'
    else
      text = nil
    end
    @progressbar.set_text(text)
    @progressbar.set_show_text(show_text)
  end

  # ========================================================================= #
  # === on_activity_mode_toggled
  # ========================================================================= #
  def on_activity_mode_toggled(button)
    @activity_mode = button.active?
    if @activity_mode
      # ===================================================================== #
      # This here is the pulsing-bar going left to right.
      # Via set_pulse_step() we can set the pulse-step size.
      # ===================================================================== #
      @progressbar.set_pulse_step(0.001)
      @progressbar.pulse
    else
      @progressbar.set_fraction(0.0)
    end
  end

  # ========================================================================= #
  # === on_right_to_left_toggled
  # ========================================================================= #
  def on_right_to_left_toggled(button)
    value = button.active?
    @progressbar.set_inverted(value) if @progressbar.respond_to? :set_inverted
  end

  # ========================================================================= #
  # === on_timeout
  # ========================================================================= #
  def on_timeout
    if @activity_mode
      @progressbar.pulse
    else
      new_value = @progressbar.fraction + 0.01
      if new_value > 1
        new_value = 0
      end
      @progressbar.set_fraction(new_value)
    end
    return true
  end

  # ========================================================================= #
  # === create_and_add_another_progressbar
  #
  # This second progress-bar is used as an indicator widget, so it won't
  # change.
  # ========================================================================= #
  def create_and_add_another_progressbar
    provider = ::Gtk::CssProvider.new
    provider.load(data: CSS_TO_USE)
    ::Gtk::StyleContext.add_provider_for_screen(
      Gdk::Screen.default,
      provider,
      Gtk::StyleProvider::PRIORITY_APPLICATION
    )

    progress_bar = ::Gtk::ProgressBar.new
    progress_bar.set_fraction(0.5) # 50%.
    progress_bar.set_show_text(true)
    progress_bar.style_context.add_class('GtkProgress')
    progress_bar.set_text('Hello world!')
    progress_bar.set_name('progressbar1')
    @vbox.pack_start(progress_bar, expand: true, fill: true, padding: 0)
  end

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    set_border_width(15)
    show_all
  end

end; end

if __FILE__ == $PROGRAM_NAME
  _ = Gtk::ProgressBarWindow.new
  window = ::Gtk::Window.new
  window.add(_)
  window.set_size_request(500, -1)
  window.move(0, 0)
  window.show_all
  Gtk.main  
end