#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# A brief demonstration of:
#
#   - Gtk::Window
#   - Gtk::ScrolledWindow
#   - Gtk::ListStore
#   - Gtk::TreeView
#   - Gtk::TreeModelSort
#   - Gtk::TreeModelFilter
#   - Gtk::CellRenderText
#   - Gtk::TreeViewcolumn
#   - Gtk::TreeIter
#
# =========================================================================== #
# require 'sorted_tree.rb'
# =========================================================================== #
# $GTK_PARADISE/examples/gtk3/sorted_tree.rb
# =========================================================================== #
require 'gtk3'

class SortedTree

  alias e puts

  # =========================================================================== #
  # === register_sigint
  # =========================================================================== #
  def register_sigint
    Signal.trap('SIGINT') { exit }
  end

  # =========================================================================== #
  # === initialize
  # =========================================================================== #
  def initialize
    register_sigint
    # ======================================================================= #
    # Create a list store with:
    #  - an Integer (ID)
    #  - a String (date)
    #  - a TrueClass (represents whether or not the row should be shown or 
    #   not)
    # ======================================================================= #
    list_store = Gtk::ListStore.new(Integer, String, TrueClass)
    # ======================================================================= #
    # Create a filtered list store that wraps the list store
    # and set the filter column criteria
    # ======================================================================= #
    list_store_filter = Gtk::TreeModelFilter.new(list_store)
    list_store_filter.visible_column = 2
    # ======================================================================= #
    # === Gtk::TreeModelSort
    #
    # Create a sorted list store and set the sorting function for the
    # first column.  Note that we are wrapping the sorted model around
    # filtered model -- which in turn wraps the original list store model.
    # ======================================================================= #
    list_store_sort = Gtk::TreeModelSort.new(list_store_filter)
    list_store_sort.set_sort_func(0) { |a, b| a[0] <=> b[0] }
    # ======================================================================= #
    # Fill the list store with stuff - 3 elements per iter.
    # ======================================================================= #
    10.times { |i|
      iter = list_store.append
      iter[0] = i
      iter[1] = Time.now.to_s
      iter[2] = true
    }
    # ======================================================================= #
    # Create a window with a size of 800x400 pixels next.
    # ======================================================================= #
    window = Gtk::Window.new
    window.set_size_request(800, 400)
    window.signal_connect(:destroy) { Gtk::main_quit }
    # ======================================================================= #
    # Create a treeview, giving it the sorted list store (which is
    # wrapping the
    # ======================================================================= #
    tree_view = Gtk::TreeView.new(list_store_sort)
    # ======================================================================= #
    # Create a scrollable window, with automatic scrollbars, and
    # pack it into the window
    # ======================================================================= #
    scrolled_window = Gtk::ScrolledWindow.new
    scrolled_window.set_policy(:automatic, :automatic)
    outer_vbox = Gtk::Box.new(:vertical)
    outer_vbox.add(scrolled_window)
    button_trigger_event = Gtk::Button.new(label: 'Trigger sorting-event')
    button_trigger_event.signal_connect(:clicked) {
      # tree_view.signal_emit(:button_press_event)
      e 'Currently not working - we want to emit a button-press event.'
    }
    outer_vbox.add(button_trigger_event)
    window << outer_vbox
    tree_view.signal_connect(:event) {|widget, event|
      case event.to_s
      when /Button/
        e 'The event-button was clicked.'
      end
    }
    scrolled_window.add_with_viewport(tree_view)
    # ======================================================================= #
    # Create a CellRenderer that's in charge of rendering the ID. Also
    # colourize it a little bit.
    # ======================================================================= # 
    renderer = Gtk::CellRendererText.new
    renderer.set_property('background', 'lavender')
    renderer.set_property('foreground', 'black')
    # ======================================================================= #
    # Create the ID column which uses the cell renderer defined above and
    # is hooked up to the first column of the model. Sorting is turned on
    # on this column (which also uses the first column in the model
    # to figure out what to sort)
    # ======================================================================= #
    id_column = Gtk::TreeViewColumn.new('ID', renderer, text: 0)
    id_column.sort_column_id = 0
    id_column.alignment = 0.5 # Center it.
    id_column.set_resizable(true)
    tree_view.append_column(id_column)
    # ======================================================================= #
    # Create a column that renders the current time (the second column
    # in the model.
    # ======================================================================= #
    time_column = Gtk::TreeViewColumn.new('Current Time', renderer, text: 1)
    tree_view.append_column(time_column)
    tree_view.signal_connect(:button_release_event) { |widget,event|
      e "The id is: #{widget.selection.selected[0]}"
    }
    tree_view.rules_hint = true
    tree_view.headers_clickable = true
    tree_view.columns.each_with_index {|content, index|
      content.set_sort_column_id(index)
    }
    tree_view.columns.each {|this_column|
      this_column.set_resizable(true)
    }
    # ======================================================================= #
    # Show everything
    # ======================================================================= #
    window.show_all
    window.move(0, 0)
    # ======================================================================= #
    # Every once in 10 seconds will we loop through each row in the list
    # and update the time and set half of the rows' visibility attribute 
    # to false and the other half to true.
    # ======================================================================= #
    GLib::Timeout.add(10_000) {
      list_store.each { |model, path, iter|
        iter[1] = Time.now.to_s
        iter[2] = rand(2) == 1
      }
    }
    # ======================================================================= #
    # Start the Gtk main loop next:
    # ======================================================================= #
    Gtk.main
  end

end

if __FILE__ == $PROGRAM_NAME
  SortedTree.new  
end