#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::Run
#
# This specialized class replaced the older class called
# Gtk::StandaloneAppsWindow. The latter is no longer in use
# past 2021.
#
# class Gtk::Run is used to embed standalone apps in GTK, by
# subclassing from Gtk::Window.
#
# Thus, esssentially, Gtk::Run is a specialized Gtk::Window.
#
# This is the main use case for this class, after all - it is just
# a fancified Gtk::Window widget, which allows you to put other,
# smaller widgets into it.
#
# You can also add shortcut_keys to it, in a fairly trivial manner.
#
# These days, this class is part of the gtk_paradise gem, and it may
# help you if you want to get up quick, feature-rich, standalone
# Gtk-Applications.
#
# See the bottom of this file for a more complete example as to how
# to use this class.
#
# This class will accept and evaluate any block given to it. That block
# could, for example, contain a Hash with a key called :title. This
# will then be used as a cue to set the title of the widget at once.
# =========================================================================== #
# require 'gtk_paradise/run/run.rb'
# Gtk.run_this_widget(ARGV)
# =========================================================================== #
require 'gtk3'

module Gtk

class Run < ::Gtk::Window # === Gtk::Run

  # ========================================================================= #
  # Add support from the module called Gtk::RunModule next. This step
  # is extremely important, as most functionality made available in
  # Gtk::Run depends on that module.
  # ========================================================================= #
  require 'gtk_paradise/run_module/run_module.rb'
  include ::Gtk::RunModule # This pulls in Gtk::BaseModule as well.

  # ========================================================================= #
  # === @runner
  # ========================================================================= #
  @runner = nil

  # ========================================================================= #
  # === Gtk::Run.runner?
  # ========================================================================= #
  def self.runner?
    @runner
  end

  # ========================================================================= #
  # === initialize
  #
  # The format specification that is used by default can be found in the
  # ARRAY_ARGUMENTS_TO_INITIALIZE. You can see that width comes first,
  # then height.
  # ========================================================================= #
  def initialize(
      commandline_arguments = ARGV,
      &block
    )
    super() # We must call super() to properly initialize the widget.
    reset
    register_sigint
    @runner = self
    create_accel_group # our accel groups, i.e. key shortcuts
    # create_accel_keys ← May have to re-enable this at a later time.
    add_delete_event
    # ======================================================================= #
    # === Handle Arrays next
    # ======================================================================= #
    if commandline_arguments.is_a? Array
      # ===================================================================== #
      # Must flatten here because of input like this: [[120, 120]]
      # ===================================================================== #
      commandline_arguments.flatten!
    end
    # ======================================================================= #
    # === Handle Hashes next
    # ======================================================================= #
    if commandline_arguments.is_a? Hash
      # ===================================================================== #
      # === :width
      # ===================================================================== #
      if commandline_arguments.has_key?(:width)
        set_width(commandline_arguments.delete(:width))
      end
      # ===================================================================== #
      # === :height
      # ===================================================================== #
      if commandline_arguments.has_key?(:height)
        set_height(commandline_arguments.delete(:height))
      end
      # ===================================================================== #
      # === :title
      # ===================================================================== #
      if commandline_arguments.has_key?(:title)
        set_my_title_then_apply_it(commandline_arguments.delete(:title))
        # =================================================================== #
        # The next line has to be explained: the clause here is only
        # entered if the user specifically passed :title to this class.
        # In that case, we will not allow any further modifications,
        # as it would make this entry point pointless otherwise.
        # =================================================================== #
        disallow_modification_of_the_title
      end
      # ===================================================================== #
      # === :add
      # ===================================================================== #
      if commandline_arguments.has_key? :add
        add(commandline_arguments.delete(:add))
      end
    end
    # ======================================================================= #
    # === Handle blocks given to the constructor next
    # ======================================================================= #
    if block_given?
      yielded = yield
      # ===================================================================== #
      # === Handle Symbols next
      # ===================================================================== #
      if yielded.is_a? Symbol
        case yielded
        # =================================================================== #
        # === use_gtk3
        #
        # This clause may be entered by issuing something like this:
        #
        #   run = Gtk::Run.new { :use_gtk3 }
        #
        # =================================================================== #
        when :use_gtk3
          Gtk.we_use_this_gtk_version = :gtk3
        # =================================================================== #
        # === :show_all
        #
        # An example for this entry-point would be like this:
        #
        #   Gtk.run { :show_all }
        #
        # =================================================================== #
        when :show_all
          show_all
        # =================================================================== #
        # === :add_a_new_accel_group
        # =================================================================== #
        when :add_a_new_accel_group,
             :auto_accel
          add_accel_group(@internal_hash[:accel_group] = ::Gtk::AccelGroup.new)
        end
      # ===================================================================== #
      # === Handle Hashes next
      # ===================================================================== #
      elsif yielded.is_a? Hash
        # =================================================================== #
        # === :title
        # =================================================================== #
        if yielded.has_key? :title
          set_my_title_then_apply_it(yielded.delete(:title))
        # =================================================================== #
        # === :my_title
        # =================================================================== #
        elsif yielded.has_key? :my_title
          set_my_title_then_apply_it(yielded.delete(:my_title))
        end
        # =================================================================== #
        # === :border_size
        # =================================================================== #
        if yielded.has_key? :border_size
          set_border_size(yielded.delete(:border_size))
        end
        # =================================================================== #
        # === :favicon
        # =================================================================== #
        if yielded.has_key? :favicon
          set_this_favicon(yielded.delete(:favicon))
        end
        # =================================================================== #
        # === :width
        # =================================================================== #
        if yielded.has_key? :width
          set_width(yielded.delete(:width))
        end
        # =================================================================== #
        # === :height
        # =================================================================== #
        if yielded.has_key? :height
          set_height(yielded.delete(:height))
        end
      end
    end
    # ======================================================================= #
    # In the past we used run() here, but since as of 2021 we will try
    # to require the user to do so in his or her own application instead.
    # ======================================================================= #
    # run if run_already
  end

  # ========================================================================= #
  # === reset
  # ========================================================================= #
  def reset
    reset_towards_the_default_state
  end

end

# =========================================================================== #
# === Gtk::App
# =========================================================================== #
App = Run # Legacy constant, aka Gtk::App.
# Runner = Run # This is a legacy "alias". In the future it may be removed.

# =========================================================================== #
# === Gtk.runner?
# =========================================================================== #
def self.runner?
  Gtk::Run.runner?
end

# =========================================================================== #
# === Gtk.run
#
# We also add a toplevel-method here for the Gtk namespace, called Gtk.run()
# or Gtk.app(), respectively.
#
# The block argument will be delegated towards class Gtk::App.
#
# Usage example:
#
#   r = Gtk.run { title: widget.my_title? }
#
# =========================================================================== #
def self.run(
    i = ARGV,
    &block
  )
  return ::Gtk::Run.new(i, &block) # Instantiate a new Gtk::Run object here.
end; self.instance_eval { alias app    run } # === Gtk.app
     self.instance_eval { alias runner run } # === Gtk.runner

# =========================================================================== #
# === Gtk.gtk_runner_factory
#
# This method is a factory-method. It will return a Gtk::Run class which
# will hold our child-widget. It will also automatically call methods
# that should make sense for most widgets that are ever created, on a
# regular PC system.
#
# The whole point of this method is to call as many useful methods as
# possible. This may not always be what you want, but it helps simplify
# our downstream code: after all we only want to use ONE method call
# and then be done with it.
# =========================================================================== #
def self.gtk_runner_factory(this_widget)
  r = ::Gtk.run
  r << this_widget
  if this_widget.respond_to?(:width?) and this_widget.respond_to?(:height?)
    r.set_default_size(
      this_widget.width?,
      this_widget.height?
    )
  end
  r.set_padding(padding?) if this_widget.respond_to?(:set_padding)
  r.set_border_size(border_size?) if this_widget.respond_to?(:set_border_size)
  r.use_this_font = this_widget.font? if this_widget.respond_to?(:font?)
  r.easy_exit
  r.background_colour :white
  r.automatic_top_left_then_run
  return r
end; self.instance_eval { alias runner_factory gtk_runner_factory } # === Gtk.runner_factory

# =========================================================================== #
# === Gtk.run_this_widget
#
# The idea behind this method is to enable a "one-line" instantiation of
# widgets, following a common format. This file, in turn, makes use of
# Gtk::Run.
#
# Code for enabling this may look like the following:
#
#   require 'gtk_paradise/requires/run_this_widget.rb'
#
#   Gtk.run_this_widget(
#     Roebe::Gui::Gtk::ShowAliases
#   ) {{ position: :top_left }}
#
# Or just simpler:
#
#   Gtk.run_this_widget(Roebe::Gui::Gtk::ShowAliases)
#
# Input to this method should be a constant "pointing" to a particular
# class, which has to respond to the method .new(). This method will
# then be called, at the "instantiated_widget" line.
# =========================================================================== #
def self.run_this_widget(i = Box)
  i = i.join(' ').strip if i.is_a? Array
  if i.is_a?(String) and i.end_with?('.rb')
    require i # Add support for loading .rb files.
  end
  instantiated_widget = i.new # Instantiate the class.
  # ========================================================================= #
  # === Set sensible defaults next:
  #
  # We will use a default-setter, which can lateron be overriden by
  # the widget that is passed. This may not always be what the user
  # wanted to have, but this is then a case where the user should
  # NOT use this method here. This method is really a convenience
  # method to get things going, reducing the boilerplate-code.
  # ========================================================================= #
  r = Gtk.run(
    '85%',
    '60%',
   :tabble,     # The default favicon.
   'Title of the widget',  # The default title.
    5           # The default padding.
  )
  r << instantiated_widget # Add this widget to Gtk::Runner here.
  r.enable_quick_exit # Enable alt+x to quit.
  # r.show_all
  # ========================================================================= #
  # Customize Gtk::Runner next, if the child-widget responds to these
  # methods.
  # ========================================================================= # 
  if instantiated_widget.respond_to? :title?
    r.set_title(instantiated_widget.title?)
  end
  # ========================================================================= #
  # === :width?
  # ========================================================================= #
  if instantiated_widget.respond_to? :width?
    r.set_width(instantiated_widget.width?)
  end
  # ========================================================================= #
  # === :height?
  # ========================================================================= #
  if instantiated_widget.respond_to? :height?
    r.set_height(instantiated_widget.height?)
  end
  # ========================================================================= #
  # === :favicon?
  # ========================================================================= #
  if instantiated_widget.respond_to? :favicon?
    r.set_favicon(instantiated_widget.favicon?)
  end
  # ========================================================================= #
  # === Handle blocks next
  # ========================================================================= #
  if block_given?
    yielded = yield
    if yielded.is_a? Hash
      # ===================================================================== #
      # === :position
      # ===================================================================== #
      if yielded.has_key? :position
        case yielded[:position]
        # =================================================================== #
        # === :top_left
        # =================================================================== #
        when :top_left
          r.top_left
        end
      end
    end
  end
  r.run # And finally, .run() it.
  return r # And explicitely return Gtk::Runner here, just in case we may need it. 
end

end

if __FILE__ == $PROGRAM_NAME
  alias e puts
  run = Gtk::Run.new(title: 'Yo there') { :use_gtk3 }
  button = Gtk.button('Hey there')
  button.on_clicked {
    e button.text?
  }
  run.add(button)
  run.show_all
  run.set_size_request(400, 100)
  run.run
  # ========================================================================= #
  # The code here in the first if-clause just demonstrates the above
  # functionality. It is the default "testing code" for this .rb file.
  # ========================================================================= #
  # if ARGV.empty?
  #   require 'roebe/gui/gtk3/show_aliases/show_aliases.rb'
  #   Gtk.run_this_widget(
  #     Roebe::Gui::Gtk::ShowAliases
  #   ) {{ position: :top_left }}
  # else
  #   # This here is unfinished code. In case we have a .rb file, we
  #   # instantiate after we load that file.
  #   Gtk.run_this_widget(
  #     ARGV.first
  #   ) {{ position: :top_left }}
  # end
end