#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::RunModule
#
# This module is specifically tailored towards a Gtk::Window or a
# Gtk::Application widget.
#
# It contains parts of a Gtk::App that can be used like a module. This
# thus allows us more flexibility in how to use this module, as we
# can also re-use it in other classes.
#
# You can also include this module into instances of Gtk::Window via
# extend(). Look at the end of this file for an example in this regard.
#
# Up until November 2023, the name of this module was Gtk::AppModule.
# =========================================================================== #
# require 'gtk_paradise/run_module/run_module.rb'
# include ::Gtk::RunModule
# =========================================================================== #
module Gtk

module RunModule # === Gtk::RunModule

  # ========================================================================= #
  # First, add all necessary and useful requires.
  # ========================================================================= #
  require 'fileutils'
  require 'yaml'

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === DEFAULT_WIDTH
  # ========================================================================= #
  DEFAULT_WIDTH = '60%'

  # ========================================================================= #
  # === DEFAULT_HEIGHT
  # ========================================================================= #
  DEFAULT_HEIGHT = '40%'

  # ========================================================================= #
  # === ARRAY_ARGUMENTS_TO_INITIALIZE
  #
  # Specify the important attributes that we will use by default.
  #
  # Since as of January 2023 this is no longer really in use - or,
  # at the least, not hugely important anymore.
  #
  # These include:
  #
  #   :width
  #   :height
  #   :title
  #   :border
  #   :background_colour
  #   :favicon
  #   :padding
  #
  # ========================================================================= #
  ARRAY_ARGUMENTS_TO_INITIALIZE = [
    :width,
    :height,
    :title,
    :border,
    :background_colour,
    :favicon,
    :padding
  ]

  # ========================================================================= #
  # === original_width?
  #
  # Determine the default width of the screen on the target computer system.
  # ========================================================================= #
  def original_width?
    Gdk::Screen.default.width
  end; alias max_width? original_width? # === max_width?

  # ========================================================================= #
  # === disallow_resizing
  #
  # This method will prevent the user from resizing the window.
  # ========================================================================= #
  def disallow_resizing
    set_resizable(false)
  end

  # ========================================================================= #
  # === is_resizable
  # ========================================================================= #
  def is_resizable
    self.resizable = true
  end

  # ========================================================================= #
  # === do_maximize
  #
  # Use this when you want to maximize the window, to the full-screen.
  #
  # For this to work, the widget has to respond to .maximize(), which
  # is the case for a Gtk::Window.
  # ========================================================================= #
  def do_maximize
    self.maximize
  end; alias max do_maximize # === max

  # ========================================================================= #
  # === window_position_none
  # ========================================================================= #
  def window_position_none(
      i = :none
    )
    set_window_position(i)
  end

  # ========================================================================= #
  # === filename=
  # ========================================================================= #
  def filename=(i)
    @internal_hash[:filename] = i
  end; alias set_filename filename= # === set_filename

  # ========================================================================= #
  # === filename?
  # ========================================================================= #
  def filename?
    @internal_hash[:filename]
  end

  # ========================================================================= #
  # === set_my_border_width
  #
  # Set the border-width for Gtk::Runner.
  # ========================================================================= #
  def set_my_border_width(
      i = 0
    )
    case i
    when nil
      i = 0
    end
    @internal_hash[:border_width] = i.to_i
    set_border_width(@internal_hash[:border_width])
  end; alias padding=        set_my_border_width # === padding=
       alias border=         set_my_border_width # === border=
       alias set_padding     set_my_border_width # === set_padding
       alias set_border_size set_my_border_width # === set_border_size

  # ========================================================================= #
  # === middle_of_the_screen_then_run
  #
  # Use this method if you want to position the Gtk::App widget in the
  # center of the screen, before calling .run() on it.
  # ========================================================================= #
  def middle_of_the_screen_then_run
    middle_of_the_screen
    run
  end; alias middle_then_run     middle_of_the_screen_then_run # === middle_then_run
       alias start_in_the_middle middle_of_the_screen_then_run # === start_in_the_middle

  # ========================================================================= #
  # === top_left_then_show_then_run
  # ========================================================================= #
  def top_left_then_show_then_run
    top_left
    show
    run
  end

  # ========================================================================= #
  # === set_internal_child_widget
  # ========================================================================= #
  def set_internal_child_widget(i)
    @internal_hash[:child_widget] = i
  end; alias set_child_widget set_internal_child_widget # === set_internal_child_widget

  # ========================================================================= #
  # === border_width?
  # ========================================================================= #
  def border_width?
    @internal_hash[:border_width]
  end; alias padding? border_width? # === padding?

  # ========================================================================= #
  # === set_my_title_then_apply_it
  # ========================================================================= #
  def set_my_title_then_apply_it(i)
    if @internal_hash[:may_we_modify_the_title]
      i = __FILE__ if i == '__FILE__'
      set_my_title(i)
      # ======================================================================= #
      # Next delegate towards apply_the_internal_title().
      # ======================================================================= #
      apply_the_internal_title
    end
  end; alias set_title_then_apply set_my_title_then_apply_it # === set_title_then_apply

  # ========================================================================= #
  # === infer_the_title
  # ========================================================================= #
  def infer_the_title(
      i = :obtain_the_title_from_the_child_widget
    )
    case i
    # ======================================================================= #
    # === :obtain_the_title_from_the_child_widget
    # ======================================================================= #
    when :obtain_the_title_from_the_child_widget
      if @internal_hash[:child_widget]
        i = @internal_hash[:child_widget].my_title?
      end
    end
    set_my_title_then_apply_it(i)
  end

  # ========================================================================= #
  # === easy_exit_top_left_then_run
  #
  # This method combines the easy-exit functionality, with the "put widget
  # window on the top left corner of the screen", before "running" the
  # application at hand (aka, starting it).
  # ========================================================================= #
  def easy_exit_top_left_then_run
    easy_exit
    top_left_then_run
  end; alias easy_exit_then_top_left_and_run          easy_exit_top_left_then_run # === easy_exit_then_top_left_and_run
       alias enable_quick_exit_then_top_left_then_run easy_exit_top_left_then_run # === enable_quick_exit_then_top_left_then_run
       alias easy_exit_then_top_left_then_run         easy_exit_top_left_then_run # === easy_exit_then_top_left_then_run

  # ========================================================================= #
  # === top_left_and_show_all
  # ========================================================================= #
  def top_left_and_show_all
    top_left
    show_all
  end; alias top_left_then_show     top_left_and_show_all # === top_left_then_show
       alias top_left_then_show_all top_left_and_show_all # === top_left_then_show_all
       alias top_left_show_all      top_left_and_show_all # === top_left_show_all

  # ========================================================================= #
  # === top_left_then_run
  #
  # This method allows an optional argument, which should be a gtk-widget
  # that is to be added. This then eliminates a step such as:
  #
  #   r << @box
  #
  # ========================================================================= #
  def top_left_then_run(
      optional_add_this_widget = nil,
      &block
    )
    if optional_add_this_widget
      # ===================================================================== #
      # === Handle Hashes next. We have several "aliases" such as
      #     :with, :with_this_widget or :use_this_widget.
      # ===================================================================== #
      if optional_add_this_widget.is_a? Hash
        # =================================================================== #
        # === :use_this_widget
        #
        # This entry point is entered e. g. when code like this is used:
        #
        #   r.top_left_then_run(use_this_widget: dictionary)
        #
        # =================================================================== #
        if optional_add_this_widget.has_key? :use_this_widget
          optional_add_this_widget = optional_add_this_widget.delete(:use_this_widget)
        # =================================================================== #
        # === :with
        # =================================================================== #
        elsif optional_add_this_widget.has_key? :with
          optional_add_this_widget = optional_add_this_widget.delete(:with)
        # =================================================================== #
        # === :with_this_widget
        # =================================================================== #
        elsif optional_add_this_widget.has_key? :with_this_widget
          optional_add_this_widget = optional_add_this_widget.delete(:with_this_widget)
        end
      end
      add(optional_add_this_widget)
    end
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      case yielded
      # ===================================================================== #
      # === :easy_exit
      #
      # This entry point handles code such as this:
      #
      #   r.top_left_then_run { :easy_exit }
      #
      # ===================================================================== #
      when :easy_exit
        enable_quick_exit
      end
    end
    top_left
    run
  end; alias top_left_run           top_left_then_run # === top_left_run
       alias move_top_left_then_run top_left_then_run # === move_top_left_then_run

  # ========================================================================= #
  # === middle_of_the_screen
  #
  # This method allows the user to position the window in the middle of
  # the screen.
  # ========================================================================= #
  def middle_of_the_screen(
      use_this_as_the_positional_argument =
        ::Gtk::WindowPosition::CENTER
    ) 
    set_window_position(
      use_this_as_the_positional_argument
    )
  end; alias in_the_middle middle_of_the_screen # === in_the_middle

  # ========================================================================= #
  # === modify_bg
  #
  # Modify the background here.
  # ========================================================================= #
  def modify_bg(
      state,
      colour = 'lightblue'
    )
    if colour.is_a? String
      colour = Gdk::Color.parse(colour)
    elsif colour.is_a? NameError
      pp caller()
    end
    # if use_gtk2?
    #   super(state, colour)
    e 'Presently override_background_color(state, colour) is '\
      'not implemented.'
  end

  # ========================================================================= #
  # === max_resolution?
  # ========================================================================= #
  def max_resolution?
    ::Gtk.screen_resolution?
  end

  # ========================================================================= #
  # === position=
  #
  # Usage example:
  #
  #   .position = :mouse
  #
  # ========================================================================= #
  def position=(i)
    case i # case tag
    # ======================================================================= #
    # === :mouse
    # ======================================================================= #
    when :mouse
      i = Gtk::Window::POS_MOUSE if ::Gtk.use_gtk2?
    # ======================================================================= #
    # === :none
    # ======================================================================= #
    when :none
      i = Gtk::Window::POS_NONE if ::Gtk.use_gtk2?
    # ======================================================================= #
    # === :center_always
    # ======================================================================= #
    when :center_always,
         :always_center,
         :center2,
         nil
      i = Gtk::Window::POS_CENTER_ALWAYS
    # ======================================================================= #
    # === :center_on_parent
    # ======================================================================= #
    when :center_on_parent
      i = Gtk::Window::POS_CENTER_ON_PARENT
    # ======================================================================= #
    # === :center
    # ======================================================================= #
    when :center
      i = Gtk::Window::POS_CENTER
    else
      e "We do not know: #{simp(i.to_s)}"
    end
    set_window_position(i)
  end

  require 'gtk_paradise/toplevel_methods/screen_resolution.rb'
  # ========================================================================= #
  # === resolution?
  #
  # Query the resolution of the target computer.
  # ========================================================================= #
  def resolution?
    ::Gtk.resolution?
  end

  # ========================================================================= #
  # === connect_key
  #
  # This method can be used to simply map a String, such as 'alt+1' to
  # a specific key combination. In fact: it is required that the input
  # at hand includes a '+' character.
  #
  # Usage example:
  #
  #   connect_key('alt+1') { 'xid' }
  #
  # ========================================================================= #
  def connect_key(i)
    unless i.include? '+'
      e 'The input to connect_key() needs to have a "+" character.'
      return
    end
    splitted = i.split('+')
    key      = splitted[0]
    shortcut = splitted[1]
    case key.to_sym
    # ======================================================================= #
    # === :control
    # ======================================================================= #
    when :control,
         :ctrl,
         :strg,
         :str,
         nil,
         :default,
         '',
         'strg',
         'string'
      key = :control_mask
    # ======================================================================= #
    # === :alt
    # ======================================================================= #
    when :alt
      key = Gdk::Window::MOD1_MASK
    # ======================================================================= #
    # === :shift
    # ======================================================================= #
    when :shift,
         :pfeil_oben
      key = Gdk::Window::SHIFT_MASK
    end
    # ======================================================================= #
    # Obtain the shortcut-key next, which wraps over Gdk::Keyval.from_name().
    # ======================================================================= #
    shortcut = Gtk.get_shortcut_key(key)
    @internal_hash[:accel_group].connect(shortcut, key, :visible) {
    #   # then we use arguments
    #   if this_method.include? '('
    #     _ = this_method.split '('
    #     method_name= _[0]
    #     argument_to_it= _[1][0..-2] # get arguments
    #     child_widget?.send(method_name, argument_to_it)
    #   else
    #     @child_widget.send(this_method)
    #   end
      e 'PRESSED!'
    }
    # ======================================================================= #
    # The next part is probably debug-related code, but I forgot to add
    # a comment explaining what this code was supposed to be used for. :(
    # result = yield if block_given? # Example: xid
    # if result
    #   e 'The result was:'
    #   pp result
    # end
    # ======================================================================= #
  end

  # ========================================================================= #
  # === add_delete_event
  #
  # Register the :delete_event here.
  # ========================================================================= #
  def add_delete_event
    if use_gtk4?
      e 'The .delete_event() has not yet been added to ruby-gtk4.'
      e 'At a later time this will be reviewed.'
    else
      signal_connect(:delete_event) { exit_application }
    end
  end

  # ========================================================================= #
  # === favicon?
  # ========================================================================= #
  def favicon?
    @internal_hash[:favicon]
  end

  # ========================================================================= #
  # === automatic_top_left_then_run
  # ========================================================================= #
  def automatic_top_left_then_run
    automatic_size_then_automatic_title
    top_left_then_run
  end

  # ========================================================================= #
  # === automatic_middle_then_run
  # ========================================================================= #
  def automatic_middle_then_run
    automatic_size_then_automatic_title
    middle_then_run
  end

  # ========================================================================= #
  # === create_accel_group
  #
  # Normally you dont need to change anything here.
  # ========================================================================= #
  def create_accel_group
    @internal_hash[:accel_group] = ::Gtk::AccelGroup.new
    add_accel_group(@internal_hash[:accel_group]) if respond_to?(:add_accel_group)
  end; private :create_accel_group

  # ========================================================================= #
  # === enable_simple_exit_then_run
  # ========================================================================= #
  def enable_simple_exit_then_run
    enable_simple_exit
    run
  end

  # ========================================================================= #
  # === add_context_menu_with_the_default_accel_group
  # ========================================================================= #
  def add_context_menu_with_the_default_accel_group
    add_accel_group(
      @internal_hash[:accel_group] = gtk_accel_group
    )
    enable_context_menu
  end

  # ========================================================================= #
  # === default_width_and_height
  #
  # This variant is a bit different for two reasons:
  #
  #   (1) by default it will use a percentage value for both width
  #       and height
  #   (2) it will also do a set_size_request() to try to get the
  #       new width and height; in particular this point is quite
  #       different to the regular .width_height() method.
  #
  # ========================================================================= #
  def default_width_and_height(
      width  = '80%',
      height = '75%'
    )
    set_width(width)
    set_height(height)
    set_size_request(width?, height?)
  end; alias default_width_height       default_width_and_height # === default_width_height
       alias determine_width_and_height default_width_and_height # === determine_width_and_height

  # ========================================================================= #
  # === autotitle
  # ========================================================================= #
  def autotitle(
      title = title?
    )
    set_title(title) if title
  end

  # ========================================================================= #
  # === set_size
  # ========================================================================= #
  def set_size(
      width  = width?,
      height = height?
    )
    set_size_request(width, height)
  end

  # ========================================================================= #
  # === determine_the_title_from_the_child_widget
  #
  # The first argument is the widget that we will use, under the assumption
  # that it will be the child-widget.
  # ========================================================================= #
  def determine_the_title_from_the_child_widget(
      child_widget = child_widget?
    )
    if child_widget
      if child_widget.respond_to?(:my_title?)
        title = child_widget.my_title?
        set_my_title_then_apply_it(title)
      else
        e 'The child-widget (class '+child_widget.name.to_s+
          ') does not respond to the method called .my_title?' unless child_widget.nil?
      end
    else
      # Next we will display any error in a pretty variant.
      # Well, this is spammy, so it was disabled in December 2023.
      # May have to re-enable again one day.
      # e tomato('In the method ')+
      #   steelblue(__method__.to_s+'()')+
      #   tomato(':')
      # e tomato('No child-widget was provided. Showing '+
      #   'the caller() trace next:')
      # e
      # result = caller()
      # result.each {|line|
      #   e mediumseagreen("  #{line}")
      # }
      # e
    end
  end; alias determine_the_title                   determine_the_title_from_the_child_widget # === determine_the_title
       alias determine_the_title_automatically     determine_the_title_from_the_child_widget # === determine_the_title_automatically
       alias infer_the_title_from_the_child_widget determine_the_title_from_the_child_widget # === infer_the_title_from_the_child_widget
       alias title_from_child_widget               determine_the_title_from_the_child_widget # === title_from_child_widget
       alias automatic_title                       determine_the_title_from_the_child_widget # === automatic_title

  # ========================================================================= #
  # === shortcut_keys                         (keybindings tag, shortcut tag)
  #
  # Use this to add a new shortcut to your application.
  #
  # The second argument, `this_method`, specifies which method we will use 
  # and invoke.
  #
  # Usage examples:
  #
  #   r.add_shortcut(:q,  'name_of_method_that_will_be_called')
  #   r.add_shortcut(:q,  'fill_up_buffer')
  #   r.add_shortcut(:e,  'focus_entry', :alt)
  #   r.add_shortcut('e', 'focus_entry', :shift)
  #   r.add_shortcut(1,   'focus_entry(1)', :alt)
  #   r.add_shortcut(0,   'invoke_this_method', :alt)
  #
  # Or with self:
  #
  #   self.add_shortcut(1, '@notebook.set_page(0)', :ctrl)
  #
  # An even simpler variant is the following API:
  #
  #   r.keybinding 'Alt+1', 'do_ask_new_question'
  #   r.keybinding 'Alt+2', 'do_reveal_the_answer'
  #
  # ========================================================================= #
  def shortcut_keys(
      shortcut_key_to_use,
      this_method,
      modifier_key = :control,
      &block
    )
    # ======================================================================= #
    # Remove ' ' characters.
    # ======================================================================= #
    if shortcut_key_to_use.is_a?(String) and
       shortcut_key_to_use.include?(' ')
      # ===================================================================== #
      # Since as of 12.05.2021 we will remove ' '.
      # ===================================================================== #
      shortcut_key_to_use.delete!(' ')
    end
    # ======================================================================= #
    # Next handle cases such as "strg+s".
    # ======================================================================= #
    if (shortcut_key_to_use.is_a?(String) and
        shortcut_key_to_use.include?('+')) or
       (shortcut_key_to_use.is_a?(Symbol) and
        shortcut_key_to_use.to_s.include?('+'))
      splitted = shortcut_key_to_use.to_s.split('+')
      modifier_key = splitted.first.to_sym
      shortcut_key_to_use = splitted.last
    end
    case modifier_key
    # ======================================================================= #
    # === :strg_alt
    # ======================================================================= #
    when :strg_alt, # recall ourself
         :strgalt
      add_shortcut(shortcut_key_to_use, this_method, :strg)
      modifier_key = Gdk::Window::MOD1_MASK # but pass through for ALT
    # ======================================================================= #
    # === :control
    # ======================================================================= #
    when :control,
         :ctrl,
         :strg,
         :str,
         nil,
         :default,
         '',
         'strg',
         'string'
      modifier_key = :control_mask
    # ======================================================================= #
    # === :alt
    # ======================================================================= #
    when :alt, # This is the alt-key.
         'Alt',
         /^alt/
      modifier_key = :mod1_mask
    # ======================================================================= #
    # === :shift
    # ======================================================================= #
    when :shift,
         :pfeil_oben
      modifier_key = Gdk::Window::SHIFT_MASK
    end
    # ======================================================================= #
    # Obtain the proper shortcut key next.
    # ======================================================================= #
    case shortcut_key_to_use
    # ======================================================================= #
    # Match against Alt+1, Alt+2, Alt+3 and so forth.
    #
    # Regex to test this:
    #
    #   https://rubular.com/r/vA6bxNh9C2oB26
    #
    # ======================================================================= #
    when /^Alt\+(\d{1,2})/i,
         /^Alt\+([A-Za-z←→])/i
      modifier_key = :mod1_mask
      shortcut_key_to_use = $1.to_s.dup
    end
    case modifier_key
    # ======================================================================= #
    # === :Alt
    # ======================================================================= #
    when :Alt
      modifier_key = :mod1_mask
    end
    shortcut_key_to_use = Gtk.get_shortcut_key(shortcut_key_to_use) # Gdk::Keyval.from_name()
    # ======================================================================= #
    # === Ensure that @internal_hash[:accel_group] exists
    #
    # Create a new accel-group if no prior accel group exists:
    # ======================================================================= #
    if @internal_hash[:accel_group].nil?
      create_accel_group
    end
    # ======================================================================= #
    # Pass to the @internal_hash[:accel_group] next.
    # ======================================================================= #
    if use_gtk4?
      e 'accel_group.connect() has not yet been added for ruby-gtk4.'
      e 'It may be implemented at a later time.'
    else
      @internal_hash[:accel_group].connect(shortcut_key_to_use, modifier_key, :visible) {
        # then we use arguments
        if this_method.include? '('
          _ = this_method.split '('
          method_name = _[0]
          argument_to_it = _[1][0..-2] # get arguments
          # =================================================================== #
          # We send the method to the child widget next.
          # =================================================================== #
          child_widget?.send(method_name, argument_to_it)
        else
          if child_widget?.respond_to? this_method.to_sym
            child_widget?.send(this_method)
          else
            case this_method.to_s
            # ================================================================= #
            # === exit_program
            # ================================================================= #
            when 'exit_program'
              ::Gtk.main_quit
            end
          end
        end
      }
    end
  end; alias add_this_shortcut shortcut_keys # === add_this_shortcut
       alias add_shortcut      shortcut_keys # === add_shortcut
       alias keybinding        shortcut_keys # === keybinding

  # ========================================================================= #
  # === set_global_font_size
  #
  # The first input-argument to this method should be a number.
  # ========================================================================= #
  def set_global_font_size(
      i                    = 20,
      use_this_font_family = 'Sans'
    )
    Gtk::Settings.default.gtk_font_name = "#{use_this_font_family} #{i}"
  end

  # ========================================================================= #
  # === handle_the_destroy_event
  # ========================================================================= #
  def handle_the_destroy_event
    signal_connect(:destroy) { exit_application }
  end

  # ========================================================================= #
  # === exit_application
  #
  # Use this to quit from an application.
  # ========================================================================= #
  def exit_application
    ::Gtk.main_quit
  end; alias exit_program exit_application # === exit_program
       alias do_quit      exit_application # === do_quit

  # ========================================================================= #
  # === interactively_create_directory
  #
  # This convenience method can be used to create a new directory.
  # ========================================================================= #
  def interactively_create_directory(
      use_this_as_parent_widget
    )
    result = ''
    text   = gtk_text(
      'Input the name of the directory to be created:'
    )
    entry = create_entry
    
    popup_box = create_vbox
    popup_box.set_padding(12)
    popup_box.set_size_request(800, 240)
    popup_box.set_font(:hack_20)
    popup_box.minimal(text, 2)
    popup_box.maximal(entry, 2)

    dialog = ::Gtk::Dialog.new(parent:  use_this_as_parent_widget,
                               title:   'Interactively create a directory',
                               flags:   [:modal, :destroy_with_parent],
                               buttons: [['_Cancel', :cancel],
                                         ['_Create the directory', :ok]])
    content_area = dialog.content_area
    content_area.pack_start(popup_box)
    content_area.show_all

    response = dialog.run

    case response
    when :ok # This is <Gtk::ResponseType ok>
      result << entry.text?
    end

    if File.directory? result
      e 'The directory '+result+' already exists.'
    else
      e 'Creating the directory '+result+' next.'
      FileUtils.mkdir_p(result)
    end

    dialog.destroy
  end

  # ========================================================================= #
  # === top_middle
  #
  # Position the Gtk::Runner widget on the very top, but middle position.
  #
  # This must be calculated properly:
  #
  #   (1) first we must determine the max-width of the monitor, and then
  #       divide it by 2
  #   (2) then we also need to determine the width of the widget
  #       that will be spawned. This will default to a hardcoded
  #       value, but Gtk::Runner can modify this accordingly via
  #       the argument
  #
  # ========================================================================= #
  def top_middle(
      use_this_subwidget_width = :default # This argument can be a number.
    )
    case use_this_subwidget_width
    # ======================================================================= #
    # === :default
    # ======================================================================= #
    when :default
      use_this_subwidget_width = 250
    end
    calculated_value  = Gdk::Screen.default.width / 2
    calculated_value -= use_this_subwidget_width
    move(calculated_value, 0)
  end

  # ========================================================================= #
  # === top_middle_then_run
  # ========================================================================= #
  def top_middle_then_run(
      i = :default
    )
    top_middle(i)
    run
  end

  # ========================================================================= #
  # === top_left_show_all_then_run
  # ========================================================================= #
  def top_left_show_all_then_run
    top_left_and_show_all
    run
  end; alias top_left_show_all_then_run      top_left_show_all_then_run # === top_left_show_all_then_run
       alias top_left_then_show_all_then_run top_left_show_all_then_run # === top_left_then_show_all_then_run

  # ========================================================================= #
  # === do_maximize_width
  # ========================================================================= #
  def do_maximize_width(
      width  = max_width?, # Obtain the max-width here.
      height = height?
    )
    set_size_request(width, height)
  end

  # ========================================================================= #
  # === infer_default_size
  #
  # This variant will use the size from the child-widget. Thus, for this
  # method to properly work, a child-widget has to exist.
  # ========================================================================= #
  def infer_default_size
    if child_widget?
      set_default_size(
        child_widget?.width?,
        child_widget?.height?
      )
    end
  end; alias clever_size    infer_default_size # === clever_size
       alias infer_the_size infer_default_size # === infer_the_size

  # ========================================================================= #
  # === feedback
  #
  # Just feedback some info to the user.
  # ========================================================================= #
  def feedback
    unless Object.const_defined? :Colours
      begin
        require 'colours'
      rescue LoadError; end
    end
    e
    e "  #{::Colours.rev}widget-width is:  #{::Colours.sfancy(@width)}"
    e "  #{::Colours.rev}widget-height is: #{::Colours.sfancy(@height)}"
    e
  end

  # ========================================================================= #
  # === enable_quick_exit_then_run
  # ========================================================================= #
  def enable_quick_exit_then_run
    enable_quick_exit
    run
  end

  # ========================================================================= #
  # === type_hint=
  # ========================================================================= #
  def type_hint=(i = :dock)
    window.type_hint = Gdk::WindowTypeHint[i]
  end

  # ========================================================================= #
  # === child_widget?
  # ========================================================================= #
  def child_widget?
    @internal_hash[:child_widget]
  end; alias child_widget child_widget? # === child_widget
       alias child?       child_widget? # === child?

  # ========================================================================= #
  # === width_height
  # ========================================================================= #
  def width_height(width, height)
    set_size_request(width, height)
  end

  # ========================================================================= #
  # === set_default_size
  # ========================================================================= #
  def set_default_size(
      w = DEFAULT_WIDTH,
      h = DEFAULT_HEIGHT
    )
    set_width(w)
    set_height(h)
  end

  # ========================================================================= #
  # === resize_to_child_widget_dimension
  #
  # This method can be used to resize to the child-widget's dimension.
  # ========================================================================= #
  def resize_to_child_widget_dimension
    _ = child_widget?
    set_size_request(_.width?, _.height?) if _
  end

  # ========================================================================= #
  # === size=
  #
  # Set width and height here. Input argument to this method should be an
  # Array.
  # ========================================================================= #
  def size=(array)
    set_width(array[0])
    set_height(array[1])
  end

  # ========================================================================= #
  # === set_cursor
  #
  # main_window.set_cursor('rand')
  #
  # This method allows you to easily set a new cursor. By default, the
  # boat cursor will be used.
  # ========================================================================= #
  def set_cursor(
      which_cursor_to_use =
        :in_method_default
    )
    case which_cursor_to_use
    # ======================================================================= #
    # === :rand
    # ======================================================================= #
    when :rand
      cursor = AVAILABLE_CURSORS.sample
    # ======================================================================= #
    # === :boat
    # ======================================================================= #
    when :boat,
         :in_method_default
      cursor = Gdk::Cursor::BOAT
    # ======================================================================= #
    # === :x
    # ======================================================================= #
    when :x
      cursor = Gdk::Cursor::X_CURSOR
    # ======================================================================= #
    # === :arrow
    # ======================================================================= #
    when :arrow
      cursor = Gdk::Cursor::ARROW
    # ======================================================================= #
    # === :down
    # ======================================================================= #
    when :down
      cursor = Gdk::Cursor::BASED_ARROW_DOWN
    # ======================================================================= #
    # === :up
    # ======================================================================= #
    when :up
      cursor = Gdk::Cursor::BASED_ARROW_UP
    # ======================================================================= #
    # === :bogosity
    # ======================================================================= #
    when :bogosity
      cursor = Gdk::Cursor::BOGOSITY
    # ======================================================================= #
    # === :left
    # ======================================================================= #
    when :left
      cursor = Gdk::Cursor::BOTTOM_LEFT_CORNER
    # ======================================================================= #
    # === :right
    # ======================================================================= #
    when :right
      cursor = Gdk::Cursor::BOTTOM_RIGHT_CORNER
    # ======================================================================= #
    # === :side
    # ======================================================================= #
    when :side
      cursor = Gdk::Cursor::BOTTOM_SIDE
    # ======================================================================= #
    # === :tee
    # ======================================================================= #
    when :tee
      cursor = Gdk::Cursor::BOTTOM_TEE
    # ======================================================================= #
    # === :spiral
    # ======================================================================= #
    when :spiral
      cursor = Gdk::Cursor::BOX_SPIRAL
    # ======================================================================= #
    # === :center
    # ======================================================================= #
    when :center
      cursor = Gdk::Cursor::CENTER_PTR
    # ======================================================================= #
    # === :circle
    # ======================================================================= #
    when :circle
      cursor = Gdk::Cursor::CIRCLE
    # ======================================================================= #
    # === :clock
    # ======================================================================= #
    when :clock
      cursor = Gdk::Cursor::CLOCK
    # ======================================================================= #
    # === :mug
    # ======================================================================= #
    when :mug
      cursor = Gdk::Cursor::COFFEE_MUG
    # ======================================================================= #
    # === :gobbler
    # ======================================================================= #
    when :gobbler
      cursor = Gdk::Cursor::GOBBLER
    # ======================================================================= #
    # === :gumby
    # ======================================================================= #
    when :gumby
      cursor = Gdk::Cursor::GUMBY
    # ======================================================================= #
    # === :hand1
    # ======================================================================= #
    when :hand1
      cursor = Gdk::Cursor::HAND1
    # ======================================================================= #
    # === :hand2
    # ======================================================================= #
    when :hand2
      cursor = Gdk::Cursor::HAND2
    # ======================================================================= #
    # === :heart
    # ======================================================================= #
    when :heart
      cursor = Gdk::Cursor::HEART
    # ======================================================================= #
    # === :icon
    # ======================================================================= #
    when :icon
      cursor = Gdk::Cursor::ICON
    # ======================================================================= #
    # === :iron
    # ======================================================================= #
    when :iron
      cursor = Gdk::Cursor::IRON_CROSS
    # ======================================================================= #
    # === :default
    # ======================================================================= #
    when :default_cursor,
         :default
      cursor = Gdk::Cursor::LEFT_PTR
    # ======================================================================= #
    # === :left_side
    # ======================================================================= #
    when :left_side
      cursor = Gdk::Cursor::LEFT_SIDE
    # ======================================================================= #
    # === :left_tee
    # ======================================================================= #
    when :left_tee
      cursor = Gdk::Cursor::LEFT_TEE
    # ======================================================================= #
    # === :left_button
    # ======================================================================= #
    when :left_button,
         :leftbutton
      cursor = Gdk::Cursor::LEFTBUTTON
    # ======================================================================= #
    # === :left_angle
    # ======================================================================= #
    when :left_angle
      cursor = Gdk::Cursor::LL_ANGLE
    else
      cursor = Gdk::Cursor::WATCH
    end
    unless which_cursor_to_use == 'rand'
      window.set_cursor(Gdk::Cursor.new(cursor))
    end
  end

  # ========================================================================= #
  # === top_left                                               (top_left tag)
  #
  # Position the Gtk::Runner widget on the very top left position.
  # ========================================================================= #
  def top_left
    if respond_to?(:move)
      move(0, 0)
    else
      pp __LINE__
      e 'The widget does not respond to :move.'
    end
  end; alias move_top_left top_left # === move_top_left

  # ========================================================================= #
  # === move_to
  #
  # This method is basically a wrapper over move(). It allows one to pass
  # a Hash, though, to specifically handle input like this:
  #
  #   { x: 150, y: 0}
  #
  # I wanted such an API to instantly see which value is the x-coordinate
  # and which value is the y-coordinate.
  # ========================================================================= #
  def move_to(
      i            = { x: 150, y: 0 },
      y_coordinate = nil
    )
    # ======================================================================= #
    # === Next handle Hashes as input
    # ======================================================================= #
    if i.is_a? Hash
      # ===================================================================== #
      # === :y
      #
      # :y should be handled before :x.
      # ===================================================================== #
      if i.has_key? :y
        y_coordinate = i.delete(:y)
      end
      # ===================================================================== #
      # === :x
      # ===================================================================== #
      if i.has_key? :x
        i = i.delete(:x)
      end
    end
    move(i, y_coordinate)
  end

  # ========================================================================= #
  # === shortcuts
  #
  # This method may be used like so:
  #
  #   r.shortcuts {
  #     [1, 'click_on_button_number(0)', :alt], # for alt+1 key combination
  #   }
  #
  # ========================================================================= #
  def shortcuts(&block)
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      if yielded.is_a? Array
        yielded.each {|inner_array|
          add_shortcut(
            inner_array[0],
            inner_array[1],
            inner_array[2]
          )
        }
      end
    end
  end

  # ========================================================================= #
  # === shortcut
  #
  # This variant requires a '+' character to be used.
  #
  # Usage example:
  #
  #   r.shortcut('alt+1') { 'focus_on(@entry_pid_number)' }
  #
  # ========================================================================= #
  def shortcut(
      key_combination_to_use, &block
    )
    splitted = key_combination_to_use.split('+')
    number = splitted.last.to_i
    modifier_key_to_use = splitted.first.to_sym
    add_shortcut(
      number,
      yield,
      modifier_key_to_use
    )
  end

  # ========================================================================= #
  # === status_icon
  #
  # This method add a StatusIcon to a gtk-widget.
  # ========================================================================= #
  def status_icon(
      use_this_as_the_icon = favicon?
    )
    _ = ::Gtk::StatusIcon.new(filename: use_this_as_the_icon)
    return _
  end

  # ========================================================================= #
  # === set_background_colour
  #
  # This method (or an alias to this method) can be used to set the
  # background colour.
  # ========================================================================= #
  def set_background_colour(i)
    if i
      @internal_hash[:background_colour] = i.to_s
      # ===================================================================== #
      # Note that if we input an incorrect colour then this will fail. This
      # explains the following code in the rescue-clause.
      # ===================================================================== #
      begin
        parsed_colour = ::Gdk::Color.parse(@internal_hash[:background_colour])
      rescue ArgumentError
        @internal_hash[:background_colour] = 'white'
        parsed_colour = ::Gdk::Color.parse('white')
      end
      # else assume gtk3 is in use
      if parsed_colour.is_a? Gdk::Color
        parsed_colour = Gdk::RGBA.new(
          parsed_colour.red,
          parsed_colour.green,
          parsed_colour.blue
        )
      end
      override_background_color(:normal, parsed_colour)
    end
  end; alias background_colour set_background_colour # === background_colour

  # ========================================================================= #
  # === minimum_size
  #
  # This can be used to specify a minimum size. It is advised to pass
  # a Hash.
  # ========================================================================= #
  def minimum_size(
      hash = {
        width:  '25%',
        height: '15%'
      }
    )
    set_size_request(hash[:width], hash[:height])
  end

  # ========================================================================= #
  # === accel_group?
  #
  # So that we can manipulate the AccelGroup.
  # ========================================================================= #
  def accel_group?
    @internal_hash[:accel_group]
  end; alias accel_group accel_group? # === accel_group

  # ========================================================================= #
  # === try_to_load_the_configuration_file
  # ========================================================================= #
  def try_to_load_the_configuration_file(
      path_to_the_configuration_file = 
        'configuration_options_for_class_gtk_runner.yml'
    )
    _ = ::Gtk.project_base_directory?+
        'yaml/'+
        path_to_the_configuration_file
    if File.exist? _
      @internal_hash[:yaml_dataset] = YAML.load_file(_)
      # ===================================================================== #
      # Next check any configuration value - for now we only handle one
      # setting.
      # ===================================================================== #
      if @internal_hash[:yaml_dataset].has_key? 'shall_we_enable_simple_exit_for_all_instances_of_gtk_runner'
        if @internal_hash[:yaml_dataset]['shall_we_enable_simple_exit_for_all_instances_of_gtk_runner'] == true
          enable_easy_exit
        end
      end
    else
      e "No file exists at `#{sfile(_)}`."
    end
  end

  # ========================================================================= #
  # === alt_shortcut
  #
  # This method enables a key-shortcut via the alt-key.
  #
  # The second argument is the method that you wish to call / invoke.
  #
  # Specific usage examples:
  #
  #   r.alt_key(1, 'focus_entry(1)')
  #   r.alt_shortcut(1) { 'focus_entry(1)' }
  #
  # ========================================================================= #
  def alt_shortcut(
      key_combination_to_use,
      method_to_call = nil
    )
    use_this_shortcut = "alt+#{key_combination_to_use}"
    if block_given?
      yielded = yield
      method_to_call = yielded if method_to_call.nil?
      shortcut(use_this_shortcut) { method_to_call }
    else
      shortcut(use_this_shortcut) { method_to_call }
    end
  end; alias alt_key alt_shortcut # === alt_key

  # ========================================================================= #
  # === automatic_size
  #
  # This method currently combines two methods. I am unsure whether this
  # will be retained in the long run, but for now (2022) it will be
  # retained as-is.
  # ========================================================================= #
  def automatic_size(
      width  = width?,
      height = height?
    )
    set_default_size(width.to_i, height.to_i)
    # ======================================================================= #
    # The next line actually causes the resize-action.
    # ======================================================================= #
    set_size_request(width.to_i, height.to_i)
  end; alias use_default_size_from_the_config_file    automatic_size # ===  use_default_size_from_the_config_file
       alias determine_the_size_from_the_child_widget automatic_size # === determine_the_size_from_the_child_widget
       alias size_from_child                          automatic_size # === size_from_child

  # ========================================================================= #
  # === automatic_size_then_automatic_title
  #
  # This method will simply combine automatic_title() with
  # automatic_size().
  # ========================================================================= #
  def automatic_size_then_automatic_title(
      use_this_subwidget = 
        child_widget?
    )
    automatic_size
    automatic_title(use_this_subwidget)
  end; alias automatic_size_and_automatic_title  automatic_size_then_automatic_title # === automatic_size_and_automatic_title
       alias automatic_title_and_size            automatic_size_then_automatic_title # === automatic_size_and_automatic_title
       alias automatic_title_and_automatic_size  automatic_size_then_automatic_title # === automatic_title_and_automatic_size
       alias automatic_size_and_title            automatic_size_then_automatic_title # === automatic_size_and_title
       alias automatic                           automatic_size_then_automatic_title # === automatic

  # ========================================================================= #
  # === disallow_modification_of_the_title
  # ========================================================================= #
  def disallow_modification_of_the_title
    @internal_hash[:may_we_modify_the_title] = false
  end

  # ========================================================================= #
  # === may_we_modify_the_title?
  # ========================================================================= #
  def may_we_modify_the_title?
    @internal_hash[:may_we_modify_the_title]
  end

  # ========================================================================= #
  # === set_favicon
  #
  # Ensures that the icon exists before setting it.
  #
  # Some symbol-shortcuts are allowed here, which are hardcoded towards my
  # own system.
  # ========================================================================= #
  def set_favicon(i)
    if i.nil? and is_on_roebe?
      i = :default
    end
    if i.is_a?(String) and i.start_with?(':')
      i = i.delete(':').to_sym
    end
    case i
    # ======================================================================= #
    # === :RMC
    # ======================================================================= #
    when :RMC,
         :rmc
      i = ::Gtk.project_base_directory?+'images/misc/RMC.jpg'
    # ======================================================================= #
    # === :lotus
    # ======================================================================= #
    when :lotus
      i = '/home/x/data/images/RPG/STD/LOTUS_MASKE.png'
    # ======================================================================= #
    # === :tabble_image
    #
    # This entry point is currently aliased to ":default" - thus, it is
    # the default value.
    # ======================================================================= #
    when :tabble_image,
         :tabble,
         :default
      i = ::Gtk.project_base_directory?+'images/misc/tabble.png'
    # ======================================================================= #
    # === :pc_favicon
    # ======================================================================= #
    when :pc_favicon
      i = '/home/x/data/images/PC/PC_FAVICON.png'
    # ======================================================================= #
    # === :task_image
    # ======================================================================= #
    when :tasks_image,
         :tasks
      i = '/home/x/data/images/PC/TASKS.jpg'
    # ======================================================================= #
    # === :audio_favicon
    # ======================================================================= #
    when :audio_favicon
      i = '/home/x/data/images/AUDIO/AUDIO_FAVICON.png'
    end
    if i
      begin
        if File.exist? i
          @internal_hash[:favicon] = i
        else
          e "(in gtk_paradise/run.rb) No file exists at this location: `#{i}`"
        end
      rescue Exception => error
        pp error
        pp i
      end
    else
      if is_on_roebe?
        e 'Please provide an icon_location to the method set_favicon()'
      end
    end
    @internal_hash[:favicon] = i
  end; alias set_icon_location set_favicon # === set_icon_location
       alias icon_location=    set_favicon # === icon_location=
       alias icon=             set_favicon # === icon?
       alias set_this_favicon  set_favicon # === set_this_favicon
       alias set_logo          set_favicon # === set_logo
       alias favicon=          set_favicon # === favicon=

  # ========================================================================= #
  # === show_then_run
  # ========================================================================= #
  def show_then_run
    show_all
    run
  end; alias show_all_then_run show_then_run # === show_all_then_run

  # ========================================================================= #
  # === enable_quick_exit
  #
  # This method enables quick "alt+q" key-combination to quit the
  # application. Since as of September 2020, we also use strg+q 
  # (which is ctrl+q, actually).
  # ========================================================================= #
  def enable_quick_exit
    add_shortcut(:q, 'exit_program', :alt)  # press: alt+q
    add_shortcut(:q, 'exit_program', :strg) # press: strg+q
  end; alias easy_exit          enable_quick_exit # === easy_exit
       alias easy_quit          enable_quick_exit # === easy_quit
       alias quick_exit         enable_quick_exit # === quick_exit
       alias enable_simple_exit enable_quick_exit # === enable_simple_exit
       alias simple_exit        enable_quick_exit # === simple_exit
       alias enable_easy_exit   enable_quick_exit # === enable_easy_exit

  # ========================================================================= #
  # === <<                                                          (add tag)
  #
  # This method will add a widget to the Gtk::App (which is a Gtk::Window
  # actually). It delegates towards .add() for this.
  #
  # Additionally the method will do a few more calls, such as by
  # querying the title in use (if there is one in use that is.
  #
  # Last but not least, it will keep a reference to the child-widget
  # too, via set_internal_child_widget().
  # ========================================================================= #
  def <<(child_widget)
    # unless @internal_hash.is_a? Hash
    #   initialize_the_internal_hash
    # end
    builder = ::Gtk::Builder.new
    set_internal_child_widget(child_widget)
    unless use_gtk4?
    # This part may have to be re-visited at a later time.
    #   builder.set_child(child_widget)
    # else
      add_child(builder, child_widget) # add_child(builder, child, type)
    end
    # ======================================================================= #
    # === Apply the ad-hoc CSS rules next
    #
    # Add any CSS content found next automatically..
    # ======================================================================= #
    if child_widget.respond_to?(:css_string?) and
       child_widget.css_string? and
      !child_widget.css_string?.empty?
      child_widget.apply_the_css_string
    end
    # ======================================================================= #
    # Next we will try to find out whether that child-widget contains
    # some data we can use.
    # ======================================================================= #
    # ======================================================================= #
    # === :my_title?
    # ======================================================================= #
    if child_widget and
       child_widget.respond_to?(:my_title?) and
       may_we_modify_the_title?
      set_my_title(child_widget.my_title?)
    end
    # ======================================================================= #
    # === :width?
    # ======================================================================= #
    if child_widget.respond_to? :width?
      set_width(child_widget.width?) if child_widget.width?
    end
    # ======================================================================= #
    # === :height?
    # ======================================================================= #
    if child_widget.respond_to? :height?
      set_height(child_widget.height?)
    end
    # ======================================================================= #
    # === :favicon
    # ======================================================================= #
    if child_widget.respond_to? :favicon?
      set_favicon(child_widget.favicon?)
    end
    # ======================================================================= #
    # === :padding
    # ======================================================================= #
    if child_widget.respond_to? :padding?
      set_padding(child_widget.padding?)
    end
    # ======================================================================= #
    # === :border_size
    # ======================================================================= #
    if child_widget.respond_to?(:border_size?) and
       child_widget.respond_to?(:set_border_width?)
      child_widget.set_border_width(child_widget.border_size?)
    end
    # ======================================================================= #
    # === enable_context_menu
    #
    # Sync enable_context_menu onto Gtk.app().
    # ======================================================================= #
    if child_widget.respond_to? :enable_context_menu
      enable_context_menu
    end
    # ======================================================================= #
    # === :use_config_file?
    #
    # Next we check for a file called .config, which would be stored within
    # the instance variable called @configuration - the latter is
    # accessible via the method call .configuration?().
    #
    # If this method exists, it will be used to override everything else,
    # but only if the method called .configuration?() has been defined.
    # ======================================================================= #
    if child_widget.respond_to? :configuration?
      _ = child_widget.configuration?
      # ===================================================================== #
      # This may look like so:
      #
      #   /home/x/programming/ruby/src/gtk_paradise/lib/gtk_paradise/classes/gidnight_commander
      #
      # ===================================================================== #
      if _ and _.first
        this_file = (_[1]+'/'+File.basename(_[1])+'.config').squeeze('/')
      end
      if this_file and _.first and File.exist?(this_file)
        dataset = File.readlines(this_file)
        # =================================================================== #
        # Now that we loaded the dataset, proceed to evaluate every non-empty
        # line.
        #
        # This may look like so:
        #
        #     ["width:                                      1200\n",
        #      "height:                                      800\n",
        #      "padding:                                      15\n",
        #      "title:                        Gidnight-Commander\n",
        #      "favicon:          /home/x/DATA/images/PC/RMC.jpg\n",
        #      "font_in_use:                             Sans 20"]
        #
        # Note that favicon may also be :tabble, that is have a leading
        # : character, to denote a Symbol.
        # =================================================================== #
        stripped = dataset.map {|line|
          line.strip.squeeze(' ')
        }
        stripped.each {|line|
          if line.include? ':' # This ':' is mandatory.
            # =============================================================== #
            # We must split at the first ':', as the line may include 
            # more than one ':'.
            # =============================================================== #
            left_index = line.index(':')
            first = line[0 .. left_index].strip.delete(':')
            last  = line[(left_index+1) .. -1].strip
            case first
            # =============================================================== #
            # === title
            # =============================================================== #
            when 'title'
              set_my_title(last)
            # =============================================================== #
            # === width
            # =============================================================== #
            when 'width'
              set_width(last)
            # =============================================================== #
            # === height
            # =============================================================== #
            when 'height'
              set_height(last)
            # =============================================================== #
            # === padding
            # =============================================================== #
            when 'padding'
              set_padding(last)
            # =============================================================== #
            # === favicon
            # =============================================================== #
            when 'favicon'
              set_favicon(last)
            # =============================================================== #
            # === font_in_use
            # =============================================================== #
            when 'font_in_use'
              set_font_in_use(last)
            end
          end
        }
      end
      return
    end
  end #; alias add << # === add

  # ========================================================================= #
  # === set_the_title_if_it_was_set
  #
  # This method should be used whenever we wish to apply the internally
  # set title.
  #
  # It has a few if/else clauses largely because there was a bug in effect
  # where .set_title() was not called properly. Now there is no way around
  # notifying the user when something goes awry.
  # ========================================================================= #
  def set_the_title_if_it_was_set(
      i                       = internal_title?,
      may_we_modify_the_title = @internal_hash[:may_we_modify_the_title]
    )
    if i and may_we_modify_the_title
      if respond_to?(:set_title)
        set_title(i)
      else
        e "Input was given to #{__method__.to_s} but the class does not"
        e 'respond to .set_title().'
      end
    # else
      # e 'No title was given. This is unusual - please investigate the cause.'
      # pp caller()
    end
  end; alias apply_the_internal_title set_the_title_if_it_was_set # === apply_the_internal_title

  # ========================================================================= #
  # === populate_the_internal_hash                                (reset tag)
  # ========================================================================= #
  def populate_the_internal_hash
    # ======================================================================= #
    # === :title
    #
    # The title must be nil initially, so that we can distinguish it from
    # the case where the user specifically set it to another value. So
    # if the value is NOT nil then we know the user (or something else)
    # modified it as-is.
    # ======================================================================= #
    @internal_hash[:title] = nil
    # ======================================================================= #
    # === :may_we_modify_the_title
    #
    # This can be disabled via a method call. If disabled then we will not
    # allow the internal title-setter to rename the widget anymore.
    # ======================================================================= #
    @internal_hash[:may_we_modify_the_title] = true
    # ======================================================================= #
    # === :width
    # ======================================================================= #
    set_width(DEFAULT_WIDTH)
    # ======================================================================= #
    # === :height
    # ======================================================================= #
    set_height(DEFAULT_HEIGHT)
    # ======================================================================= #
    # === :favicon
    # ======================================================================= #
    @internal_hash[:favicon] = nil
    # ======================================================================= #
    # === :filename
    #
    # :filename is initially a "dummy" variable. It was added to allow
    # the file dialog chooser to assign to a specific filename, so it
    # is only useful for some widgets, but not all widgets.
    # ======================================================================= #
    @internal_hash[:filename] = nil
    # ======================================================================= #
    # === :child_widget
    # ======================================================================= #
    @internal_hash[:child_widget] = nil
    # ======================================================================= #
    # === :border_width
    # ======================================================================= #
    @internal_hash[:border_width] = 0
    # ======================================================================= #
    # === :yaml_dataset
    #
    # This variable will eventually contain the dataset from the yaml
    # file that stores the default configuration to use for Gtk::Runner.
    # ======================================================================= #
    @internal_hash[:yaml_dataset] = nil
    # ======================================================================= #
    # === :use_select_all
    #
    # This variable will keep track of whether to enable or disable
    # strg-a.
    # ======================================================================= #
    @internal_hash[:use_select_all] = true
  end; alias initialize_the_main_hash  populate_the_internal_hash # === initialize_the_main_hash
       alias reset_internal_app_module populate_the_internal_hash # === reset_internal_app_module

  # ========================================================================= #
  # === reset_towards_the_default_state                           (reset tag)
  # ========================================================================= #
  def reset_towards_the_default_state
    reset_the_internal_variables_as_well_as_the_internal_hash
    # ======================================================================= #
    # === :border_width
    #
    # Set the initial border width to 0. This is not strictly necessary,
    # but I want to put an additional emphasis here.
    # ======================================================================= #
    set_my_border_width(0)
    handle_the_destroy_event
    populate_the_internal_hash
    try_to_load_the_configuration_file
  end; alias internal_reset reset_towards_the_default_state # === internal_reset

  # ========================================================================= #
  # === run                                     (run tag, start tag, def run)
  #
  # Start the application via this method.
  # ========================================================================= #
  def run(
      optional_show_all = true
    )
    super() if respond_to?(:super)
    show_all if optional_show_all
    set_the_title_if_it_was_set
    consider_determining_width_and_height
    # ======================================================================= #
    # The next code clause is probably no longer necessary, as .add() will
    # handle these cases.
    # ======================================================================= #
    # if width? and height?
    #   set_default_size(width?, height?)
    # end
    if use_gtk3?
      ::Gtk.main # Finally, run the application here.
    end
  end; alias start run # === start

  # ========================================================================= #
  # === consider_determining_width_and_height
  # ========================================================================= #
  def consider_determining_width_and_height
    height = height?
    width  = width?
    if width and height
      set_size_request(width, height)
    end
  end

end; end

if __FILE__ == $PROGRAM_NAME
  alias e puts
  require 'gtk3'
  # require 'gtk_paradise/gtk_classes/widget.rb'
  # require 'gtk_paradise/gtk_classes/window.rb'
  require 'gtk_paradise/gtk_classes/window.rb'
  e
  e 'Testing Gtk::AppModule next:'
  e
  window = Gtk::Window.new
  window.extend(Gtk::RunModule)
  window.internal_reset
  e window.internal_title?
  window.set_internal_title('YO THERE - THIS IS A GREAT TITLE!')
  e window.internal_title?
  window.apply_the_internal_title
  window.use_this_font = :hack_30
  window.add(Gtk::Label.new('This is a test.'))
  window.show_all
  window.top_left
  window.width_height 550,300
  window.position = :mouse
  Gtk.main
end