#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# require 'gtk_paradise/toplevel_methods/determine_which_gtk_version_is_in_use.rb'
# Gtk.set_we_use_this_gtk_version
# Gtk.we_use_this_gtk_version = :gtk4
# Gtk.we_use_this_gtk_version = :gtk3
# Gtk.we_use_this_gtk_version = :gtk2
# =========================================================================== #
module Gtk

  # ========================================================================= #
  # === @we_use_this_gtk_version
  #
  # This must be initialized to nil, so that we allow the user to change
  # this value at a later time, as a distinguishing 'feature'.
  #
  # This variable must either be a Symbol or nil.
  # ========================================================================= #
  @we_use_this_gtk_version = nil

  # ========================================================================= #
  # === Gtk.we_use_this_gtk_version?
  #
  # Query-method about the toplevel variable that keeps track as to which
  # version is in use.
  # ========================================================================= #
  def self.we_use_this_gtk_version?
    @we_use_this_gtk_version
  end; self.instance_eval { alias we_use_which_gtk_version? we_use_this_gtk_version? } # === Gtk.we_use_which_gtk_version?
       self.instance_eval { alias version_in_use?           we_use_this_gtk_version? } # === Gtk.version_in_use?

  # ========================================================================= #
  # === Gtk.we_use_this_gtk_version=
  #
  # The new default since as of April 2019 is to require gtk3.
  #
  # Usage examples:
  #
  #   Gtk.we_use_this_gtk_version = :gtk2
  #   Gtk.we_use_this_gtk_version = :gtk3
  #   Gtk.we_use_this_gtk_version = :gtk4
  #
  # ========================================================================= #
  def self.we_use_this_gtk_version=(
      i = :default
    )
    case i
    # ======================================================================= #
    # === :default
    # ======================================================================= #
    when :default
      i = DEFAULT_GTK_VERSION_TO_USE
    end
    # ======================================================================= #
    # puts "Debug: We now set the gtk version to: #{i}" # Some debug-line.
    # ======================================================================= #
    @we_use_this_gtk_version = i
  end; self.instance_eval { alias set_we_use_this_gtk_version we_use_this_gtk_version= } # === Gtk.set_we_use_this_gtk_version

  # ========================================================================= #
  # === Gtk.do_we_use_gtk4?
  # ========================================================================= #
  def self.do_we_use_gtk4?
    @we_use_this_gtk_version == :gtk4
  end; self.instance_eval { alias are_we_using_gtk4? do_we_use_gtk4? } # === Gtk.are_we_using_gtk4
       self.instance_eval { alias use_gtk4?          do_we_use_gtk4? } # === Gtk.use_gtk4?

  # ========================================================================= #
  # === Gtk.use_gtk3?
  #
  # Unfortunately MAJOR_VERSION does not exist for ruby-gtk3.
  # ========================================================================= #
  def self.use_gtk3?
    @we_use_this_gtk_version == :gtk3
  end; self.instance_eval { alias are_we_using_gtk3? use_gtk3? } # === Gtk.are_we_using_gtk3?
       self.instance_eval { alias is_gtk3_in_use?    use_gtk3? } # === Gtk.is_gtk3_in_use?
       self.instance_eval { alias is_gtk3?           use_gtk3? } # === Gtk.is_gtk3?
       self.instance_eval { alias use_gtk3?          use_gtk3? } # === Gtk.use_gtk3?
       self.instance_eval { alias do_we_use_gtk3?    use_gtk3? } # === Gtk.do_we_use_gtk3?

  # ========================================================================= #
  # === Gtk.do_we_use_gtk2?
  # ========================================================================= #
  def self.do_we_use_gtk2?
    @we_use_this_gtk_version == :gtk2
  end; self.instance_eval { alias is_gtk2_in_use?    do_we_use_gtk2? } # === Gtk.is_gtk2_in_use?
       self.instance_eval { alias are_we_using_gtk2? do_we_use_gtk2? } # === Gtk.are_we_using_gtk2?

  # ========================================================================= #
  # === Gtk.true_query_whether_gtk2_is_in_use?
  #
  # This method can be used to determine whether gtk2 is in use.
  #
  # An alternative would be to use Gtk.check_version() such as via:
  #
  #   Gtk.check_version(2, 6, 0)
  #
  # ========================================================================= #
  def self.true_query_whether_gtk2_is_in_use?
    Object.const_defined?(:Gtk) and
    Gtk.const_defined?(:MAJOR_VERSION) and
    (::Gtk::MAJOR_VERSION.to_i == 2) 
  end; self.instance_eval { alias is_gtk2?  true_query_whether_gtk2_is_in_use? } # === Gtk.is_gtk2?
       self.instance_eval { alias use_gtk2? true_query_whether_gtk2_is_in_use? } # === Gtk.use_gtk2?

end

if __FILE__ == $PROGRAM_NAME
  alias e puts
  require 'gtk2' # or require 'gtk3'
  e 'is gtk2 in use: '+Gtk.is_gtk2_in_use?.to_s
  e 'is gtk3 in use: '+Gtk.is_gtk3_in_use?.to_s
end