#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::ClipboardDemo
#
# The code in this file will demonstrate how the Gtk::Clipboard works; or
# how the user can interact with it..
#
# The clipboard functionality allows you to reuse data between different
# components.
#
# Besides that, there is a Gtk::Editable module that provides some
# clipboard functionality as well:
#
#   http://ruby-gnome2.sourceforge.jp/hiki.cgi?Gtk%3A%3AEditable
# 
# And the classes that have that module as a mixin should provide the
# basic functionality, like the Gtk::Entry:
#
#   http://ruby-gnome2.sourceforge.jp/hiki.cgi?Gtk%3A%3AEntry
#
# Usage example:
#
#   Gtk::ClipboardDemo.new
#
# =========================================================================== #
# require 'gtk_paradise/widgets/gtk2/editable_combo_field.rb'
# =========================================================================== #
require 'gtk_paradise/require_gtk2.rb'

module Gtk

class ClipboardDemo < Gtk::VBox # === Gtk::ClipboardDemo

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  require 'colours'
  include ::Colours

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = 920

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      commandline_arguments = nil,
      run_already           = true
    )
    super()
    reset
    set_commandline_arguments(
      commandline_arguments
    )
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, NAMESPACE]
  end

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    increase_font_size
    set_size_request(width?, 780)
    @clipboard = create_clipboard
    # ======================================================================= #
    # Our buttons will be defined next:
    # ======================================================================= #
    @button_retrieve = button('Retrieve text from the clipboard')
    @button_retrieve.fancy_tooltip = 'This will simply retrieve (obtain) '\
      'the current content of the clipboard.'
    pack_minimal(@button_retrieve, 0)
    @button_set_and_reassign_text_to_the_clipboard = button('Send '\
      'and Reassign text to the clipboard')
    pack_minimal(@button_set_and_reassign_text_to_the_clipboard, 0)
    # ======================================================================= #
    # === Reset the clipboard
    # ======================================================================= #
    @button_reset_the_clipboard = button('Reset the clipboard')
    @button_reset_the_clipboard.fancy_tooltips = 'This will '\
      '<b>empty</b> the clipboard.'
    pack_minimal(@button_reset_the_clipboard, 0)
    # ======================================================================= #
    # === Output the current content of the clipboard:
    # ======================================================================= #
    @button_output_the_clipboard = button('Output the current clipboard')
    @button_output_the_clipboard.fancy_tooltip = 
      'Pressing this button will simply <b>output the current '\
      'content of the clipboard</b>.'
    pack_minimal(@button_output_the_clipboard, 0)
    # ======================================================================= #
    # Make the button respond to click events:
    # ======================================================================= #
    @button_output_the_clipboard.on_clicked {
      @clipboard.request_text { |cb, text| e text }
    }
    vbox_where_the_user_can_designate_the_new_text_for_the_clipboard = gtk_vbox
    vbox_where_the_user_can_designate_the_new_text_for_the_clipboard << gtk_label(
      'Designate the new text for the clipboard below this text'
    )
    @entry_new_clipboard_content = gtk_entry
    vbox_where_the_user_can_designate_the_new_text_for_the_clipboard << @entry_new_clipboard_content
    # ======================================================================= #
    # set new clipboard
    # ======================================================================= #
    @button_set_and_reassign_text_to_the_clipboard.on_clicked {
      use_this_as_new_text = 'We have now assigned the text of '\
                             'the clipboard. Abc def ghi.'
      unless @entry_new_clipboard_content.text.to_s.empty?
        use_this_as_new_text = @entry_new_clipboard_content.text.to_s
      end
      ::Gtk.set_clipboard_text(use_this_as_new_text)
    }
    # ======================================================================= #
    # Reset the clipboard again:
    # ======================================================================= #
    @button_reset_the_clipboard.on_clicked {
      ::Gtk.reset_clipboard
    }
    # ======================================================================= #
    # Add the "user can designate" widget next.
    # ======================================================================= #
    minimal(vbox_where_the_user_can_designate_the_new_text_for_the_clipboard, 0)
    # ======================================================================= #
    # Next, use a Gtk::Label to display the clipboard-content.
    # ======================================================================= #
    @label_showing_the_current_clipboard = gtk_label(
      '(The new buffer content will appear here.)'
    )
    @label_showing_the_current_clipboard.make_bold
    maximal(
      gtk_hbox << @label_showing_the_current_clipboard,
      0
    )
    @button_retrieve.on_clicked {
      # ===================================================================== #
      # cb is of class Gtk::Clipboard
      # ===================================================================== #
      clipboard_text = clipboard_text?
      @label_showing_the_current_clipboard.set_text(clipboard_text)
      @label_showing_the_current_clipboard.make_bold
      e clipboard_text
    }
    array_all_buttons = [
      @button_retrieve,
      @button_output_the_clipboard,
      @button_set_and_reassign_text_to_the_clipboard,
      @button_reset_the_clipboard
    ]
    array_all_buttons.each {|this_button|
      this_button.lightblue
    }
    show_all
  end

  # ========================================================================= #
  # === width?
  # ========================================================================= #
  def width?
    WIDTH
  end

  # ========================================================================= #
  # === Gtk::ClipboardDemo[]
  # ========================================================================= #
  def self.[](i = '')
    new(i)
  end

  # ========================================================================= #
  # === Gtk::ClipboarDemo.return_widget
  # ========================================================================= #
  def self.return_widget
    Gtk::ClipboardDemo
  end

end; end

if __FILE__ == $PROGRAM_NAME
  require 'gtk_paradise/run.rb'
  clipboard_demo = Gtk::ClipboardDemo.new
  r = Gtk.runner(title: 'Clipboard Demo')
  r << clipboard_demo
  r.width = clipboard_demo.width?
  r.enable_quick_exit
  r.top_left_then_run
end # clipboarddemo