#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::EditableComboField::ListStore
#
# A demo of the CellRendererCombo in a TreeView
#
# A table of 1000 rows and 9 columns is displayed.
#
# Each column supports 1000 choices.
#
# All values are displayed as strings to test performance.
#
# The first column is editable, the remaining ones are not.
# The code should not be used as an example of how to properly
# build CellRendererCombo boxes - it simply illustrates the GUI.
#
# Usage example:
#
#   Gtk::EditableComboField::ListStore.new
#
# =========================================================================== #
# require 'gtk_paradise/classes/editable_combo_field.rb'
# =========================================================================== #
require 'gtk_paradise/require_gtk2'

module Gtk

module EditableComboField

class ListStore < Gtk::ScrolledWindow # === Gtk::EditableComboField::ListStore

  include ::Colours::E

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === NAMESPACE
  # ========================================================================= #
  NAMESPACE = inspect

  # ========================================================================= #
  # === MAX_VALUE
  # ========================================================================= #
  MAX_VALUE = 100

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      commandline_arguments = nil,
      run_already           = true
    )
    super(nil, nil)
    reset
    self.shadow_type = Gtk::SHADOW_ETCHED_IN
    set_policy(Gtk::POLICY_NEVER, Gtk::POLICY_AUTOMATIC)
    set_commandline_arguments(
      commandline_arguments
    )
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, NAMESPACE]
    increase_font_size
  end

  # =========================================================================== #
  # === run
  # =========================================================================== #
  def run
    set_size_request(width?, 800)
    array1 = []
    (1 .. MAX_VALUE).each { |i|
      ar2 = []
      (1..9).each{ |j| ar2 << (i + j).to_s }
      array1 << ar2
    }
    # bug = Struct.new('Bug', :a, :b, :c, :d, :e, :f, :g, :h, :i)
    @list_store = Gtk::ListStore.new(
      String, String, String,
      String, String, String,
      String, String, String
    )
    array1.each { |bug|
      iter = @list_store.append
      (0 .. 8).each { |i|
        iter[i] = bug[i]
      }
    }
    array_picks = []
    (1 .. 50).each {|entry| array_picks << entry.to_s }
    @tree_view = Gtk::TreeView.new(@list_store) # Create tree view.
    # ==================================================================== #
    # Add editable column to the tree view
    # ==================================================================== #
    @cell_renderer_combo = gtk_cell_renderer_combo
    @cell_renderer_combo.signal_connect(:edited) { |*args| # Support data entry.
      iter = @tree_view.model.get_iter(args[1])
      iter[0] = args[2]
    }
    @list_store = Gtk::ListStore.new(String)
    array_picks.each { |entry|
      iter = @list_store.append
      iter[0] = entry
    }
    @cell_renderer_combo.model       = @list_store            # Set values to select from
    @cell_renderer_combo.editable    = true            # User can change values
    @cell_renderer_combo.has_entry   = true            # User can type in text as well 
    # as select from list
    @cell_renderer_combo.text_column = 0
    @tree_view_column = Gtk::TreeViewColumn.new(
      'Editable Combo', @cell_renderer_combo, 'text' => 0
    )
    @tree_view_column.set_sort_column_id(0)
    @tree_view_column.resizable = true
    @tree_view.append_column(@tree_view_column)
    # ==================================================================== #
    # Add 8 non-editable columns to the tree view
    # ==================================================================== #
    (1..8).each{ |cnum|
      @cell_renderer_combo = gtk_cell_renderer_combo
      @cell_renderer_combo.signal_connect(:edited) { |*args| ## Data entry
        iter = @tree_view.model.get_iter(args[1])
        iter[cnum] = args[2]
      }
      @list_store = Gtk::ListStore.new(String)
      array_picks.each { |i| iter = @list_store.append; iter[0] = i }
      @cell_renderer_combo.model       = @list_store # Set values to select from
      @cell_renderer_combo.editable    = true  # User can edit
      @cell_renderer_combo.has_entry   = false # User can't type in text
      @cell_renderer_combo.text_column = 0
      @tree_view_column = Gtk::TreeViewColumn.new(
        'Combo '+cnum.to_s,
        @cell_renderer_combo, 
        'text'=> cnum.to_i
      )
      @tree_view_column.set_sort_column_id(cnum)
      @tree_view_column.resizable = true
      @tree_view.append_column(@tree_view_column)
    }
    add(@tree_view)
  end

  # =========================================================================== #
  # === width?
  # =========================================================================== #
  def width?
    1000
  end

  # ========================================================================= #
  # === Gtk::EditableComboField::ListStore[]
  # ========================================================================= #
  def self.[](i = '')
    new(i)
  end

end; end; end

if __FILE__ == $PROGRAM_NAME
  require 'gtk_paradise/run.rb'
  editable_combo_field = Gtk::EditableComboField::ListStore.new
  r = Gtk.runner(title: 'Editable Combo Field')
  r << editable_combo_field
  r.width = editable_combo_field.width?
  r.enable_quick_exit
  r.top_left_then_run
end # editable_combo_field