#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# require 'gtk_paradise/sys_cmd_button.rb'
# =========================================================================== #
module Gtk

module MsWin

# =========================================================================== #
# === SysCmdButton
# =========================================================================== #
class SysCmdButton < ::Gtk::Button

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === ARRAY_REMOVE_THESE_STRINGS
  #
  # Sanitize these strings.
  # ========================================================================= #
  ARRAY_REMOVE_THESE_STRINGS = [
    '\/Programme\/',
    'GIMP-2.0/',
    'WinSCP/',
    'OzoneSoft/',
    'VDub-1.6.19/',
    'Vdub-1.6.19/',
    'Inkscape/',
    'program/',
    'ContextFree/',
    'Notepad++/',
  ]
  # Extend Windows path
  ENV['PATH'] += ':/WINDOWS/System32'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH  = 45

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 35

  # ========================================================================= #
  # === initialize
  #
  #   1) command_to_run
  #   2) optional_tooltip
  #   3) optional_entry
  #      If given, we will set it.
  #   4) optional_icon
  #
  # ========================================================================= #
  def initialize(
      command_to_run   = '',
      master_widget    = nil,
      optional_tooltip = nil,
      optional_entry   = nil,
      optional_icon    = nil,
      optional_colour  = 'slateblue',
      optional_real_location_of_exe = nil
    )
    super(
      @text_of_label = sanitized_command_to_run(command_to_run)
    )
    @path_to_exe = command_to_run
    set_command_to_run(command_to_run)
    @master_entry = optional_entry
    unless optional_real_location_of_exe.nil?
      @real_location_of_exe = '/home/x/src/windows/'+
                              optional_real_location_of_exe
    end
    on_clicked {
      master_widget.run_sys_command(@command_to_run)
      @master_entry.set_text(@command_to_run) if @master_entry
      if label.include? 'MISSING' # so that we can install it.
        pp @real_location_of_exe
        system(@real_location_of_exe) unless @real_location_of_exe.nil? 
      end
    }
    set_border_width 2
    set_size_request(WIDTH, HEIGHT) 
    create_tooltip(optional_tooltip) unless optional_tooltip.nil?
    set_colours(optional_colour)
  end
  
  # ========================================================================= #
  # === set_colours
  #
  # Set all colours used in our widget.
  # ========================================================================= #
  def set_colours(
      this_colour = 'honeydew'
    )
    modify_background(:active, this_colour.to_sym)
    modify_background(:normal, this_colour.to_sym)
  end

  # ========================================================================= #
  # === sanitized_command_to_run
  #
  # This removes some ugly path things which we do not want to
  # display on our buttons.
  # ========================================================================= #
  def sanitized_command_to_run(string)
    string = string.dup if string.frozen?
    # testing whether file exists or not for all programs that 
    # reside under /Programme
    if string.include? 'Programme'
      val = File.exist? string
      case val
      when true
        # e 'File exists: '+string
      when false
        e 'File does not exist: `'+string+'`'
        string << ' IS MISSING!'
      end
    end
    ARRAY_REMOVE_THESE_STRINGS.each { |entry| 
      entry = entry.gsub(/\+/, '\+')
      string = string.gsub(/#{entry}/,'')
    }
    return string.gsub(/bin\//, '')
  end

  # ========================================================================= #
  # === set_command_to_run
  #
  # This setter will do some sanitizing.
  # ========================================================================= #
  def set_command_to_run(i)
    case i
    when 'calc'
      i = '/mnt/c/windows/system32'
    end
    @command_to_run = i
  end

  # ========================================================================= #
  # === create_tooltip
  # ========================================================================= #
  def create_tooltip(string_to_display)
    unless string_to_display.empty?
      ::Gtk::Tooltips.new.set_tip(self, string_to_display, nil)
    end
  end

end; end; end # msys