#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::ButtonOpenRemoteWebpage
#
# This button can open a remote webpage, upon the user clicking on it.
# =========================================================================== #
# require 'gtk_paradise/gui/gtk3/button_open_remote_webpage/button_open_remote_webpage.rb'
# Gtk::ButtonOpenRemoteWebpage.run(ARGV)
# =========================================================================== #
require 'gtk_paradise/require_gtk3'

module Gtk

class ButtonOpenRemoteWebpage < ::Gtk::Button # === Gtk::ButtonOpenRemoteWebpage

  begin
    require 'open'
  rescue LoadError; end

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Button open remote webpage'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = '25% or minimum 200px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = '10% or minimum 100px'

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :dejavu_condensed_20

  # ========================================================================= #
  # === DEFAULT_URL
  #
  # This URL is just for testing-purposes; it can be changed at any
  # moment in time.
  # ========================================================================= #
  DEFAULT_URL = 'https://www.youtube.com/watch?v=-VAKfwCem68'

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      commandline_arguments = ARGV,
      run_already           = true
    )
    super()
    reset
    set_commandline_arguments(
      commandline_arguments
    )
    set_remote_webpage(commandline_arguments)
    set_label(return_modified_label_based_on_the_remote_webpage)
    # ======================================================================= #
    # === Handle blocks given next
    # ======================================================================= #
    if block_given?
      yielded = yield
      case yielded
      # ===================================================================== #
      # === :make_bold
      # ===================================================================== #
      when :make_bold
        make_bold
      end
    end
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    # ======================================================================= #
    # === Set the title, width, height and the font in use.
    # ======================================================================= #
    title_width_height_font(TITLE, WIDTH, HEIGHT, USE_THIS_FONT)
    use_gtk_paradise_project_css_file  
    infer_the_size_automatically
    # ======================================================================= #
    # === @remote_webpage
    # ======================================================================= #
    @remote_webpage = nil
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    4
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    2
  end

  # ========================================================================= #
  # === respond_to_the_button_clicked_action
  # ========================================================================= #
  def respond_to_the_button_clicked_action(
      use_this_url = @remote_webpage
    )
    on_clicked {
      Open.in_browser(use_this_url)
    }
  end

  # ========================================================================= #
  # === return_modified_label_based_on_the_remote_webpage
  # ========================================================================= #
  def return_modified_label_based_on_the_remote_webpage
    "➡️ #{@remote_webpage}"
  end

  # ========================================================================= #
  # === set_remote_webpage
  # ========================================================================= #
  def set_remote_webpage(i)
    i = [i].flatten.compact
    if i.is_a? Array
      if i.empty?
        i << DEFAULT_URL
      else
        i.map! {|entry|
          case entry
          # =================================================================== #
          # === :default_url
          # =================================================================== #
          when :default_url
            entry = DEFAULT_URL
          end
          entry
        }
      end
    end
    i = i.first if i.is_a? Array
    @remote_webpage = i
  end

  # ========================================================================= #
  # === which_url?
  # ========================================================================= #
  def which_url?
    @remote_webpage
  end

  # ========================================================================= #
  # === determine_the_tooltips_in_use
  # ========================================================================= #
  def determine_the_tooltips_in_use
    self.hint = "Click on this button to open the URL at <b>"\
                "#{which_url?}</b>."
  end

  # ========================================================================= #
  # === create_skeleton                            (create tag, skeleton tag)
  # ========================================================================= #
  def create_skeleton
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    abort_on_exception
    set_name('button1')
    determine_the_tooltips_in_use
    disallow_resizing
    set_size_request(100, 50)
    respond_to_the_button_clicked_action
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    create_skeleton_then_connect_skeleton
  end

  # ========================================================================= #
  # === Gtk::ButtonOpenRemoteWebpage.run
  # ========================================================================= #
  def self.run(
      i = ARGV
    )
    require 'gtk_paradise/run'
    if i.is_a? Array
      i = i.first
    end
    if i.nil? or i.empty?
      i = :default_url
    end
    _ = ::Gtk::ButtonOpenRemoteWebpage.new(i) { :make_bold }
    r = ::Gtk.run
    r << _
    r.automatic_size_then_automatic_title
    r.top_left_then_run
  end

end; end

if __FILE__ == $PROGRAM_NAME
  Gtk::ButtonOpenRemoteWebpage.run
end # ButtonOpenRemoteWebpage