#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::ColouredUnit
#
# This class can be used to make a coloured status bar.
# =========================================================================== #
# require 'gtk_paradise/gui/gtk3/coloured_unit/coloured_unit.rb'
# Gtk::ColouredUnit.run
# =========================================================================== #
require 'gtk_paradise/require_gtk3'

module Gtk

class ColouredUnit < ::Gtk::EventBox # === Gtk::ColouredUnit 

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Coloured Unit'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = '25% or minimum 200px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = '15% or minimum 100px'

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :dejavu_condensed_20

  # ========================================================================= #
  # === DEFAULT_COLOUR
  # ========================================================================= #
  DEFAULT_COLOUR = :steelblue

  # ========================================================================= #
  # === initialize
  #
  # widget_bg_colour is the colour of our parent.
  # ========================================================================= #
  def initialize(
      colour_to_use    = 'orange',
      widget_bg_colour = 'gray',
      size_request     = 7 # in n px
    )
    super()
    reset
    set_bg_colour(colour_to_use)
    modify_background(:normal,   @bg_colour)
    modify_background(:prelight, @bg_colour)
    modify_background(:active,   @bg_colour)
    set_size_request(size_request, size_request)
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    # ======================================================================= #
    # === Set the title, width, height and the font in use.
    # ======================================================================= #
    title_width_height_font(TITLE, WIDTH, HEIGHT, USE_THIS_FONT)
    use_gtk_paradise_project_css_file # or use use_project_css_file 
    infer_the_size_automatically
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    4
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    2
  end

  # ========================================================================= #
  # === set_bg_colour
  # ========================================================================= #
  def set_bg_colour(i = nil)
    case i
    when :nil
      i = DEFAULT_COLOUR
    end
    if i.is_a? Array
      if i.empty?
        i << DEFAULT_COLOUR
      end
      i = i.join(' ').strip
    end
    @bg_colour = i.to_sym
  end

  # ========================================================================= #
  # === change_colour
  #
  # This method will set the same colour to all three states:
  #
  #   :normal, :prelight and :active
  #
  # ========================================================================= #
  def change_colour(
      new_colour = 'red'
    )
    if new_colour.is_a? Array
      new_colour = new_colour.join(' ').strip
    end
    modify_background(:normal,   Gdk::Color.parse(new_colour))
    modify_background(:prelight, Gdk::Color.parse(new_colour))
    modify_background(:active,   Gdk::Color.parse(new_colour))
  end

  # ========================================================================= #
  # === Gtk::ColouredUnit.run
  # ========================================================================= #
  def self.run(
      i = ARGV
    )
    require 'gtk_paradise/run'
    _ = ::Gtk::ColouredUnit.new(i)
    r = ::Gtk.run
    r << _
    r.automatic_size_then_automatic_title
    r.top_left_then_run
  end

end; end

if __FILE__ == $PROGRAM_NAME
  Gtk::ColouredUnit.run
end