#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::Editor
#
# This is an editor in ruby-gtk3, which should one day replace my old
# gtk2-based editor that I am still using (bluefish).
#
# The very first ruby-gtk2 editor that I wrote was created on
# 20.03.2006, based mostly on vim.
# =========================================================================== #
# require 'gtk_paradise/widgets/gtk3/editor/editor.rb'
# Gtk::Editor.run
# =========================================================================== #
require 'gtk_paradise/widgets/gtk3/editor/connect_skeleton.rb'
require 'gtk_paradise/widgets/gtk3/editor/create.rb'
require 'gtk_paradise/widgets/gtk3/editor/misc.rb'

module Gtk

class Editor < ::Gtk::BaseModuleBox # === Gtk::Editor

  begin
    require 'save_file'
  rescue LoadError; end

  begin
    require 'colours'
  rescue LoadError; end

  require 'gtk_paradise/widgets/gtk3/insert_character/insert_character.rb'
  require 'gtk_paradise/widgets/gtk3/font_size_widget/font_size_widget.rb'
  require 'gtk_paradise/widgets/gtk3/numbered_lines/numbered_lines.rb'
  require 'gtk_paradise/widgets/gtk3/colour_palette/colour_palette.rb'

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      commandline_arguments = ARGV,
      run_already           = true
    )
    super(:vertical)
    reset
    set_commandline_arguments(
      commandline_arguments
    )
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      case yielded
      # ===================================================================== #
      # === :do_not_use_the_colour_box_widget
      # ===================================================================== #
      when :do_not_use_the_colour_box_widget
        do_not_use_the_colour_box_widget
      end
    end
    run if run_already
  end

  # ========================================================================= #
  # === handle_CSS_rules                                            (CSS tag)
  # ========================================================================= #
  def handle_CSS_rules
    use_gtk_paradise_project_css_file
    more_CSS '
#colour_palette_text {
  color: darkblue;
  border-bottom: 1px solid darkgreen;
  padding: 1px; 
}

#scrollbar_for_the_editor scrollbar {
  padding: 1px;
  min-width: 5px;
}

#scrollbar_for_the_editor trough {
  background-color: transparent;
  background-repeat: no-repeat;
	background-position: center;
}

#scrollbar_for_the_editor slider {
  /* Size of the slider */
  min-width:     5px;
  min-height:    5px;
  border-radius: 5px;
  margin:         0;
  min-height:    5px;
  min-width:     5px;
  background-color: steelblue;

  /* Padding around the slider */
  border: 2px solid steelblue;
}

notebook {
  padding: 2px;
  margin:  2px;
}

#button_clear_the_textbuffer {
  border: 2px solid midnightblue;
}

  '
  end

  # ========================================================================= #
  # === array_containing_the_text_buffers?
  # ========================================================================= #
  def array_containing_the_text_buffers?
    @hash_all_tab_widgets.map {|object_id, inner_hash|
      inner_hash[:text_buffer]
    }
  end

  # ========================================================================= #
  # === array_containing_the_numbered_lines?
  # ========================================================================= #
  def array_containing_the_numbered_lines?
    @hash_all_tab_widgets.map {|object_id, inner_hash|
      inner_hash[:numbered_lines]
    }
  end

  # ========================================================================= #
  # === array_containing_the_scrolled_windows?
  # ========================================================================= #
  def array_containing_the_scrolled_windows?
    return @hash_all_tab_widgets.map {|object_id, inner_hash|
      inner_hash[:scrolled_window]
    }
  end

  # ========================================================================= #
  # === entry_for_insert_this_text?
  #
  # This method will yield the entry that contains the "insert this text"
  # part.
  # ========================================================================= #
  def entry_for_insert_this_text?
    @entry_insert_this_text
  end

  # ========================================================================= #
  # === retrieve_text
  #
  # Retrieve the selected text from the GtkTextView and display it to 
  # the user.
  # ========================================================================= #
  def retrieve_text(
      textview
    )
    start_iter, end_iter, selected = textview.buffer.selection_bounds
    # ======================================================================= #
    # The next clause can be used to select everything.
    #   if !selected
    #     start_iter, end_iter = textview.buffer.bounds
    #   end
    # ======================================================================= #
    if selected.nil?
      selected = ''
    end
    if start_iter and end_iter
      text = textview.buffer.get_text(start_iter, end_iter, false) # false for "include_hidden_chars"
    else
      text = ''
    end
    @entry_insert_this_text.set_text(text)
    return text
  end

  # ========================================================================= #
  # === insert_text                                              (insert tag)
  #
  # Insert the text from the GtkEntry into the GtkTextView.
  # ========================================================================= #
  def insert_text(
      entry = @entry_insert_this_text, text_view
    )
    text_buffer = text_view.buffer
    mark = text_buffer.selection_bound # First, obtain the text-mark.
    iter = text_buffer.get_iter_at(mark: mark) # The text-mark is translated into an iter.
    text_buffer.insert(iter, entry.text) # Put the content of the gtk-entry into the buffer.
  end

  # ========================================================================= #
  # === set_colour
  # ========================================================================= #
  def set_colour(i)
    @entry_insert_this_text.set_text(i.to_s)
  end

  # ========================================================================= #
  # === focus_on_tab
  # ========================================================================= #
  def focus_on_tab(i = 1)
    @notebook.focus_on_this_tab(i.to_i - 1)
  end

  # ========================================================================= #
  # === respond_to_on_switched_tab_event
  # ========================================================================= #
  def respond_to_on_switched_tab_event
    @notebook.on_switched_tab { |widget, page, page_number|
      this_is_now_the_main_buffer(page_number)
    }
  end

  # ========================================================================= #
  # === sync_change_to_use_this_font
  # ========================================================================= #
  def sync_change_to_use_this_font(
      use_this_font = use_which_font?
    )
    array_containing_the_scrolled_windows?.each {|this_scrolled_window|
      # ===================================================================== #
      # Every scrolled-window will use the same font.
      # ===================================================================== #
      this_scrolled_window.set_font(use_this_font)
    }
    # ======================================================================= #
    # === Sync the font in use onto the numbered-lines widget
    #
    # We also have to sync to the widget on the left side that displays 
    # the line numbers.
    # ======================================================================= #
    array_containing_the_numbered_lines?.each {|this_numbered_line|
      this_numbered_line.set_font(use_this_font)
    }
  end

  # ========================================================================= #
  # === start_iter?
  #
  # Obtain the start-iter from the main buffer.
  # ========================================================================= #
  def start_iter?(
      buffer = main_text_buffer?
    )
    buffer.start_iter
  end

  # ========================================================================= #
  # === do_focus_on_the_first_tab
  # ========================================================================= #
  def do_focus_on_the_first_tab
    @notebook.focus_on_the_first_tab
    @notebook.set_focus(true)
  end

  # ========================================================================= #
  # === add_status_message
  # ========================================================================= #
  def add_status_message(i)
    @status_bar.push(@status_bar_context_id, i)
  end

  # ========================================================================= #
  # === create_the_status_bar
  # ========================================================================= #
  def create_the_status_bar
    @status_bar = return_default_status_bar
    @status_bar.use_this_font = smallest_non_monospaced_font?
    @status_bar_context_id = @status_bar.get_context_id(
      'The statusbar for this editor.'
    )
  end

  # ========================================================================= #
  # === return_current_iter
  # ========================================================================= #
  def return_current_iter(
      buffer = main_buffer?
    )
    buffer.get_iter_at(offset: -1)
  end

  # ========================================================================= #
  # === initialize_the_hash_containing_all_buffer_tags
  # ========================================================================= #
  def initialize_the_hash_containing_all_buffer_tags(
      text_buffer = array_containing_the_text_buffers?[0]
    )
    _ = Gtk::TextBufferTags.create_default_tags(text_buffer)
    @hash_all_tags[:bold]      = _.return_bold_tag
    @hash_all_tags[:underline] = _.return_underline_tag
    # ======================================================================= #
    # Set up different tags next.
    # ======================================================================= #
    text_buffer.onto_the_end(
      "underlined!!!\n",
      tags: [ @hash_all_tags[:underline] ]
    )
    text_buffer.onto_the_end(
      "bold text example!!!\n",
      tags: [ @hash_all_tags[:bold] ]
    )
    text_buffer.insert(text_buffer.start_iter, "class Foobar # this is a comment\nend\n") # Insert some regular text.
    text_buffer.insert(text_buffer.start_iter, "This is some text.\n") # Insert some regular text.
    # ======================================================================= #
    # Next, insert some markup text - bold text.
    # ======================================================================= #
    text_buffer.append_markup(
      "<b>and some bold text</b> via <b>markup-tags</b>.\n"
    )
    text_buffer.append_italic("And this will be in italic.\n")
    text_buffer.append_markup("... <i>and italic, <b>too</b>!</i> \\o/\n\n")
    %w( red gold green steelblue white ).each {|entry|
          text_buffer.insert_markup(
        text_buffer.end_iter,
        "<span color=\"#{entry}\">This is a test message via the span-tag.</span>\n",
        -1
      )
    }
    darkblue = text_buffer.create_tag(nil, { foreground: 'darkblue' })
    iter = text_buffer.return_current_iter
    text_buffer.insert(iter, 'A hyperlink: ', tags: [darkblue] )
    # text_buffer.append_markup(
    #   "\nAnd a hyperlink: <a href=\"www.google.at\">google</i>"
    # )
    # The a-href tag does not work.
    iter = text_buffer.return_current_iter
    text_buffer.insert(iter, "This is more text that "\
      "is just added (appended, actually).\n")
    text_buffer.append_markup(
      '<span foreground="blue" underline="single">ABC</span>'
    )
  end; alias initialize_the_buffer_tags initialize_the_hash_containing_all_buffer_tags # === initialize_the_buffer_tags

  # ========================================================================= #
  # === update_the_main_buffer
  # ========================================================================= #
  def update_the_main_buffer(i = '')
    main_buffer?.set_text(i.to_s)
  end

  # ========================================================================= #
  # === create_the_topbar
  # ========================================================================= #
  def create_the_topbar
    # ======================================================================= #
    # === @top_bar
    # ======================================================================= #
    @top_bar = return_top_bar
    @top_bar.spacing = 5
    # ======================================================================= #
    # Create the open-file button next.
    # ======================================================================= #
    button = gtk_button
    button.add(image_open_file)
    button.hint = 'This button can be used to open a local '\
                  'file into the main view of the editor.'
    button.on_clicked {
      do_create_a_new_file_chooser_dialog
    }
    @top_bar << button
    # ======================================================================= #
    # Create the add-a-new-tab button next.
    # ======================================================================= #
    button = gtk_button
    button.add(image_bookmark_new)
    button.hint = 'Click this button to create a new tab.'
    button.on_clicked {
      do_create_a_new_tab
    }
    @top_bar << button
    # ======================================================================= #
    # Create the remove-a-new-tab button next.
    # ======================================================================= #
    button = gtk_button
    button.add(image_list_remove)
    button.hint = 'Click this button to remove the last tab. So currently '\
      'you have to reposition the tabs properly, before then removing the '\
      'last tab as such.'
    button.on_clicked {
      remove_the_last_tab
    }
    @top_bar << button
  end; alias create_the_top_bar create_the_topbar # === create_the_top_bar

  # ========================================================================= #
  # === add_a_new_tab
  # ========================================================================= #
  def add_a_new_tab(
      fill_the_buffer_with_this_string = ''
    )
    scrolled_window, title = return_a_new_scrolled_window(fill_the_buffer_with_this_string)
    @notebook.add_flexible_tab(
      widget: scrolled_window,
      title:  title
    )
    @notebook.show_all
  end; alias do_create_a_new_tab add_a_new_tab # === do_create_a_new_tab

  # ========================================================================= #
  # === object_ids?
  # ========================================================================= #
  def object_ids?
    @hash_all_tab_widgets.keys
  end

  # ========================================================================= #
  # === remove_the_last_tab
  # ========================================================================= #
  def remove_the_last_tab
    last_tab = @notebook.remove_the_last_tab
    # ======================================================================= #
    # We must also de-register it from the main Hash if it is part of
    # the main hash.
    # ======================================================================= #
    if object_ids?.include? last_tab.object_id
      @hash_all_tab_widgets.delete(last_tab.object_id)
    end
  end

  # ========================================================================= #
  # === create_the_gtk_insert_character_widget
  # ========================================================================= #
  def create_the_gtk_insert_character_widget(
      i = 0
    )
    # ======================================================================= #
    # === @insert_character
    # ======================================================================= #
    @insert_character = ::Gtk::InsertCharacter.new
    @insert_character.css_classes('mar0px pad0px')
    @insert_character.do_delete_the_selected_text_in_the_buffer
    this_is_now_the_main_buffer(i)
  end

  # ========================================================================= #
  # === create_then_add_the_header_bar
  # ========================================================================= #
  def create_then_add_the_header_bar
    header_bar = header_bar?
    minimal(header_bar)
  end

  # ========================================================================= #
  # === delete_the_selected_text_in_the_buffer
  # ========================================================================= #
  def delete_the_selected_text_in_the_buffer
    main_buffer?.delete_selection(false, false)
  end

  # ========================================================================= #
  # === this_is_now_the_main_buffer
  # ========================================================================= #
  def this_is_now_the_main_buffer(i)
    if i and @insert_character
      if i.is_a? Numeric
        i = array_containing_the_text_buffers?[i] 
      end
      @insert_character.attach_to_this_buffer(i)
    end
  end

  # ========================================================================= #
  # === is_it_a_ruby_file?
  # ========================================================================= #
  def is_it_a_ruby_file?
    @entry_save_into_this_local_file.text?.end_with? '.rb'
  end

  # ========================================================================= #
  # === cooldown_start?
  # ========================================================================= #
  def cooldown_start?
    @cooldown_start
  end

  # ========================================================================= #
  # === is_the_cooldown_phase_over?
  #
  # When is the cooldown phase over?
  #
  #   (1) when it is nil, which is the initial case
  #   (2) when the current time is less than 0.5 seconds within the old time
  #
  # ========================================================================= #
  def is_the_cooldown_phase_over?
    _ = cooldown_start?
    if _ and (Time.now - 0.5) > _
      _ = nil
    end
    return true if _.nil?
    return false
  end

  # ========================================================================= #
  # === start_cooldown_phase
  # ========================================================================= #
  def start_cooldown_phase
    @cooldown_start = ::Time.now
  end

  # ========================================================================= #
  # === do_clear_the_textbuffer_based_on_object_id
  # ========================================================================= #
  def do_clear_the_textbuffer_based_on_object_id
    current_buffer = current_buffer?
    current_buffer.clear if current_buffer
  end; alias clear_the_textbuffer do_clear_the_textbuffer_based_on_object_id # === clear_the_textbuffer

  # ========================================================================= #
  # === insert_this_text_at_the_cursor
  # ========================================================================= #
  def insert_this_text_at_the_cursor(
      this_text
    )
    current_buffer = current_buffer?
    current_buffer.insert_at_cursor(this_text.to_s)
  end; alias insert_this_text_at_the_cursor_position insert_this_text_at_the_cursor # === insert_this_text_at_the_cursor_position

  # ========================================================================= #
  # === current_buffer?
  # ========================================================================= #
  def current_buffer?(
      this_id = @notebook.get_nth_page(active_notebook_tab?).object_id
    )
    hash = @hash_all_tab_widgets
    if hash.has_key? this_id
      pointer = hash[this_id]
      return pointer[:text_buffer]
    end
    return nil
  end; alias main_buffer?         current_buffer? # === main_buffer?
       alias current_main_buffer? current_buffer? # === current_main_buffer?
       alias current_text_buffer? current_buffer? # === current_text_buffer?

  # ========================================================================= #
  # === return_the_colour_palette
  # ========================================================================= #
  def return_the_colour_palette
    # ======================================================================= #
    # Add the Colour-Palette widget next:
    # ======================================================================= #
    colour_palette = ::Gtk::ColourPalette.new(:do_not_run_yet) {{
      dimension: '3x1' # Specify the dimension to use here, via x-axis x y-axis.
    }}
    colour_palette.set_parent_widget(self)
    colour_palette.run
    frame = gtk_frame('Colour palette')
    frame.use_this_font = smallest_font?
    frame.set_name('colour_palette_text')
    frame.make_bold
    frame.add(colour_palette)
    return frame
  end

  # ========================================================================= #
  # === set_buffer
  #
  # This method can be used to set the content of the buffer.
  # ========================================================================= #
  def set_buffer(i)
    main_text_buffer?.set_text(i.to_s)
  end

  # ========================================================================= #
  # === Gtk::Editor.run
  # ========================================================================= #
  def self.run(
      i = ARGV
    )
    require 'gtk_paradise/run'
    _ = ::Gtk::Editor.new(i)
    r = ::Gtk.run { :add_a_new_accel_group }
    r << _
    _.set_parent_widget(r) # Must come before we enable the key-combinations.
    _.enable_these_key_combinations(HASH_DESIGNATED_KEY_COMBINATIONS)
    r.automatic_size_then_automatic_title
    r.on_button_press_event { |widget, event|
      if ::Gtk.right_mouse_click?(event) # right mouse click event.
        _.context_menu?.popup_based_on_this_event(event)
      end
    }
    r.add_shortcut(1, 'focus_on_tab(1)', :strg)
    r.add_shortcut(2, 'focus_on_tab(2)', :strg)
    r.add_shortcut(3, 'focus_on_tab(3)', :strg)
    r.add_shortcut(4, 'focus_on_tab(4)', :strg)
    r.add_shortcut(5, 'focus_on_tab(5)', :strg)
    r.add_shortcut(6, 'focus_on_tab(6)', :strg)
    r.add_shortcut(7, 'focus_on_tab(7)', :strg)
    r.add_shortcut(8, 'focus_on_tab(8)', :strg)
    r.add_shortcut(9, 'focus_on_tab(9)', :strg)
    r.maximize # It is simply more convenient that way.
    r.top_left_then_run
  end

  # ========================================================================= #
  # === hash_all_tab_widgets?
  # ========================================================================= #
  def hash_all_tab_widgets?
    @hash_all_tab_widgets
  end; alias hash? hash_all_tab_widgets? # === hash?

  # ========================================================================= #
  # === set_save_into_this_local_file
  # ========================================================================= #
  def set_save_into_this_local_file(i)
    @entry_save_into_this_local_file.set_text(i.to_s)
  end

  # ========================================================================= #
  # === do_save_the_buffer
  # ========================================================================= #
  def do_save_the_buffer(
      what = main_text_buffer?.text.to_s
    )
    into = @entry_save_into_this_local_file.text?
    # ======================================================================= #
    # e "Storing into the file `#{sfile(into)}`."
    # ^^^ This is no longer necessary, as we now store into the status-bar.
    # ======================================================================= #
    add_status_message("Saving into: #{into}")
    SaveFile.write_what_into(what, into)
    add_a_red_border_around_the_save_file_entry_via_a_delay
    Thread.new {
      sleep 2 # Sleep for 2 seconds.
      add_status_message('') # Clear the old message again.
    }
  end; alias save_the_buffer       do_save_the_buffer # === save_the_buffer
       alias do_save_into_the_file do_save_the_buffer # === do_save_into_the_file

  # ========================================================================= #
  # === add_a_red_border_around_the_save_file_entry_via_a_delay
  # ========================================================================= #
  def add_a_red_border_around_the_save_file_entry_via_a_delay
    entry = @entry_save_into_this_local_file
    entry.remove_css_class(:bblack1)
    entry.add_css_class('border_crimson_3px')
    Thread.new {
      sleep 2
      entry.remove_css_class('border_crimson_3px')
      entry.add_css_class(:bblack1)
    }
  end

  # ========================================================================= #
  # === create_the_notebook_tab
  # ========================================================================= #
  def create_the_notebook_tab
    @notebook = return_default_notebook
    respond_to_on_switched_tab_event
    # ======================================================================= #
    # React to on-tab-switch events next:
    # ======================================================================= #
    @notebook.signal_connect_after(:switch_page) { |widget, page, page_number|
      update_the_label_showing_how_many_lines_are_in_the_current_buffer
      update_the_entry_save_the_file
    }
    # ======================================================================= #
    # Add 8 tabs by default next:
    # ======================================================================= #
    8.times {
      # ===================================================================== #
      # The following is just test-code. It was commented out as of
      # February 2022, but we will keep it for the time being just
      # in case.
      # ===================================================================== #
      # n_times = rand(5)+1
      # _ = ''.dup
      # n_times.times {
      #   _ << "AHIFOUVEUHIVE\n"
      # }
      # add_a_new_tab(_)
      add_a_new_tab
    }
  end

  # ========================================================================= #
  # === update_the_entry_save_the_file
  # ========================================================================= #
  def update_the_entry_save_the_file
    entry = @entry_save_into_this_local_file
    hash = @hash_all_tab_widgets
    this_id = return_current_notebook_tab.object_id
    # ======================================================================= #
    # We must obtain the current file path next.
    # ======================================================================= #
    file_path = nil
    if hash.has_key? this_id
      pointer = hash[this_id]
      file_path = pointer[:file_path]
    end
    if file_path
      entry.set_text(file_path.to_s)
    end
  end

  # ========================================================================= #
  # === return_a_new_scrolled_window
  #
  # This method will return a proper scrolled-window.
  # ========================================================================= #
  def return_a_new_scrolled_window(
      fill_the_buffer_with_this_string = ''
    )
    text_buffer = gtk_text_buffer(fill_the_buffer_with_this_string)
    text_buffer.on_cursor_position_changed {|widget, event|
      update_the_label_indicating_the_cursor_position(
        text_buffer.cursor_position? + 1
      )
      update_the_label_displaying_how_many_lines_are_in_the_given_file(
        text_buffer.text?.count("\n") + 1
      )
    }
    # ======================================================================= #
    # Let the buffer respond to certain events next; most importantly
    # the on-changed event. Be careful about infinite loops, though - if
    # we re-act to every on-changed event, and a method call triggers
    # another one, then we have an infinite loop.
    # ======================================================================= #
    text_buffer.signal_connect(:changed) { |widget, event|
      update_n_lines(text_buffer)
      # ===================================================================== #
      # The next check is used to avoid the infinite loop problem described
      # above. This is currently (October 2021) a bit buggy in the sense
      # that the old iter is not kept; this may have to be changed at
      # a later time.
      # ===================================================================== #
      # if is_it_a_ruby_file? and is_the_cooldown_phase_over?
      #   treat_the_buffer_as_a_ruby_file
      #   start_cooldown_phase
      # end
    }
    text_view = gtk_text_view(text_buffer)
    text_view.left_margin = LEFT_MARGIN_TO_USE_FOR_THE_TEXTVIEW_WIDGET
    text_view.pad2px
    text_view.set_wrap_mode(:word) # Used to be :word until May 2022.
    # ======================================================================= #
    # Each such scrolled window will carry the main text-buffer in use.
    # ======================================================================= #
    hbox = gtk_hbox
    numbered_lines = Gtk::NumberedLines.new
    hbox.minimal(numbered_lines)
    hbox.maximal(text_view)
    scrolled_window = gtk_scrolled_window(hbox) { :up_and_down_always }
    scrolled_window.set_hexpand(true)
    scrolled_window.set_vexpand(true)
    scrolled_window.set_name('scrollbar_for_the_editor')
    scrolled_window.width_height(580, 500)
    scrolled_window.pad5px
    scrolled_window.bblack2

    text_view.signal_connect(:key_release_event) {|widget, event|
      name_of_the_event = Gdk::Keyval.to_name(event.keyval).to_s
      case name_of_the_event
      when 'Return','KP_Enter'
        # =================================================================== #
        # Here the user hit the enter-key. Now we check if the line above,
        # above the cursor, begins with a "def" line. In this case we will
        # indent via two ' ' characters.
        # =================================================================== #
        cursor_position = text_buffer.cursor_position?
        string = text_buffer.text?[0 .. cursor_position]
        splitted = string.split("\n")
        last = splitted.last
        if last.strip.start_with? 'def ' # Re-act on "def" lines.
          # ================================================================= #
          # We will also consider the real indent at hand.
          # ================================================================= #
          indent_level = '  ' * (last.rstrip.count(' ') / 2)
          insert_this_text_at_the_cursor_position(indent_level+'end')
        end
      end
      false
    }
    title = '#'+(array_containing_the_text_buffers?.size+1).to_s
    # ======================================================================= #
    # Build up our Hash next:
    # ======================================================================= #
    @hash_all_tab_widgets[scrolled_window.object_id] = {
      numbered_lines:  numbered_lines,
      text_buffer:     text_buffer,
      text_view:       text_view,
      scrolled_window: scrolled_window,
      title:           title,
      file_path:       nil # This has to be filled at a later time.
    }
    return [scrolled_window, title]
  end

  # ========================================================================= #
  # === main_text_buffer?
  # ========================================================================= #
  def main_text_buffer?
    of_this_buffer = active_notebook_tab?
    of_this_buffer = array_containing_the_text_buffers?[of_this_buffer]
    return of_this_buffer
  end; alias buffer_content? main_text_buffer? # === buffer_content?

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    super()
    sync_change_to_use_this_font
    Thread.new {
      sleep 0.0015
      do_focus_on_the_first_tab
      initialize_the_hash_containing_all_buffer_tags
      menu
    }
  end

end; end

if __FILE__ == $PROGRAM_NAME
  Gtk::Editor.run(ARGV)
end