#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# require 'gtk_paradise/widgets/gtk3/editor/misc.rb'
# =========================================================================== #
require 'gtk_paradise/require_gtk3'

module Gtk

class Editor < ::Gtk::BaseModuleBox # === Gtk::Editor

  require 'gtk_paradise/widgets/gtk3/editor/internal_widgets/goto_line_widget.rb'

  begin
    require 'roebe/toplevel_methods/misc.rb'
  rescue LoadError; end

  begin
    require 'roebe/classes/mbl.rb'
  rescue LoadError; end

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'ruby-gtk3-based editor'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = '98% or minimum 500px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = '92% or minimum 400px'

  # ========================================================================= #
  # === WIDTH_OF_A_COLOUR_BUTTON
  # ========================================================================= #
  WIDTH_OF_A_COLOUR_BUTTON = 90

  # ========================================================================= #
  # === HEIGHT_OF_A_COLOUR_BUTTON
  # ========================================================================= #
  HEIGHT_OF_A_COLOUR_BUTTON = 32

  # ========================================================================= #
  # === LEFT_MARGIN_TO_USE_FOR_THE_TEXTVIEW_WIDGET
  # ========================================================================= #
  LEFT_MARGIN_TO_USE_FOR_THE_TEXTVIEW_WIDGET = 5

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :hack_17

  # ========================================================================= #
  # === USE_THIS_SMALLER_FONT
  # ========================================================================= #
  USE_THIS_SMALLER_FONT = :hack_15

  # ========================================================================= #
  # === USE_THIS_SMALLEST_FONT
  # ========================================================================= #
  USE_THIS_SMALLEST_FONT = :hack_12

  # ========================================================================= #
  # === HASH_DESIGNATED_KEY_COMBINATIONS
  #
  # Add useful key-combinations to the following Hash.
  # ========================================================================= #
  HASH_DESIGNATED_KEY_COMBINATIONS = {
    'strg+o': 'do_open_a_local_file',
    'strg+s': 'do_save_into_the_file',
    'strg+g': 'go_to_that_line_after_asking_the_user',
    'alt+1':  'activate_this_tab(1)',
    'alt+2':  'activate_this_tab(2)',
    'alt+3':  'activate_this_tab(3)',
    'alt+4':  'activate_this_tab(4)',
    'alt+5':  'activate_this_tab(5)',
    'alt+6':  'activate_this_tab(6)',
    'alt+7':  'activate_this_tab(7)',
    'alt+8':  'activate_this_tab(8)',
    'alt+9':  'activate_this_tab(9)'
  }

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    title_width_height_font(TITLE, WIDTH, HEIGHT, use_which_font?)
    infer_the_size_automatically
    # ======================================================================= #
    # === @hash_all_tags
    #
    # This Hash keeps track of all buffer-tags that are usable by the
    # notebook.
    # ======================================================================= #
    @hash_all_tags = {}
    # ======================================================================= #
    # === @use_widget_with_different_colour_buttons
    #
    # This refers to the widget on the very right, containing individual
    # buttons. By default they will be shown, but not every user may
    # want this, so we have to be able to toggle this on or off as
    # necessary.
    # ======================================================================= #
    @use_widget_with_different_colour_buttons = true
    # ======================================================================= #
    # === @hash_all_tab_widgets
    # ======================================================================= #
    @hash_all_tab_widgets = {}
    @cooldown_start = nil
    # ======================================================================= #
    # === @goto_line_widget
    # ======================================================================= #
    @goto_line_widget = ::Gtk::GotoLineWidget.new
    handle_CSS_rules
  end

  # ========================================================================= #
  # === return_smaller_font
  # ========================================================================= #
  def return_smaller_font
    USE_THIS_SMALLER_FONT
  end; alias smaller_font? return_smaller_font # === smaller_font?

  # ========================================================================= #
  # === return_font_size_widget
  # ========================================================================= #
  def return_font_size_widget
    # ======================================================================= #
    # === Gtk::FontSizeWidget
    #
    # Next create various fonts, via Gtk::FontSizeWidget.
    # ======================================================================= #
    font_size_widget = ::Gtk::FontSizeWidget.new {{
      use_this_font: smallest_font?,
      display_n_font_buttons_per_line: 3
    }}
    font_size_widget.set_vbox_width_and_height(150, 30)
    font_size_widget.set_use_this_font(USE_THIS_SMALLER_FONT)
    font_size_widget.set_parent_widget(self)
    font_size_widget.css_class_for_the_entry('BG_lightgrey')
    return font_size_widget
  end

  # ========================================================================= #
  # === do_not_use_the_colour_box_widget
  # ========================================================================= #
  def do_not_use_the_colour_box_widget
    @use_widget_with_different_colour_buttons = false
  end

  # ========================================================================= #
  # === smallest_non_monospaced_font?
  # ========================================================================= #
  def smallest_non_monospaced_font?
    :dejavu_14
  end

  # ========================================================================= #
  # === smallest_font?
  # ========================================================================= #
  def smallest_font?
    USE_THIS_SMALLEST_FONT
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    5
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    2
  end

  # ========================================================================= #
  # === top_bar?
  # ========================================================================= #
  def top_bar?
    @top_bar
  end; alias header_bar? top_bar? # === header_bar?

  # ========================================================================= #
  # === return_widget_with_different_colour_buttons
  #
  # This method will return a widget that will contain colour buttons,
  # if the userdecided that it should be used at all to begin with.
  # ========================================================================= #
  def return_widget_with_different_colour_buttons(
      text_buffer = current_main_buffer?
    )
    vbox_containing_the_various_colour_buttons = gtk_vbox
    # ======================================================================= #
    # Obtain all HTML-colours, then iterate over this Array. These are
    # gtk-buttons that will be put into that widget. That widget will
    # normally be on the right-hand side.
    # ======================================================================= #
    return_the_html_colours.each {|this_html_colour|
      this_html_colour = this_html_colour.to_s
      button = gtk_left_aligned_button(this_html_colour)
      button.set_size_request(WIDTH_OF_A_COLOUR_BUTTON, HEIGHT_OF_A_COLOUR_BUTTON)
      button.set_border_width(1)
      button.use_this_colour = this_html_colour
      button.bblack2
      # ===================================================================== #
      # Keep a reference-copy in our main hash for the colour, in regards 
      # to the foreground.
      # ===================================================================== #
      @hash_all_tags[this_html_colour] =
        text_buffer.create_tag(nil, { foreground: this_html_colour })
      button.on_clicked { # This button adds the tag, with the colour at hand.
        # =================================================================== #
        # Since as of February 2021, if the entry for inserting a
        # specific text into the main buffer is NOT empty, then
        # it will be used.
        # =================================================================== #
        insert_this_text_into_the_buffer = "foobar\n"
        unless entry_for_insert_this_text?.text?.empty?
          insert_this_text_into_the_buffer = 
            entry_for_insert_this_text?.text?
        end
        current_text_buffer?.insert(
          current_text_buffer?.end_iter,
          insert_this_text_into_the_buffer,
          tags: [ @hash_all_tags[this_html_colour] ] # This carries the tag defined above.
        )
      }
      # ===================================================================== #
      # Add the button as well:
      # ===================================================================== #
      vbox_containing_the_various_colour_buttons.minimal(button)
    }
    scrolled_window_containing_the_vbox = gtk_scrolled_window(
      vbox_containing_the_various_colour_buttons
    )
    scrolled_window_containing_the_vbox.set_size_request(100, 200)
    scrolled_window_containing_the_vbox.css_class('pad0px')
    scrolled_window_containing_the_vbox.use_this_font = smaller_font?
    return scrolled_window_containing_the_vbox
  end

  # ========================================================================= #
  # === use_which_font?
  # ========================================================================= #
  def use_which_font?
    USE_THIS_FONT
  end; alias main_font? use_which_font? # === main_font?

  # ========================================================================= #
  # === context_menu?
  # ========================================================================= #
  def context_menu?
    @context_menu
  end

  # ========================================================================= #
  # === create_the_context_menu
  # ========================================================================= #
  def create_the_context_menu
    # ======================================================================= #
    # === @context_menu
    # ======================================================================= #
    @context_menu = create_context_menu(self) {{
      actions: {
        '• Colourize the text':    :do_colourize_the_text,
        '• Clear the textbuffer':  :clear_the_textbuffer,
        '• Obtain the selection':  :obtain_the_selection,
        '• Go to a specific line': :go_to_a_specific_line
      }
    }}
    @context_menu.use_this_font(main_font?)
    @context_menu.css_class('bblack3_and_padding')
  end

  # ========================================================================= #
  # === go_to_a_specific_line                                      (goto tag)
  # ========================================================================= #
  def go_to_a_specific_line(
      go_to_this_line = 3
    )
    go_to_this_line = go_to_this_line.to_i
    current_buffer = current_buffer?
    lines = current_buffer.text?.lines
    go_to_this_line -= 1 unless go_to_this_line < 1
    if go_to_this_line.size > lines.size
      go_to_this_line = lines.size
    end
    # ======================================================================= #
    # We must calculate the true cursor position next. We do this by 
    # calculating the number of lines via a method available in the
    # gtk_paradise gem.
    # ======================================================================= #
    real_cursor_position = Gtk.return_n_characters_up_to_array_position(go_to_this_line)
    current_buffer.place_cursor_at_this_position(
      real_cursor_position
    )
    do_place_the_cursor
  end; alias go_to_that_line go_to_a_specific_line # === go_to_that_line

  # ========================================================================= #
  # === dialog_for_go_to_a_specific_line
  # ========================================================================= #
  def dialog_for_go_to_a_specific_line
    @goto_line_widget.show_all # Always show it when this is called.
    @goto_line_widget.runner_widget.show_all if @goto_line_widget.runner_widget
    # ======================================================================= #
    # We must re-sync onto the real buffer.
    # ======================================================================= #
    @goto_line_widget.connect_to_this_text_buffer(current_text_buffer?)
    # ======================================================================= #
    # The rest is only called if the goto-line widget hasn't been run yet.
    # ======================================================================= #
    unless @goto_line_widget.runner_widget?
      @goto_line_widget.set_parent_widget(self)
      r = ::Gtk.run
      r.title = 'Goto Line'
      r << @goto_line_widget
      r.automatic_size_then_automatic_title
      @goto_line_widget.set_runner_widget(r)
      r.top_left_then_run
    end
  end; alias go_to_that_line_after_asking_the_user dialog_for_go_to_a_specific_line # === go_to_that_line_after_asking_the_user

  # ========================================================================= #
  # === grab_focus
  # ========================================================================= #
  def grab_focus(
      text_view = main_textview?
    )
    text_view.grab_focus
  end; alias activate_cursor     grab_focus # === activate_cursor
       alias do_place_the_cursor grab_focus # === do_place_the_cursor

  # ========================================================================= #
  # === consider_adding_a_vte_terminal_to_this_notebook             (vte tag)
  # ========================================================================= #
  def consider_adding_a_vte_terminal_to_this_notebook(
      notebook, tab_title = '3'
    )
    begin
      require 'vte3'
    rescue LoadError; end
    if Object.const_defined?(:Vte) and Vte.const_defined?(:PtyFlags)
      # ===================================================================== #
      # Add a tab for VTE specifically.
      # ===================================================================== #
      terminal = Vte::Terminal.new
      if terminal.respond_to? :use_this_font=
        terminal.use_this_font = smallest_font?
      end
      if terminal.respond_to? :spawn_sync
        _succeeded, _pid = terminal.spawn_sync(
          ::Vte::PtyFlags::DEFAULT,
          return_pwd,
          [ENV['SHELL'] || '/bin/sh'],
          nil,
          GLib::Spawn::CHILD_INHERITS_STDIN | GLib::Spawn::SEARCH_PATH
        )                             
        notebook.add_tab(terminal, tab_title)
      else
        e 'The vte-terminal does not respond to spawn_sync().'
        e 'Investigate again at a later time.'
      end
    end
  end

  # ========================================================================= #
  # === activate_this_tab
  #
  # The input-argument starts at 1, which means "activate the
  # first tab".
  # ========================================================================= #
  def activate_this_tab(i = 1)
    notebook?.activate_this_tab(i.to_i - 1)
  end

  # ========================================================================= #
  # === update_the_label_indicating_the_cursor_position
  #
  # This method may only display the given argument - it may not modify
  # the argument.
  # ========================================================================= #
  def update_the_label_indicating_the_cursor_position(
      i = text_buffer?.cursor_position?
    )
    @label_indicate_the_cursor_position.set_markify_text(
      "Cursor position: <b>#{i}</b>"
    )
  end

  # ========================================================================= #
  # === return_current_notebook_tab
  # ========================================================================= #
  def return_current_notebook_tab(
      notebook = notebook?
    )
    return notebook.get_nth_page(active_notebook_tab?)
  end

  # ========================================================================= #
  # === current_textview?
  # ========================================================================= #
  def current_textview?
    this_id = return_current_notebook_tab.object_id
    hash = @hash_all_tab_widgets
    if hash.has_key? this_id
      pointer = hash[this_id]
      return pointer[:text_view]
    end
    return nil
  end; alias textview?      current_textview? # === textview?
       alias main_textview? current_textview? # === main_textview?

  # ========================================================================= #
  # === do_colourize_the_text_for_ruby_code
  #
  # This method can be used to colourize the text-buffer via ruby.
  # ========================================================================= #
  def do_colourize_the_text_for_ruby_code
    current_buffer = current_buffer?
    _ = current_buffer.text?
    # ======================================================================= #
    # We need to keep track of the old cursor position.
    # ======================================================================= #
    old_cursor_position = current_buffer.cursor_position?
    class_regex = /class (\w+)/ # https://rubular.com/r/iTIFTUmI5TjGzp
    comment_regex = /#(.+)/     # https://rubular.com/r/gF5DPTJamYJi4N
    current_buffer.clear
    lines = _.lines
    lines.each_with_index {|line, index|
      copy = line.dup
      case line # Match against the full line next.
      # ===================================================================== #
      # This is a comment
      # ===================================================================== #
      when /# /,
           /#!/ # This variant here is for the shebang specifically.
        copy.gsub!(
          comment_regex,
          "<span weight=\"bold\" foreground=\"slategray\">#\\1</span>"
        )
      end
      case line
      # ===================================================================== #
      # === Handle 'def' keyword next
      #
      # We also must handle method-names such as "def run".
      # ===================================================================== #
      when /def/
        copy.gsub!(
          /def (\w+)/, "def <span weight=\"bold\" foreground=\"darkgreen\">\\1</span>"
        )
        copy.gsub!(
          /def /, "<span weight=\"bold\" foreground=\"darkblue\">def </span>"
        )
      end
      case line
      # ===================================================================== #
      # === Handle 'end' keyword next
      # ===================================================================== #
      when /end/
        copy.gsub!(
          /end/, "<span weight=\"bold\" foreground=\"darkblue\">end</span>"
        )
      end
      case line
      # ===================================================================== #
      # === Handle 'when' keyword next
      # ===================================================================== #
      when /when/
        copy.gsub!(
          /when/, "<span weight=\"bold\" foreground=\"darkblue\">when</span>"
        )
      end
      case line
      # ===================================================================== #
      # === Handle 'case' keyword next
      # ===================================================================== #
      when /case/
        # =================================================================== #
        # Colourize the remaining word via bold dark green.
        # =================================================================== #
        case line
        when /case (\w+)/
          copy.gsub!(
            /case (\w+)/, "<span weight=\"bold\" foreground=\"darkgreen\">case \\1</span>"
          )
        end
        copy.gsub!(
          /case/, "<span weight=\"bold\" foreground=\"darkblue\">case</span>"
        )
      end
      case line
      # ===================================================================== #
      # === Handle 'else' keyword next
      # ===================================================================== #
      when /else/
        copy.gsub!(
          /else/, "<span weight=\"bold\" foreground=\"indianred\">else</span>"
        )
      end
      case line
      # ===================================================================== #
      # === Handle 'if' keyword next
      # ===================================================================== #
      when /if/
        copy.gsub!(
          /if/, "<span weight=\"bold\" foreground=\"indianred\">if</span>"
        )
      end
      case line # Do it again for other regexes.
      # ===================================================================== #
      # === Handle the 'class' keyword next
      # ===================================================================== #
      when /class /
        copy.gsub!(
          class_regex,
          "<span weight=\"bold\" foreground=\"darkblue\">class</span> "+
          "<span weight=\"bold\" foreground=\"steelblue\">\\1</span>"
        )
      end
      # ===================================================================== #
      # else do no modification
      # ===================================================================== #
      current_buffer.append_markup(copy)
    }
    _iter_position = current_buffer.place_cursor_at_this_position(old_cursor_position)
    # ======================================================================= #
    # The next method call is experimental as of February 2022.
    # ======================================================================= #
    do_recalculate_the_number_of_lines
    grab_focus # And we place the cursor here.
  end; alias treat_the_buffer_as_a_ruby_file do_colourize_the_text_for_ruby_code # === treat_the_buffer_as_a_ruby_file
       alias do_colourize_the_text           do_colourize_the_text_for_ruby_code # === do_colourize_the_text
       alias do_colourize_the_buffer         do_colourize_the_text_for_ruby_code # === do_colourize_the_buffer

  # ========================================================================= #
  # === do_recalculate_the_number_of_lines
  # ========================================================================= #
  def do_recalculate_the_number_of_lines
    _ = current_buffer?.text?
    numbered_lines?.repopulate_towards(_.count("\n") + 1)
  end

  # ========================================================================= #
  # === notebook?
  # ========================================================================= #
  def notebook?
    @notebook
  end

  # ========================================================================= #
  # === numbered_lines?
  #
  # This method will always return the correct numbered_lines widget.
  # ========================================================================= #
  def numbered_lines?(
      hash = @hash_all_tab_widgets
    )
    this_id = return_current_notebook_tab.object_id
    if hash.has_key? this_id
      pointer = hash[this_id]
      return pointer[:numbered_lines]
    end
    return nil
  end

  # ========================================================================= #
  # === do_create_a_new_file_chooser_dialog         (open tag, open file tag)
  #
  # This method is run when the user wants to open a local file.
  # ========================================================================= #
  def do_create_a_new_file_chooser_dialog(
      i = main_file?.to_s
    )
    create_a_new_file_chooser_dialog(self) {{
      start_dir:              return_pwd,
      show_hidden_files:      :yes,
      current_folder:         ENV['MY_DATA'],
      additional_directories: [
                                ENV['PC'],
                                ENV['MY_TEMP'],
                                ENV['RSRC'],
                                '/Depot/j/',
                                '/Depot/jj/',
                                '/Depot/jjj/',
                                '/Depot/jjjj/'
                              ]
    }}
    _ = ::Gtk.main_file?.to_s
    if _ and File.exist?(_) and current_buffer?
      hash = @hash_all_tab_widgets
      this_id = return_current_notebook_tab.object_id
      @entry_save_into_this_local_file.set_text(_) # Also sync it onto the file-path entry.
      set_name_of_the_current_tab(File.basename(_))
      dataset = File.read(_).to_s # Read in the file here.
      current_buffer?.set_text(dataset)
      n_newlines = dataset.count("\n")
      numbered_lines?.repopulate_towards(n_newlines)

      if hash.has_key? this_id
        hash[this_id][:file_path] = _ 
      end

      # ===================================================================== #
      # === Handle .rb files next and colourize them accordingly
      # ===================================================================== #
      if _.end_with? '.rb'
        treat_the_buffer_as_a_ruby_file
      end
      update_the_label_showing_how_many_lines_are_in_the_current_buffer
    end
  end; alias do_open_a_local_file do_create_a_new_file_chooser_dialog # === do_open_a_local_file
       alias open_file            do_create_a_new_file_chooser_dialog # === open_file

  # ========================================================================= #
  # === active_notebook_tab?
  #
  # This method will return the current notebook page.
  # ========================================================================= #
  def active_notebook_tab?
    @notebook.page
  end

  # ========================================================================= #
  # === menu                                                       (menu tag)
  # ========================================================================= #
  def menu(
      i = commandline_arguments?
    )
    if i.is_a? Array
      i.each {|entry| menu(entry) }
    else
      # ===================================================================== #
      # === Locally existing files
      #
      # Next add support for reading in the content of locally existing
      # files. This requires of us to clear the old buffer content first.
      # ===================================================================== #
      if i and File.file?(i)
        set_tab_title(File.basename(i))
        content = File.read(i).to_s
        # @notebook.first_tab?.set_text(content)
        main_text_buffer?.set_text('') # Clear the old buffer.
        set_buffer(content)
      end
    end
  end

  # ========================================================================= #
  # === set_name_of_the_current_tab
  # ========================================================================= #
  def set_name_of_the_current_tab(tab_label)
    notebook?.set_tab_label_text(
      return_current_notebook_tab, tab_label
    )
  end; alias set_tab_title set_name_of_the_current_tab # === set_tab_title

  # ========================================================================= #
  # === return_widget_containing_special_buttons
  # ========================================================================= #
  def return_widget_containing_special_buttons
    hbox = gtk_hbox
    begin
      require 'roebe/classes/ruby_main.rb'
      require 'roebe/classes/ruby_header.rb'
      # ===================================================================== #
      # === ruby_header
      # ===================================================================== #
      ruby_header = button('ruby_header')
      ruby_header.css_class('border_1px_steelblue')
      ruby_header.hint = "This button will append:\n\n<b>"+
                         Roebe.return_ruby_header+
                         '</b>'
      ruby_header.on_clicked {
        current_text_buffer?.delete_the_selected_text
        current_text_buffer?.append_after_the_cursor(
          Roebe.return_ruby_header
        )
        do_colourize_the_buffer
      }
      # ===================================================================== #
      # === ruby_end
      # ===================================================================== #
      ruby_end = button('ruby_end')
      ruby_end.css_class('border_1px_steelblue')
      ruby_end.hint = "This button will append:\n\n<b>"+
                       Roebe.ruby_main_string+
                       '</b>'
      ruby_end.on_clicked {
        _ = nil
        current_text_buffer = current_text_buffer?
        if current_text_buffer and current_text_buffer.text?.include?('class ')
          _ = current_text_buffer.text?.scan(/class (\w+)/).flatten.first.to_s
        end
        if current_text_buffer
          # ================================================================= #
          # Remove any old text.
          # ================================================================= #
          current_text_buffer.delete_the_selected_text
          do_place_the_cursor
          new_string = "\n"+Roebe.ruby_main_string(_)
          current_text_buffer.append_after_the_cursor(new_string)
          do_colourize_the_buffer
        end
      }
      # ===================================================================== #
      # === ruby_class
      # ===================================================================== #
      ruby_class = button('ruby_class')
      ruby_class.css_class('border_1px_steelblue')
      ruby_class.hint = "This button will append some class definition.\n"
      ruby_class.on_clicked {
        _ = nil
        current_text_buffer = current_text_buffer?
        if current_text_buffer
          # ================================================================= #
          # Remove any old text.
          # ================================================================= #
          current_text_buffer.delete_the_selected_text
          do_place_the_cursor
          new_string = "\n# === Foobar\nclass Foobar\nend\n"
          current_text_buffer.append_after_the_cursor(new_string)
          do_colourize_the_buffer
        end
      }
      # ===================================================================== #
      # === ruby_initialize
      # ===================================================================== #
      ruby_initialize = button('ruby_initialize')
      ruby_initialize.css_class('border_1px_steelblue')
      ruby_initialize.hint = "This button will just append <b>def initialize</b>.\n"
      ruby_initialize.on_clicked {
        _ = nil
        current_text_buffer = current_text_buffer?
        if current_text_buffer
          # ================================================================= #
          # Remove any old text.
          # ================================================================= #
          current_text_buffer.delete_the_selected_text
          do_place_the_cursor
          new_string = "\n#{Roebe.default_comment_line}\n"\
                       "# === initialize\n"\
                       "#{Roebe.default_comment_line}\ndef initialize\nend\n"
          current_text_buffer.append_after_the_cursor(new_string)
          do_colourize_the_buffer
        end
      }
      # ===================================================================== #
      # === java_create_a_new_class
      # ===================================================================== #
      java_create_a_new_class = button('java_new_class')
      java_create_a_new_class.css_class('border_1px_steelblue')
      java_create_a_new_class.hint =
        "This button will create a new Java class, in the main buffer.\n"
      java_create_a_new_class.on_clicked {
        current_text_buffer = current_text_buffer?
        if current_text_buffer
          # ================================================================= #
          # Remove any old text.
          # ================================================================= #
          current_text_buffer.delete_the_selected_text
          do_place_the_cursor
          new_string = Roebe::Mbl.return_java_string_for_this_class_name('Foobar')
          current_text_buffer.append_after_the_cursor(new_string)
          do_colourize_the_buffer
        end
      }
      vbox1 = vbox
      vbox1.minimal(ruby_header,             1)
      vbox1.minimal(ruby_end,                1)
      vbox1.minimal(ruby_class,              1)
      vbox1.minimal(ruby_initialize,         1)
      vbox1.minimal(java_create_a_new_class, 1)
      scrolling_widget = scrolling_for(vbox1) { :always_up_and_down }
      hbox.minimal(scrolling_widget,    2)
      hbox.minimal(@entry_for_comments, 1)
      # hbox.css_class('BG_Black') # If you want a background colour, use this one here.
    rescue LoadError; end
    return hbox
  end

  # ========================================================================= #
  # === update_the_label_showing_how_many_lines_are_in_the_current_buffer
  # ========================================================================= #
  def update_the_label_showing_how_many_lines_are_in_the_current_buffer(
      text_to_display =
        buffer_content?
    )
    if @label_n_lines_in_the_file and text_to_display
      if text_to_display.respond_to? :text?
        text_to_display = text_to_display.text?
      end
      @label_n_lines_in_the_file.set_markify_text(
        '<b>'+text_to_display.to_s+'</b> lines'
      )
    end
  end; alias update_n_lines                                                   update_the_label_showing_how_many_lines_are_in_the_current_buffer # === update_n_lines
       alias update_the_label_displaying_how_many_lines_are_in_the_given_file update_the_label_showing_how_many_lines_are_in_the_current_buffer # === update_the_label_displaying_how_many_lines_are_in_the_given_file

end; end