#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::FourComponentsEntry
# =========================================================================== #
# require 'gtk_paradise/widgets/gtk3/four_components_entry/four_components_entry.rb'
# Gtk::FourComponentsEntry.run
# =========================================================================== #
require 'gtk_paradise/require_gtk3'

module Gtk

class FourComponentsEntry < ::Gtk::BaseModuleBox # === Gtk::FourComponentsEntry

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = '' # This will not have a title by default.

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = '5% or minimum 50px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = '4% or minimum 40px'

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :dejavu_condensed_18

  # ========================================================================= #
  # === IS_THE_MAIN_ENTRY_BOLD
  # ========================================================================= #
  IS_THE_MAIN_ENTRY_BOLD = true

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      commandline_arguments = ARGV,
      run_already           = true,
      &block
    )
    super(:vertical)
    reset
    set_commandline_arguments(
      commandline_arguments
    )
    @header_widget = create_text # This must come here.
    # ======================================================================= #
    # === Handle blocks given
    # ======================================================================= #
    if block_given?
      yielded = yield
      if yielded.is_a? Hash
        # =================================================================== #
        # === :bold_text
        #
        # This entry point must come before the :header_text check.
        # =================================================================== #
        if yielded.has_key?(:bold_text)
          new_value = yielded.delete(:bold_text)
          if new_value == true
            do_make_the_header_text_bold
          elsif !new_value.empty?
            set_new_text(new_value)
            do_make_the_header_text_bold
          end
        end
        # =================================================================== #
        # === :header_text
        # =================================================================== #
        if yielded.has_key? :header_text
          set_header_text(yielded[:header_text])
        end
      end
    end
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    title_width_height_font(TITLE, WIDTH, HEIGHT, USE_THIS_FONT)
    @header_widget = nil
    handle_CSS_rules
    infer_the_size_automatically
    populate_the_internal_hash
  end

  # ========================================================================= #
  # === populate_the_internal_hash
  # ========================================================================= #
  def populate_the_internal_hash
    # ======================================================================= #
    # === :minimal_value
    # ======================================================================= #
    @internal_hash[:minimal_value] = 0
    # ======================================================================= #
    # === :make_the_header_widget_bold
    # ======================================================================= #
    @internal_hash[:make_the_header_widget_bold] = false
    # ======================================================================= #
    # === :cap_plus_at
    # ======================================================================= #
    @internal_hash[:cap_plus_at] = nil # Default value.
    set_header_text('') # Default value.
  end

  # ========================================================================= #
  # === minimal_value?
  # ========================================================================= #
  def minimal_value?
    @internal_hash[:minimal_value]
  end

  # ========================================================================= #
  # === do_make_the_header_text_bold
  # ========================================================================= #
  def do_make_the_header_text_bold
    @internal_hash[:make_the_header_widget_bold] = true
    set_header_text(:default)
  end

  # ========================================================================= #
  # === make_the_header_widget_bold?
  # ========================================================================= #
  def make_the_header_widget_bold?
    @internal_hash[:make_the_header_widget_bold]
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    0
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    0
  end

  # ========================================================================= #
  # === handle_CSS_rules
  # ========================================================================= #
  def handle_CSS_rules
    use_gtk_paradise_project_css_file
  end

  # ========================================================================= #
  # === create_skeleton                                          (create tag)
  # ========================================================================= #
  def create_skeleton
  end

  # ========================================================================= #
  # === set_value
  # ========================================================================= #
  def set_value(i)
    @entry.set_text(i.to_s)
  end

  # ========================================================================= #
  # === entry?
  #
  # Obtain the main entry here.
  # ========================================================================= #
  def entry?
    @entry
  end

  # ========================================================================= #
  # === return_the_left_entry                                     (entry tag)
  # ========================================================================= #
  def return_the_left_entry
    @entry = create_entry
    @entry.yellow_background
    @entry.center
    @entry.make_bold if IS_THE_MAIN_ENTRY_BOLD
    @entry.make_scrollable
    @entry.on_scroll_event {|widget, event|
      if scroll_up?(event)
        increment_by(1)
      elsif scroll_down?(event)
        decrement_by(1)
      end
    }
    return @entry
  end

  # ========================================================================= #
  # === set_header_text
  # ========================================================================= #
  def set_header_text(i = '')
    case i
    when :default
      i = @internal_hash[:header_text]
    end
    i = i.to_s
    @internal_hash[:header_text] = i
    if make_the_header_widget_bold?
      if @header_widget
        @header_widget.set_bold_text(i)
      end
    end
  end; alias set_new_text set_header_text # === set_new_text

  # ========================================================================= #
  # === header_text?
  # ========================================================================= #
  def header_text?
    @internal_hash[:header_text]
  end; alias main_text? header_text? # === main_text?

  # ========================================================================= #
  # === increment_by                                               (plus tag)
  # ========================================================================= #
  def increment_by(
      i = 1
    )
    _ = @entry.text?.to_i + 1
    if @internal_hash[:cap_plus_at] and (_ > @internal_hash[:cap_plus_at])
      _ = cap_plus_at?
    end
    @entry.set_text(_.to_s)
  end

  # ========================================================================= #
  # === cap_plus_at?
  # ========================================================================= #
  def cap_plus_at?
    @internal_hash[:cap_plus_at]
  end

  # ========================================================================= #
  # === cap_plus_at
  # ========================================================================= #
  def cap_plus_at(i = 12)
    @internal_hash[:cap_plus_at] = i
  end

  # ========================================================================= #
  # === decrement_by
  # ========================================================================= #
  def decrement_by(
      i             = 1,
      minimal_value = minimal_value?
    )
    _ = @entry.text?.to_i - 1
    if _ < minimal_value
      _ = minimal_value
    end
    @entry.set_text(_.to_s)
  end

  # ========================================================================= #
  # === return_the_right_vbox
  #
  # This vbox will contain the + and - sign, both of which are clickable.
  # ========================================================================= #
  def return_the_right_vbox
    minimal_value = minimal_value?
    vbox = create_vbox
    image1 = create_image(::Gtk::PATH_TO_THE_PLUS_SYMBOL_IMAGE)
    image2 = create_image(::Gtk::PATH_TO_THE_MINUS_SYMBOL_IMAGE)
    eventbox1 = create_eventbox(image1)
    eventbox1.on_clicked { increment_by(1) }
    eventbox2 = create_eventbox(image2)
    eventbox2.on_clicked { decrement_by(1, minimal_value) }
    vbox.minimal(eventbox1)
    vbox.minimal(eventbox2)
    vbox.show_all
    return vbox
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    abort_on_exception
    right_box = create_hbox
    right_box.maximal(return_the_left_entry)
    right_box.minimal(return_the_right_vbox)
    most_outer_vbox = create_vbox
    most_outer_vbox.minimal(@header_widget)
    most_outer_vbox.minimal(right_box)
    minimal(most_outer_vbox)
  end

  # ========================================================================= #
  # === hint=
  # ========================================================================= #
  def hint=(i)
    @header_widget.hint = i
  end

  # ========================================================================= #
  # === sync_this_font
  # ========================================================================= #
  def sync_this_font(i)
    @header_widget.font = i
    @entry.font = i
    @header_widget.make_bold if make_the_header_widget_bold?
    @entry.make_bold if IS_THE_MAIN_ENTRY_BOLD
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    super()
    set_value(0) # Set a default value.
  end

  # ========================================================================= #
  # === Gtk::FourComponentsEntry.run
  # ========================================================================= #
  def self.run(
      i = ARGV,
      &block
    )
    require 'gtk_paradise/run'
    _ = ::Gtk::FourComponentsEntry.new(i, &block)
    r = ::Gtk.run { :auto_accel }
    r << _
    r.automatic
    _.set_parent_widget(r)
    r.top_left_then_run
  end

end; end

if __FILE__ == $PROGRAM_NAME
  Gtk::FourComponentsEntry.run {{
    header_text: 'n arrows',
    bold_text:   true
  }}
end # four_components_entry