#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::GidnightCommander
#
# Gtk::GidnightCommander tries to be clone of the old MC commander.
# =========================================================================== #
# require 'gtk_paradise/widgets/gtk3/gidnight_commander/gidnight_commander.rb'
# Gtk::GidnightCommander.new(ARGV)
# =========================================================================== #
require 'gtk_paradise/require_gtk3'

module Gtk

class GidnightCommander < ::Gtk::Box # === Gtk::GidnightCommander 

  require 'gtk_paradise/widgets/gtk3/directory_content_widget/directory_content_widget.rb'

  include ::Gtk::BaseModule

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Gidnight Commander'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = '75% or minimum 1200px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = '55% or minimum 800px'

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :dejavu_condensed_22

  # ========================================================================= #
  # === SMALLER_FONT
  # ========================================================================= #
  SMALLER_FONT = :dejavu_condensed_15

  # ========================================================================= #
  # === START_DIR
  #
  # Designate the start-directory.
  # ========================================================================= #
  START_DIR = "#{Dir.pwd}/".squeeze('/')

  # ========================================================================= #
  # === IMG
  #
  # This constant is only useful on my home system really.
  # ========================================================================= #
  IMG = ENV['IMG'].to_s

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      commandline_arguments = ARGV,
      run_already           = true
    )
    super(:vertical)
    reset
    set_commandline_arguments(
      commandline_arguments
    )
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    # ======================================================================= #
    # === Set the title, width, height and the font in use.
    # ======================================================================= #
    title_width_height_font(TITLE, WIDTH, HEIGHT, USE_THIS_FONT)
    use_gtk_paradise_project_css_file 
    infer_the_size_automatically
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    2
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    0
  end

  # ========================================================================= #
  # === remove_file
  #
  # Pops up a Dialog.
  # ========================================================================= #
  def remove_file(
      this_file = left_view_selection?
    )
    this_file = this_file.dup if this_file.frozen?
    this_file.squeeze!('/') if this_file.include? '//'
    @dialog = gtk_dialog
    label = gtk_label('Really delete this file? '+"\n"+this_file)
    label.show
    @dialog.vbox.pack_start(label, true, true, 20)
    @dialog.add_button('Delete', ::Gtk::Dialog::RESPONSE_OK)
    @dialog.add_button(::Gtk::Stock::CANCEL, Gtk::Dialog::RESPONSE_CANCEL)
    @dialog.set_default_response(::Gtk::Dialog::RESPONSE_CANCEL)
    @dialog.signal_connect(:response) { |widget, response|
      case response
      # ===================================================================== #
      # === ok-response
      # ===================================================================== #
      when ::Gtk::Dialog::RESPONSE_OK
        e 'Now removing this file: '+this_file
        FileUtils::Verbose.rm(this_file)
        reload_tree_view # reload so we get actual file listing again.
        @dialog.destroy
      # ===================================================================== #
      # === cancel-response
      # ===================================================================== #
      when ::Gtk::Dialog::RESPONSE_CANCEL
        e 'Okay, not deleting.'
        @dialog.destroy
      end
    }
    @dialog.show_all
    @dialog.run
  end

  # ========================================================================= #
  # === copy_from_to
  # ========================================================================= #
  def copy_from_to(
      from, to
    )
    from = from.to_s.dup.strip
    begin
      e "Coping #{from} to #{to}."
      FileUtils.cp(from, to)
    rescue Exception => error
      pp error
    end
  end

  # ========================================================================= #
  # === create_the_header_bar
  # ========================================================================= #
  def create_the_header_bar
    # ======================================================================= #
    # === @top_bar
    # ======================================================================= #
    @top_bar = return_top_bar
    @top_bar.set_main_title(TITLE)
    @top_bar.show_close_button = true
    @top_bar.spacing = 1
    event_box_open_image = gtk_eventbox(image_document_open)
    event_box_open_image.hint = 'Click this button to open '\
      'the selected file via the editor bluefish.'
    event_box_open_image.on_clicked {
      do_open_the_selected_entry_via_the_editor
    }
    @top_bar.add(event_box_open_image)
  end

  # ========================================================================= #
  # === create_entries                               (entries tag, entry tag)
  # ========================================================================= #
  def create_entries
    # ======================================================================= #
    # === @entry_left
    # ======================================================================= #
    @entry_left = entry
    @entry_left.very_light_yellowish_background
    @entry_left.allow_enter  # We want the enter key to work.
    @entry_left.bblack1
    @entry_left.set_text(START_DIR)
    @entry_left.use_this_font = SMALLER_FONT
    
    # ======================================================================= #
    # === @entry_right
    # ======================================================================= #
    @entry_right = entry
    @entry_right.very_light_yellowish_background
    @entry_right.allow_enter # We want the enter key to work.
    @entry_right.bblack1
    @entry_right.set_text(START_DIR)
    @entry_right.use_this_font = SMALLER_FONT
    @entry_left.on_enter {
      cd(@entry_left.text?, :left)
    }
    @entry_right.on_enter {
      cd(@entry_right.text?, :right)
    }
  end

  # ========================================================================= #
  # === change_directory                                             (cd tag)
  #
  # The second argument to this method may have the values :left or :right.
  # Any other argument is not handled by this method.
  # ========================================================================= #
  def change_directory(
      to_this_directory      = return_pwd,
      triggered_by_this_side = :right
    )
    if to_this_directory.is_a? Array
      # ===================================================================== #
      # We can not handle Arrays, so just use the first element instead.
      # ===================================================================== #
      to_this_directory = to_this_directory.first
    end
    to_this_directory = to_this_directory.dup if to_this_directory.frozen?
    to_this_directory.squeeze!('/')
    case to_this_directory
    when '..' # Obtain the absolute path in this case.
      to_this_directory = File.absolute_path(File.dirname(to_this_directory))+'/'
    end
    if File.directory? to_this_directory
      Dir.chdir(to_this_directory)
    else # else it is assumed that the directory does not exist.
      e "The target at #{sfancy(to_this_directory)} does not "\
        "exist, thus we can not cd into it."
    end
    case triggered_by_this_side
    # ======================================================================= #
    # === :left
    # ======================================================================= #
    when :left
      @entry_left.set_text(return_pwd)
      @directory_widget_left.set_pwd(return_pwd)
    # ======================================================================= #
    # === :right
    # ======================================================================= #
    when :right
      @entry_right.set_text(return_pwd)
      @directory_widget_right.set_pwd(return_pwd)
    end
  end; alias cd change_directory # === cd

  # ========================================================================= #
  # === main_action?
  # ========================================================================= #
  def main_action?
    @combo_box_file_actions.active_text?
  end

  # ========================================================================= #
  # === do_open_the_selected_entry_via_the_editor
  # ========================================================================= #
  def do_open_the_selected_entry_via_the_editor
    _ = left_view_selection?
    esystem "bluefish #{_} &"
  end; alias open_file_with_editor do_open_the_selected_entry_via_the_editor # === open_file_with_editor

  # ========================================================================= #
  # === right_view_selection?
  # ========================================================================= #
  def right_view_selection?
    @directory_widget_right.selection?.to_s
  end

  # ========================================================================= #
  # === left_view_selection?
  # ========================================================================= #
  def left_view_selection?
    @directory_widget_left.selection?.to_s
  end

  # ========================================================================= #
  # === left_directory_content?
  # ========================================================================= #
  def left_directory_content?
    @directory_widget_left
  end

  # ========================================================================= #
  # === create_the_combo_box_file_actions
  # ========================================================================= #
  def create_the_combo_box_file_actions
    @combo_box_file_actions = gtk_combo_box_text
    @combo_box_file_actions.bblack1
    # ======================================================================= #
    # Append our commands:
    # ======================================================================= #
    %w(
      cp
      mv
      rm
      cd
    ).each { |val| 
      @combo_box_file_actions.append_text(val)
    }
    @combo_box_file_actions.active = 0
    @combo_box_file_actions.disallow_resizing
    @combo_box_file_actions.width_height(120, 40)
  end

  # ========================================================================= #
  # === favicon?
  # ========================================================================= #
  def favicon?
    ::Gtk.project_base_directory?+'images/misc/RMC.jpg'
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    abort_on_exception
    minimal(@top_bar, 0)
    text = text('Gidnight Commander allows the user to '\
           'quickly copy files or directories via a GUI.')
    text.use_this_font(SMALLER_FONT)
    text.center
    minimal(text, 2)
    minimal(@combo_box_file_actions, 2)
    hbox = gtk_hbox
    hbox.maximal(@entry_left,  1)
    hbox.maximal(@entry_right, 1)
    minimal(hbox)
    hbox = gtk_hbox
    hbox.minimal(top_to_bottom_separator, 1)
    hbox.maximal(@scrolled_window_right,  1)
    draggable = gtk_drag_top_to_bottom(
      @scrolled_window_left,
      hbox
    )
    draggable.position = 845
    minimal(draggable)
    # ======================================================================= #
    # This is the left arrow.
    # ======================================================================= #
    @arrow_left  = event_box(gtk_label_arrow_left)
    @arrow_left.hint = 'This will do the action specified '\
      'above, onto the left side of the widget.'
    @arrow_left.on_clicked {
      main_action = main_action?
      case main_action
      # ===================================================================== #
      # === cd
      # ===================================================================== #
      when 'cd'
        to = left_directory_content?.cd_then_update
        cd(to)
      # ===================================================================== #
      # === cp
      # ===================================================================== #
      when 'cp'
        from = File.absolute_path(@directory_widget_right.selected?.to_s)
        to   = @entry_left.text?
        copy_from_to(from, to)
      else
        e 'Unhandled action: '+main_action
      end
    }
    # ======================================================================= #
    # This is the arrow that points right.
    # ======================================================================= #
    @arrow_right = event_box(gtk_label_arrow_right)
    @arrow_right.hint = 'This will do the action specified '\
      'above, onto the right side of the widget.'
    @arrow_right.on_clicked {
      case main_action?
      when 'cp'
        from = File.absolute_path(@directory_widget_left.selected?)
        to   = @entry_right.text?
        copy_from_to(from, to)
      end
    }
    hbox = gtk_hbox
    hbox.minimal(@arrow_left)
    hbox.minimal(@arrow_right)
    alignment = gtk_alignment(:center)
    alignment.add(hbox)
    minimal(alignment)
  end

  # ========================================================================= #
  # === create_scrolled_windows
  # ========================================================================= #
  def create_scrolled_windows

    # ======================================================================= #
    # === @directory_widget_left
    # ======================================================================= #
    @directory_widget_left = return_directory_widget
    @directory_widget_left.use_this_font = main_font?
    @scrolled_window_left = @directory_widget_left
    @scrolled_window_left.bblack1

    # ======================================================================= #
    # === @directory_widget_right
    # ======================================================================= #
    @directory_widget_right = return_directory_widget
    @directory_widget_right.use_this_font = main_font?
    @scrolled_window_right = @directory_widget_right
    @scrolled_window_right.bblack1
  end

  # ========================================================================= #
  # === create_skeleton                            (create tag, skeleton tag)
  # ========================================================================= #
  def create_skeleton
    create_the_header_bar
    create_the_combo_box_file_actions
    create_scrolled_windows
    create_entries
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    super()
  end

  # ========================================================================= #
  # === Gtk::GidnightCommander.run
  # ========================================================================= #
  def self.run(
      i = ARGV
    )
    require 'gtk_paradise/run'
    _ = ::Gtk::GidnightCommander.new(i)
    r = ::Gtk.run
    r.set_border_width(0)
    r.favicon = :RMC
    r << _
    r.automatic_size_then_automatic_title
    r.enable_quick_exit
    r.top_left_then_run
  end

end; end

if __FILE__ == $PROGRAM_NAME
  Gtk::GidnightCommander.run
end # gidnightcommander