#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::MyImageMenuItem
#
# A specialized Menuitem with a nice little icon to the left. (Context Menu)
#
# To use it, an example follows:
# 
#   @menu = Gtk::Menu.new
#   pad_to_this_amount = 26
#   @item1 = MyImageMenuItem.new('Paste',nil,pad_to_this_amount-my_label.size+4)
#   @item1.signal_connect(:activate) {  }
#   @menu.append(@item1)
#
# Since as of November 2020 the following variant can also be used instead:
#
#   item = ::Gtk::MyImageMenuItem.new('foo',nil,0)
#   item = image_menu_item('foo',nil,0)
#
# =========================================================================== #
# require 'gtk_paradise/widgets/gtk3/my_image_menu_item/my_image_menu_item.rb'
# Gtk::MyImageMenuItem.run
# =========================================================================== #
require 'gtk_paradise/require_gtk3'

module Gtk

class MyImageMenuItem < ::Gtk::Box # === Gtk::MyImageMenuItem 

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'My image menu item'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = '25% or minimum 200px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = '15% or minimum 100px'

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :dejavu_condensed_20

  # ========================================================================= #
  # === initialize
  #
  # img_location gives the path of the image.
  #
  # The second argument specifies which image to use. More specifically
  # it assumes that you provide the path to a local image.
  #
  # By default it will be a small dot.
  # ========================================================================= # 
  def initialize(
      text_to_display = 'Play Video', 
      img_location    = nil,
      n_pad           = 0
    ) # default: add 0 spaces to our text
    super(:horizontal)
    reset
    set_image_location(img_location)
    if text_to_display.is_a? Array
      text_to_display = text_to_display.join(' ').strip
    end
    @text_to_display = text_to_display
    @text_to_display = pad_to(n_pad)
    image = gtk_image
    if @image_location
      image.set_file(@image_location) if File.exist?(@image_location)
    end
    @menu_item = ::Gtk::MenuItem.new(label: @text_to_display)
    hbox = gtk_hbox
    hbox.minimal(image)
    hbox.minimal(@menu_item)
    add(hbox)
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    # ======================================================================= #
    # === Set the title, width, height and the font in use.
    # ======================================================================= #
    title_width_height_font(TITLE, WIDTH, HEIGHT, USE_THIS_FONT)
    use_gtk_paradise_project_css_file 
    infer_the_size_automatically
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    4
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    2
  end

  # ========================================================================= #
  # === set_image_location
  # ========================================================================= #
  def set_image_location(
      i = nil
    )
    if i.is_a? Array
      i = i.first
    end
    case i
    when nil # default is to show the mini dot
      i = '/home/x/data/images/STD/MINI_DOT.png'
    end
    case i # now, h andle special symbols here.
    # ======================================================================= #
    # === :quit
    # ======================================================================= #
    when :quit
      i = '/home/x/data/images/STD/SQUARE_RED.png'
    # ======================================================================= #
    # === :open
    # ======================================================================= #
    when :open,
         :open_file
      i = 'gtk-open'.to_sym #'/home/x/DATA/images/GUI/GTK/GTK_OPEN2.png'
    end
    @image_location = i
  end; alias set_img_location set_image_location # === set_img_location

  # ========================================================================= #
  # === pad_to
  #
  # Just padd our string a bit
  # ========================================================================= #
  def pad_to(n_times = 0)
    pad = ''.dup
    pad = ' ' * n_times unless n_times == 0
    return @text_to_display+pad
  end

  # ========================================================================= #
  # === Gtk::MyImageMenuItem.run
  # ========================================================================= #
  def self.run(
      i = ARGV
    )
    require 'gtk_paradise/run'
    _ = Gtk::MyImageMenuItem.new(i)
    r = ::Gtk.run
    r << _
    r.automatic_size_then_automatic_title
    r.top_left_then_run
  end

end; end

if __FILE__ == $PROGRAM_NAME
  Gtk::MyImageMenuItem.run
end