#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::NonNumbersOnly
#
# Gtk::NonNumbersOnly is a specialized Gtk::Entry, that can be restricted
# so that only non-numbers, such as A, B, C and so forth, can be entered
# by the user.
#
# Additionally the '.' dot character is also allowed, even though it is
# not a number in itself - but it is often used for floats, such as "2.3"
# or "-4.1".
# =========================================================================== #
# require 'gtk_paradise/gui/gtk3/non_numbers_only/non_numbers_only.rb'
# Gtk::NonNumbersOnly.run
# =========================================================================== #
require 'gtk_paradise/require_gtk3'

module Gtk

class NonNumbersOnly < ::Gtk::Entry # === Gtk::NonNumbersOnly 

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Non Numbers Only'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = '14% or minimum 14px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = '10% or minimum 60px'

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :dejavu_condensed_16

  # ========================================================================= #
  # === initialize
  #
  # Initialize a new Gtk::NonNumbersOnly object.
  # ========================================================================= #
  def initialize(
      commandline_arguments = ARGV,
      run_already           = true
    )
    super()
    reset
    register_the_signals
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    # ======================================================================= #
    # === Set the title, width, height and the font in use.
    # ======================================================================= #
    title_width_height_font(TITLE, WIDTH, HEIGHT, USE_THIS_FONT)
    use_gtk_paradise_project_css_file 
    infer_the_size_automatically
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    0
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    0
  end

  # ========================================================================= #
  # === register_the_signals
  # ========================================================================= #
  def register_the_signals
    signal_connect(:insert_text) {|widget, string| # string contains the string
      signal_emit_stop :insert_text if is_a_valid_number?(string)
    }
  end

  # ========================================================================= #
  # === is_a_valid_number?
  #
  # Checks if the input is a valid number.
  # ========================================================================= #
  def is_a_valid_number?(i)
    i =~ /[0-9]/
  end; private :is_a_valid_number?

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    create_skeleton_then_connect_skeleton
  end

  # ========================================================================= #
  # === Gtk::NonNumbersOnly.run
  # ========================================================================= #
  def self.run(
      i = ARGV
    )
    require 'gtk_paradise/run'
    _ = ::Gtk::NonNumbersOnly.new
    r = ::Gtk.run
    r << _
    r.automatic_size_then_automatic_title
    r.top_left_then_run
  end

end; end

if __FILE__ == $PROGRAM_NAME
  Gtk::NonNumbersOnly.run
end