#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::WidgetSwitcher
#
# This class can be used to switch between two widgets, arbitrarily called
# widget1 and widget2.
# =========================================================================== #
# require 'gtk_paradise/widgets/gtk3/widget_switcher/widget_switcher.rb'
# Gtk::WidgetSwitcher.run
# =========================================================================== #
require 'gtk_paradise/require_gtk3'

module Gtk

class WidgetSwitcher < ::Gtk::BaseModuleBox # === Gtk::WidgetSwitcher 

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Widget Switcher'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = '10% or minimum 100px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = '10% or minimum 100px'

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :dejavu_condensed_20

  # ========================================================================= #
  # === SMALLER_FONT
  # ========================================================================= #
  SMALLER_FONT =  :dejavu_condensed_18

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      commandline_arguments = ARGV,
      run_already           = true
    )
    super(:vertical)
    reset
    set_commandline_arguments(
      commandline_arguments
    )
    if commandline_arguments and commandline_arguments.is_a?(Array)
      set_widget1(commandline_arguments[0])
      set_widget2(commandline_arguments[1])
    end
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    # ======================================================================= #
    # === Set the title, width, height and the font in use.
    # ======================================================================= #
    title_width_height_font(TITLE, WIDTH, HEIGHT, USE_THIS_FONT)
    use_gtk_paradise_project_css_file 
    infer_the_size_automatically
    # ======================================================================= #
    # === @widget1
    # ======================================================================= #
    set_widget1(nil)
    # ======================================================================= #
    # === @widget2
    # ======================================================================= #
    set_widget2(nil)
    # ======================================================================= #
    # === @has_been_clicked_already
    # ======================================================================= #
    @has_been_clicked_already = false
  end

  # ========================================================================= #
  # === has_been_clicked_already?
  # ========================================================================= #
  def has_been_clicked_already?
    @has_been_clicked_already
  end

  # ========================================================================= #
  # === widget1?
  # ========================================================================= #
  def widget1?
    @widget1
  end; alias widget1 widget1? # === widget1

  # ========================================================================= #
  # === widget2?
  # ========================================================================= #
  def widget2?
    @widget2
  end; alias widget2 widget2? # === widget2

  # ========================================================================= #
  # === smaller_font?
  # ========================================================================= #
  def smaller_font?
    SMALLER_FONT
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    0
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    0
  end

  # ========================================================================= #
  # === create_skeleton                            (create tag, skeleton tag)
  # ========================================================================= #
  def create_skeleton
    @event_box = create_event_box
  end

  # ========================================================================= #
  # === event_box?
  # ========================================================================= #
  def event_box?
    @event_box
  end

  # ========================================================================= #
  # === connect_the_event_box_to_widget1
  # ========================================================================= #
  def connect_the_event_box_to_widget1
    @event_box.add(@widget1)
  end

  # ========================================================================= #
  # === let_the_event_box_respond_to_on_clicked_events
  # ========================================================================= #
  def let_the_event_box_respond_to_on_clicked_events
    @event_box.on_clicked {
      transition_into_widget2
    }
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    let_the_event_box_respond_to_on_clicked_events
    abort_on_exception
    connect_the_event_box_to_widget1
    minimal(@event_box)
  end

  # ========================================================================= #
  # === widget1=
  # ========================================================================= #
  def widget1=(i)
    i.hcenter if i
    @widget1 = i
  end; alias set_widget1 widget1= # === set_widget1

  # ========================================================================= #
  # === widget2=
  # ========================================================================= #
  def widget2=(i)
    @widget2 = i
  end; alias set_widget2 widget2= # === set_widget2

  # ========================================================================= #
  # === transition_into_widget2                                 (clicked tag)
  # ========================================================================= #
  def transition_into_widget2
    if @has_been_clicked_already
      # Pass through in this case.
    else
      remove(@event_box)
      maximal(@widget2)
      @widget2.show_all
      @has_been_clicked_already = true # Keep track of whether it was clicked already.
    end
  end; alias do_click                                             transition_into_widget2 # === do_click
       alias do_perform_actions_associated_with_the_clicked_event transition_into_widget2 # === do_perform_actions_associated_with_the_clicked_event

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    super()
  end

  # ========================================================================= #
  # === Gtk::WidgetSwitcher.run
  #
  # Pass the two widgets into the method here.
  # ========================================================================= #
  def self.run(
      i = ARGV
    )
    require 'gtk_paradise/run'
    _ = ::Gtk::WidgetSwitcher.new(i)
    r = ::Gtk.run
    r << _
    _.set_parent_widget(r)
    r.automatic_size_then_automatic_title
    r.enable_quick_exit
    r.top_left_then_run
  end

end; end

if __FILE__ == $PROGRAM_NAME
  alias e puts
  widget1 = Gtk.text('Some introduction.')
  widget2 = Gtk.button('Click this button please.')
  widget2.on_clicked {
    e 'The button was clicked.'
  }
  Gtk::WidgetSwitcher.run([widget1, widget2])
end