#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::Wiki
# =========================================================================== #
# require 'gtk_paradise/widgets/gtk3/wiki/wiki.rb'
# Gtk::Wiki.run
# =========================================================================== #
require 'gtk_paradise/require_gtk3'

module Gtk

class Wiki < ::Gtk::BaseModuleBox # === Gtk::Wiki 

  # ========================================================================= #
  # === TITLE
  #
  # This was originally named "Adamant-Wiki" before I rewrote the old code
  # in 2021 or so.
  # ========================================================================= #
  TITLE = 'Wiki'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = '75% or minimum 800px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = '45% or minimum 500px'

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :dejavu_condensed_20

  # ========================================================================= #
  # === SMALLER_FONT
  # ========================================================================= #
  SMALLER_FONT =  :dejavu_condensed_18

  # ========================================================================= #
  # === DEFAULT_YAML_FILE_TO_USE
  # ========================================================================= #
  DEFAULT_YAML_FILE_TO_USE =
    '/Depot/j/wiki.yml'

  # ========================================================================= #
  # === HASH_DESIGNATED_KEY_COMBINATIONS
  # ========================================================================= #
  HASH_DESIGNATED_KEY_COMBINATIONS = {
    'alt+1': 'button_number(1)',
    'alt+2': 'button_number(2)'
  }

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      commandline_arguments = ARGV,
      run_already           = true
    )
    super(:vertical)
    reset
    set_commandline_arguments(
      commandline_arguments
    )
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    # ======================================================================= #
    # === Set the title, width, height and the font in use.
    # ======================================================================= #
    title_width_height_font(TITLE, WIDTH, HEIGHT, USE_THIS_FONT)
    use_gtk_paradise_project_css_file  
    infer_the_size_automatically
    # ======================================================================= #
    # === @yaml_file_to_use
    # ======================================================================= #
    @yaml_file_to_use = DEFAULT_YAML_FILE_TO_USE
    unless File.directory?(File.dirname(@yaml_file_to_use))
      mkdir(File.dirname(@yaml_file_to_use))
    end
    # ======================================================================= #
    # === @content_of_the_wiki
    # ======================================================================= #
    @content_of_the_wiki = {}
    # ======================================================================= #
    # Load up any existing .yml file next:
    # ======================================================================= #
    if File.exist? @yaml_file_to_use
      dataset = load_this_yaml_file(@yaml_file_to_use)
      @content_of_the_wiki.update(dataset)
    end
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    6
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    4
  end

  # ========================================================================= #
  # === text_view?
  # ========================================================================= #
  def text_view?
    @main_text_view
  end

  # ========================================================================= #
  # === set_textview_content
  # ========================================================================= #
  def set_textview_content(new_content)
    text_view = text_view?
    text_view.set_text(new_content.to_s)
    # text_view.do_markify # Always add markup styling.
    # ^^^ this currently does not work
  end

  # ========================================================================= #
  # === do_clear_the_textview
  # ========================================================================= #
  def do_clear_the_textview
    text_view.set_text('')
  end

  # ========================================================================= #
  # === consider_displaying_the_content_of_the_wiki
  # ========================================================================= #
  def consider_displaying_the_content_of_the_wiki
    _ = main_hash?.map{|key, value| "#{key}\n#{value}" }.join("\n")
    set_textview_content(_)
  end

  # ========================================================================= #
  # === create_the_buttons                          (buttons tag, button tag)
  # ========================================================================= #
  def create_the_buttons
    # ======================================================================= #
    # === @save_button
    # ======================================================================= #
    @save_button = button('_Save the content')
    @save_button.on_clicked {
      do_save_the_current_page
    }
    # ======================================================================= #
    # === @button_load_the_page
    # ======================================================================= #
    @button_load_the_page = button('_Load the page')
    @button_load_the_page.hint = 'This will attempt to '\
      'load the assigned page from the entry defined above.'
    @button_load_the_page.on_clicked {
      do_load_the_new_page
    }
    # ======================================================================= #
    # === @button_clear_the_page
    # ======================================================================= #
    @button_clear_the_page = button('_Clear the page')
    @button_clear_the_page.on_clicked {
      do_clear_the_textview
    }
  end

  # ========================================================================= #
  # === context_menu?
  # ========================================================================= #
  def context_menu?
    @context_menu
  end

  # ========================================================================= #
  # === create_the_context_menu
  #
  # This is the context-menu which will appear on a right-mouse
  # click event.
  # ========================================================================= #
  def create_the_context_menu
    # ======================================================================= #
    # === @context_menu
    # ======================================================================= #
    @context_menu = create_context_menu(self) {{
      actions: {
        'Debug':  :do_debug
      }
    }}
    @context_menu.make_bold
    @context_menu.set_font(:hack_18)
  end

  # ========================================================================= #
  # === create_skeleton                            (create tag, skeleton tag)
  # ========================================================================= #
  def create_skeleton
    create_the_entries # Call this one first.
    create_the_buttons
    # ======================================================================= #
    # === @main_text_view
    # ======================================================================= #
    @main_text_view = gtk_text_view
    @main_text_view.set_left_margin(8)
    @main_text_view.set_right_margin(8)
    @main_text_view.set_top_margin(8)
    @main_text_view.set_bottom_margin(8)

    # ======================================================================= #
    # === @scrolled_window
    # ======================================================================= #
    @scrolled_window = scrolled_window(@main_text_view)
    @scrolled_window.width_height(520, 420)
    @scrolled_window.clear_background
    @scrolled_window.bblack2
    @scrolled_window.pad8px
    @scrolled_window.set_border_width(6)
    create_the_context_menu
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    abort_on_exception
    hbox_top = gtk_hbox
    if File.exist? base_dir?+'images/misc/wiki_logo.png'
      hbox_top.minimal(gtk_image(base_dir?+'images/misc/wiki_logo.png'), 5)
    end
    hbox_top.minimal(gtk_left_aligned_label('Wiki'), 5)
    minimal(hbox_top, 2)
    
    minimal(@scrolled_window, 2)
    hbox1 = gtk_hbox
    text1 = bold_text('Name of the article: ')
    text1.darkblue
    hbox1 << text1
    hbox1 << @entry_name_of_the_article
    hbox1.use_this_font = smaller_font?
    minimal(hbox1, 5)

    hbox2 = create_hbox
    hbox2.hcenter
    # ======================================================================= #
    # Add all buttons next:
    # ======================================================================= #
    all_buttons?.each {|this_button|
      do_style_this_button_in_a_uniform_manner(this_button)
      hbox2.minimal(this_button, 2)
    }
    debug_button = button('_Debug')
    debug_button.on_clicked {
      do_debug
    }
    do_style_this_button_in_a_uniform_manner(debug_button)
    hbox2.minimal(debug_button, 2)
    minimal(hbox2)
  end

  # ========================================================================= #
  # === do_debug
  # ========================================================================= #
  def do_debug
    e tomato('Debugging: showing the content of the wiki next.')
    e
    pp @content_of_the_wiki
    e
  end

  # ========================================================================= #
  # === project_name?
  # ========================================================================= #
  def project_name?
    @entry_name_of_the_article.text?
  end

  # ========================================================================= #
  # === do_load_the_new_page
  #
  # This method will load the page at hand.
  # ========================================================================= #
  def do_load_the_new_page(
      name_of_the_article = project_name?
    )
    name_of_the_article = name_of_the_article.to_sym
    if @content_of_the_wiki.has_key? name_of_the_article
      set_textview_content(
        @content_of_the_wiki[name_of_the_article.to_sym]
      )
    else
      e 'No key called '+name_of_the_article.to_s+' found.'
    end
  end; alias try_to_load_the_page do_load_the_new_page # === try_to_load_the_page

  # ========================================================================= #
  # === smaller_font?
  # ========================================================================= #
  def smaller_font?
    SMALLER_FONT
  end

  # ========================================================================= #
  # === create_the_entries                           (entry tag, entries tag)
  # ========================================================================= #
  def create_the_entries
    # ======================================================================= #
    # === @entry_name_of_the_article
    # ======================================================================= #
    @entry_name_of_the_article = create_entry
    @entry_name_of_the_article.clear_background
    @entry_name_of_the_article.light_yellowish_background
    @entry_name_of_the_article.hint = 
      'This entry will carry the name of the page as-is.'
    @entry_name_of_the_article.bblack1
    @entry_name_of_the_article.width_height(600, 32)
    @entry_name_of_the_article.on_enter {
      try_to_load_the_page
    }
  end

  # ========================================================================= #
  # === button_number
  #
  # This method exists so that we can add some nifty keybindings.
  # ========================================================================= #
  def button_number(i = 1)
    case i.to_i
    when 1
      do_focus_on_the_main_text_view
    when 2
      do_focus_on_the_main_entry
    end
  end

  # ========================================================================= #
  # === do_focus_on_the_main_text_view
  # ========================================================================= #
  def do_focus_on_the_main_text_view
    text_view?.do_focus
  end

  # ========================================================================= #
  # === do_focus_on_the_main_entry
  # ========================================================================= #
  def do_focus_on_the_main_entry
    @entry_name_of_the_article.do_focus
  end

  # ========================================================================= #
  # === do_save_the_current_page                                   (save tag)
  #
  # This method will save the content.
  # ========================================================================= #
  def do_save_the_current_page
    name_of_the_article = @entry_name_of_the_article.text?
    if name_of_the_article.empty?
      e 'Please assign a name to the article.'
    else
      # ===================================================================== #
      # Obtain the page content next:
      # ===================================================================== #
      page_content = text_view?.text?
      @content_of_the_wiki[name_of_the_article.to_sym] = page_content
      what = YAML.dump(@content_of_the_wiki)
      into = DEFAULT_YAML_FILE_TO_USE
      e "Saving into `#{sfile(into)}`."
      write_what_into(what, into)
    end
  end; alias save_page do_save_the_current_page # === save_page

  # ========================================================================= #
  # === content_of_the_wiki?
  # ========================================================================= #
  def content_of_the_wiki?
    @content_of_the_wiki
  end; alias main_hash? content_of_the_wiki? # === main_hash?

  # ========================================================================= #
  # === do_style_this_button_in_a_uniform_manner
  # ========================================================================= #
  def do_style_this_button_in_a_uniform_manner(i)
    i.clear_background
    i.bblack2
    i.on_hover_colour(:lightgreen)
    i.pad8px
    i.use_this_font = smaller_font?
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    super()
    Thread.new {
      sleep 0.0001
      do_focus_on_the_main_entry
      consider_displaying_the_content_of_the_wiki
    }
  end

  # ========================================================================= #
  # === Gtk::Wiki.run
  # ========================================================================= #
  def self.run(
      i = ARGV
    )
    require 'gtk_paradise/run/run.rb'
    _ = ::Gtk::Wiki.new(i)
    r = ::Gtk.run
    _.set_parent_widget(r)
    r << _
    r.automatic_size_then_automatic_title
    r.add_context_menu_with_the_default_accel_group
    _.enable_these_key_combinations(HASH_DESIGNATED_KEY_COMBINATIONS)
    r.on_button_press_event { |widget, event|
      if ::Gtk.right_mouse_click?(event) # right mouse click event.
        _.context_menu?.popup_based_on_this_event(event)
      end
    }
    r.top_left_then_run
  end

end; end

if __FILE__ == $PROGRAM_NAME
  Gtk::Wiki.run
end