#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::ColourFilledWidgetModule
#
# This is a simple widget that can be "filled" with a specific colour.
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/colour_filled_widget/colour_filled_widget_module.rb'
# include ::Gtk::ColourFilledWidgetModule
# =========================================================================== #
module Gtk

module ColourFilledWidgetModule

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === MAIN_FONT_SIZE
  # ========================================================================= #
  MAIN_FONT_SIZE = 20

  # ========================================================================= #
  # === DIMENSION
  # ========================================================================= #
  DIMENSION = 85

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH  = 880

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 550

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Colour-filled Widget'

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      i           = nil,
      run_already = true
    )
    super(:vertical)
    reset
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, infer_the_namespace]
    title_height_width(TITLE, HEIGHT, WIDTH)
    # ======================================================================= #
    # === @size
    #
    # This variable sets the size of the drawing area, where the
    # colour will be applied onto.
    # ======================================================================= #
    @size = 450
    use_gtk_paradise_project_css_file
    set_font_size(MAIN_FONT_SIZE)
    infer_size_request
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    2
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    2
  end

  # ========================================================================= #
  # === Gtk::ColourFilledWidget[]
  # ========================================================================= #
  def self.[](i = ARGV)
    new
  end

  # ========================================================================= #
  # === Gtk::ColourFilledWidget.widget
  # ========================================================================= #
  def self.widget(&block)
    new(&block)
  end

  # ========================================================================= #
  # === create_widget_allowing_the_user_to_pick_a_specific_colour
  # ========================================================================= #
  def create_widget_allowing_the_user_to_pick_a_specific_colour
    @widget_allowing_the_user_to_pick_a_specific_colour = gtk_hbox
    @entry_use_this_colour_in_the_most_bottom_widget = gtk_entry
    @entry_use_this_colour_in_the_most_bottom_widget.on_enter_key_pressed {
      do_use_this_colour(
        @entry_use_this_colour_in_the_most_bottom_widget.text
      )
    }
    # === @button_use_this_colour
    @button_use_this_colour = button('_Use this assigned colour')
    @button_use_this_colour.on_hover_lightblue
    @button_use_this_colour.set_name('button1')
    @button_use_this_colour.on_clicked {
      do_use_this_colour(
        @entry_use_this_colour_in_the_most_bottom_widget.text
      )
    }
    @widget_allowing_the_user_to_pick_a_specific_colour.maximal(
      @entry_use_this_colour_in_the_most_bottom_widget
    )
    @widget_allowing_the_user_to_pick_a_specific_colour.maximal(
      @button_use_this_colour
    )
  end

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
    @label_change_this_background_colour = label
    @label_change_this_background_colour.set_size_request(400, 50)
    @label_change_this_background_colour.set_name('BG_black')
    @label_change_this_background_colour.set_text('HEY')
    @label_change_this_background_colour.foreground_colour('steelblue')
    # #BG_steelblue   { background-color: steelblue; }
    @entry_use_this_colour = gtk_entry
    @entry_use_this_colour.center
    @drawing_area = gtk_drawing_area
    @drawing_area.set_size_request(@size, @size)
    # ======================================================================= #
    # We could also modify the drawing-area here.
    #   @drawing_area.signal_connect(:expose_event) {
    #     change_colour_randomly(@drawing_area)
    #   }
    # ======================================================================= #
    button = gtk_button('_Change colour randomly')
    button.steelblue
    button.set_name('button1')
    button.on_clicked {
      do_change_the_colour_randomly
    }
    mini_vbox = gtk_vbox
    mini_vbox.maximal(
      @label_change_this_background_colour
    )
    mini_vbox.minimal(@entry_use_this_colour, 0)
    mini_vbox.maximal(button, 0)
    # ======================================================================= #
    # Use a draggable-spacer between these two elements.
    # ======================================================================= #
    _ = draggable_spacer(@drawing_area, mini_vbox)
    _.set_border_width(12)
    _.set_position(120)
    create_widget_allowing_the_user_to_pick_a_specific_colour
    @biggest_vbox = create_vbox
    @biggest_vbox.maximal(_)
    @biggest_vbox.maximal(@widget_allowing_the_user_to_pick_a_specific_colour)
  end

  # ========================================================================= #
  # === connect_skeleton
  # ========================================================================= #
  def connect_skeleton
    maximal(@biggest_vbox)
  end

  # ========================================================================= #
  # === return_random_colour
  #
  # Return a random HTML-colour via this method.
  # ========================================================================= #
  def return_random_colour
    ::Colours.random_html_colour
  end

  # ========================================================================= #
  # === change_colour_randomly
  #
  # This method can be used to change the widget-colour to a random
  # colour.
  #
  # The second argument specifies the colour that is to be used.
  # ========================================================================= #
  def change_colour_randomly(
      widget_to_change = @drawing_area,
      i                = return_random_colour,
      be_verbose       = true
    )
    case widget_to_change
    when :default
      widget_to_change = @drawing_area
    end
    e 'Using the colour '+i+' next.' if be_verbose
    if use_gtk2?
      gdkwin = widget_to_change.window
      # ===================================================================== #
      # The next line has a problem with ruby-gtk3; I have no fix for this
      # as of January 2021.
      # ===================================================================== #
      gc = ::Gdk::GC.new(gdkwin)
      begin
        gc.rgb_fg_color = Gdk::Color.parse(i)
      rescue Exception => error
        pp error
      end
      alloc = widget_to_change.allocation
      # ===================================================================== #
      # The next line will modify the colour of the widget.
      # ===================================================================== #
      gdkwin.draw_rectangle(
        gc, true, 0, 0, alloc.width, alloc.height
      )
    end
  end; alias change_colour change_colour_randomly # === change_colour

  # ========================================================================= #
  # === do_change_the_colour_randomly
  # ========================================================================= #
  def do_change_the_colour_randomly(
      use_this_colour = return_random_colour
    )
    @entry_use_this_colour.set_text(use_this_colour)
    change_colour_randomly(@drawing_area, use_this_colour)
  end

  # ========================================================================= #
  # === GtkParadise::GUI::Gtk::ColourFilledWidgetModule.run
  # ========================================================================= #
  def self.run
    require 'gtk_paradise/run'
    _ = ::Gtk::ColourFilledWidget.new(ARGV)
    r = ::Gtk.run
    r << _
    r.alt_shortcut(1, 'focus(:left)')
    r.set_size_request(_.width?, _.height?)
    r.modify_background(:normal, :white)
    r.top_left_then_run
  end

  # ========================================================================= #
  # === do_use_this_colour
  # ========================================================================= #
  def do_use_this_colour(
      use_this_colour = @entry_use_this_colour.text
    )
    @entry_use_this_colour.set_text(
      use_this_colour.to_s
    )
    change_colour(:default, use_this_colour.to_s)
    this_colour = "#BG_#{use_this_colour.to_s.downcase}"
    @label_change_this_background_colour.set_name(this_colour)
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    super()
  end

end; end