#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::FilePathWidgetModule
#
# This small widget has two components:
#
# - an arrow on the left hand side, as a button. This will be an image,
#   actually.
# - a gtk-entry component that will show the file path to a local
#   file
#
# Clicking on the arrow leads to an event where the gtk-entry
# component is assigned to the xorg-buffer. Furthermore it 
# is pasted at once.
#
# Usage example:
#
#   Gtk::FilePathWidget.new
#
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/file_path_widget/file_path_widget_module.rb'
# include ::Gtk::FilePathWidgetModule
# =========================================================================== #
module Gtk

module FilePathWidgetModule

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  begin
    require 'xorg_buffer'
  rescue LoadError; end

  # ========================================================================= #
  # === NAMESPACE
  # ========================================================================= #
  NAMESPACE = inspect

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'File Path Widget'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = 250

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 80

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      i           = ARGV,
      run_already = true
    )
    super(:horizontal)
    reset
    create_entry # This has to be called very early, unfortunately.
    set_commandline_arguments(i)
    set_this_file(i)
    run if run_already
  end

  # ========================================================================= #
  # === reset
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, NAMESPACE]
    set_use_this_font(:dejavu_condensed_25)
    # ======================================================================= #
    # === @file
    # ======================================================================= #
    @file = nil
    # ======================================================================= #
    # === @make_the_gtk_entry_bold
    # ======================================================================= #
    @make_the_gtk_entry_bold = true
    # ======================================================================= #
    # === @title
    # ======================================================================= #
    @title  = TITLE
    # ======================================================================= #
    # === @width
    # ======================================================================= #
    set_width(WIDTH)
    # ======================================================================= #
    # === @height
    # ======================================================================= #
    set_height(HEIGHT)
    use_gtk_paradise_project_css_file
    increase_font_size
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    2
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    2
  end

  # ========================================================================= #
  # === set_this_file
  # ========================================================================= #
  def set_this_file(i)
    if i.is_a? Array
      i = i.join(' ').strip
    end
    @file = i
    # ======================================================================= #
    # We always need to sync towards the main gtk-entry, so that the user
    # knows which file is the currently selected one. This, however had,
    # requires that @entry already exists.
    # ======================================================================= #
    sync_towards_the_main_entry if @file and !@file.empty?
  end

  # ========================================================================= #
  # === sync_towards_the_main_entry                                (sync tag)
  # ========================================================================= #
  def sync_towards_the_main_entry(i = @file)
    # ======================================================================= #
    # Sync towards the main-entry.
    # ======================================================================= #
    set_main_text(i) if i
  end

  # ========================================================================= #
  # === set_main_text
  # ========================================================================= #
  def set_main_text(i)
    @entry.set_text(i)
    @entry.make_bold if @make_the_gtk_entry_bold
  end; alias << set_main_text # === <<

  # ========================================================================= #
  # === create_button                               (button tag, buttons tag)
  # ========================================================================= #
  def create_button
    entry = emoji(:arrow_right)
    @button_containing_the_arrow = gtk_event_box(entry) 
    @button_containing_the_arrow.set_size_request(80, 50)
    @button_containing_the_arrow.fancy_tooltips = 'Click on this '\
      'arrow in order to assign the text on the gtk-entry to the '\
      'right hand side to the xorg-buffer.'
    @button_containing_the_arrow.on_clicked {
      do_assign_to_the_xorg_buffer
      do_focus_on_the_content_of_the_main_entry
    }
  end

  # ========================================================================= #
  # === check_for_commandline_arguments_given
  # ========================================================================= #
  def check_for_commandline_arguments_given(
      commandline_arguments = commandline_arguments?
    )
    unless commandline_arguments.empty?
      set_main_text(commandline_arguments.first.to_s)
    end
  end

  # ========================================================================= #
  # === do_assign_to_the_xorg_buffer
  # ========================================================================= #
  def do_assign_to_the_xorg_buffer(this_text = @entry.text.to_s)
    unless this_text.empty?
      XorgBuffer.copy(this_text) if Object.const_defined? :XorgBuffer
    end
  end

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    create_skeleton_then_connect_skeleton
    check_for_commandline_arguments_given
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    Thread.abort_on_exception = true
    minimal(@button_containing_the_arrow, 0)
    maximal(@entry, 0)
  end

  # ========================================================================= #
  # === create_entry                                              (entry tag)
  # ========================================================================= #
  def create_entry
    @entry = gtk_entry('')
    @entry.modify_this_font(:liberation)
  end

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
    create_button
  end

  # ========================================================================= #
  # === do_focus_on_the_content_of_the_main_entry
  # ========================================================================= #
  def do_focus_on_the_content_of_the_main_entry
    @entry.do_focus
    @entry.select_everything
  end

  # ========================================================================= #
  # === Gtk::FilePathWidgetModule.run
  # ========================================================================= #
  def self.run
    require 'gtk_paradise/run'
    file_path_widget = Gtk::FilePathWidget.new(ARGV)
    file_path_widget.set_size_request(file_path_widget.width?, file_path_widget.height?)
    r = ::Gtk.run
    r << file_path_widget
    r.set_size_request(
      file_path_widget.width?,
      file_path_widget.height?
    )
    r.automatic_title
    r.top_left_then_run
  end

end; end