#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::LspciModule
#
# Usage example:
#
#   Gtk::Lspci.new(ARGV)
#
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/lspci/lspci_module.rb'
# include ::Gtk::LspciModule
# =========================================================================== #
module Gtk

module LspciModule # === Gtk::LspciModule

  begin
    require 'colours'
    include ::Colours
    include ::Colours::E
  rescue LoadError; end

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  begin
    require 'gtksourceview3'
  rescue Exception => error
    pp error
  end

  begin
    require 'opn'
  rescue LoadError
  end

  # ========================================================================= #
  # === N
  # ========================================================================= #
  N = "\n"

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'lspci'

  # ========================================================================= #
  # === SYS_CMD
  # ========================================================================= #
  SYS_CMD = `lspci -v`

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = 1220

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 820

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :hack_16

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      i           = ARGV,
      run_already = true
    )
    super()
    reset
    set_commandline_arguments(i)
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, infer_the_namespace]
    title_width_height_font(TITLE, WIDTH, HEIGHT, USE_THIS_FONT)
    # ======================================================================= #
    # === @sys_cmd
    # ======================================================================= #
    @sys_cmd = SYS_CMD
    use_gtk_paradise_project_css_file
    infer_the_size_automatically
  end

  # ========================================================================= #
  # === create_source_buffer
  # ========================================================================= #
  def create_source_buffer
    if Gtk.const_defined? :SourceBuffer
      begin
        set_buffer ::Gtk::SourceBuffer.new
      rescue Exception => error
        pp error
        pp error.class
      end
      set_default_text_in_buffer
    else # In this case we will use TextBuffer instead.
      opn
      e 'Gtk::SourceBuffer is unavailable. We will use a '\
        'normal text-buffer widget instead.'
      @source_buffer = ::Gtk.text_buffer
      do_update_the_lspci_buffer
    end
  end

  # ========================================================================= #
  # === buffer?
  # ========================================================================= #
  def buffer?
    @source_buffer
  end

  # ========================================================================= #
  # === create_source_view
  # ========================================================================= #
  def create_source_view
    begin
      @source_view = ::Gtk::SourceView.new(buffer?)
      @source_view.set_editable(true)
      @source_view.show_line_numbers = true
      @source_view.show_line_markers = true if @source_view.respond_to? :show_line_markers
      @source_view.highlight_current_line = true
      # @source_view.smart_home_end = true
      # @source_view.left_margin = 3
      @source_view.set_editable(false)
    rescue Exception => error
      if is_on_roebe?
        opn; pp error
        opn; pp error.class
      end
      begin
        @source_view = gtk_text_view(buffer?)
      rescue ArgumentError
        set_buffer(gtk_text_buffer)
        set_default_text_in_buffer
        @source_view = gtk_text_view(buffer?)
      end
    end
  end

  # ========================================================================= #
  # === set_default_text_in_buffer
  # ========================================================================= #
  def set_default_text_in_buffer(i = @sys_cmd)
    begin
      set_text_in_buffer(i)
    rescue; end
  end

  # ========================================================================= #
  # === set_buffer
  # ========================================================================= #
  def set_buffer(i)
    @source_buffer = i
  end

  # ========================================================================= #
  # === create_scrolled_window
  # ========================================================================= #
  def create_scrolled_window
    # ======================================================================= #
    # === @scrolled_window
    # ======================================================================= #
    @scrolled_window = gtk_scrolled_window(@source_view) { :always }
    @scrolled_window.shadow_in
    @scrolled_window.modify_background(:normal, :white)
    @scrolled_window.css_class('pad4px')
  end

  # ========================================================================= #
  # === set_text_in_buffer
  #
  # This method will set the text for the lspci-buffer.
  # ========================================================================= #
  def set_text_in_buffer(i = @sys_cmd)
    # ======================================================================= #
    # Check for max length.
    # ======================================================================= #
    splitted = i.split(N)
    if splitted.any? {|entry| entry.size > 80}
      splitted.map! {|entry|
        if entry.size > 80
          entry = entry.gsub(/(.{1,80})(\s+|$)/, "\\1\n")
        end
        entry
      }
      splitted.flatten!
      i = splitted.join(N)
    end
    @source_buffer.text = i
  end

  # ======================================================================== #
  # === create_the_buttons                         (buttons tag, button tag)
  # ======================================================================== #
  def create_the_buttons
    # ====================================================================== #
    # === @button_update_the_lspci_buffer
    # ====================================================================== #
    @button_update_the_lspci_buffer = create_button('_Update')
    @button_update_the_lspci_buffer.bblack2
    @button_update_the_lspci_buffer.lightgreen
    @button_update_the_lspci_buffer.hint = 'Click on '\
      'this button to update the buffer above.'
    @button_update_the_lspci_buffer.on_clicked {
      do_update_the_lspci_buffer
    }
  end

  # ======================================================================== #
  # === create_skeleton                                         (create tag)
  # ======================================================================== #
  def create_skeleton
    create_source_buffer
    create_source_view
    create_scrolled_window
    create_the_buttons
  end

  # ========================================================================= #
  # === Lspci.width?
  # ========================================================================= #
  def self.width?
    WIDTH
  end

  # ========================================================================= #
  # === Lspci.height?
  # ========================================================================= #
  def self.height?
    HEIGHT
  end

  # ========================================================================= #
  # === favicon?
  # ========================================================================= #
  def favicon?
    ENV['IMG'].to_s+'/STD/FÄCHER.png'
  end

  # ======================================================================== #
  # === do_update_the_lspci_buffer
  #
  # This is the method that will update the lspci-buffer.
  # ======================================================================== #
  def do_update_the_lspci_buffer
    @sys_cmd = `lspci -v`
    set_text_in_buffer(@sys_cmd)
  end

  # ========================================================================= #
  # === connect_skeleton
  # ========================================================================= #
  def connect_skeleton
    vbox = gtk_vbox
    vbox.maximal(@scrolled_window)
    vbox.minimal(@button_update_the_lspci_buffer)
    add(vbox)
    show_all
  end

  # ========================================================================= #
  # === Gtk::LspciModule.run
  # ========================================================================= #
  def self.run
    require 'gtk_paradise/run'
    _ = Gtk::Lspci.new(ARGV)
    r = ::Gtk.run
    r.add_shortcut(1, 'focus_on(@entry_pid_number)', :alt)
    r.signal_connect(:delete_event) { |_widget| Gtk.main_quit }
    r.use_default_size_from_the_config_file
    r.set_background_colour :whitesmoke
    r << _
    r.top_left_then_run
  end

  # ========================================================================= #
  # === Gtk::LspciModule.widget
  # ========================================================================= #
  def self.widget(&block)
    ::Gtk::Lspci.new(&block)
  end

  # ======================================================================== #
  # === run                                                        (run tag)
  # ======================================================================== #
  def run
    super()
    show_all
  end

  # ========================================================================= #
  # === Gtk::Lspci[]
  # ========================================================================= #
  def self.[](i = ARGV)
    new
  end

end; end