#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/maximize_button/maximize_button_module.rb'
# include ::Gtk::MaximizeButtonModule
# =========================================================================== #
module Gtk

module MaximizeButtonModule

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Maximize the window'

  # ========================================================================= #
  # Use a slightly larger font.
  # ========================================================================= #
  ::Gtk::Settings.default.gtk_font_name = 'Sans 20'

  # ========================================================================= #
  # === Gtk::MaximizeButton.title?
  # ========================================================================= # 
  def self.title?
    TITLE
  end

  # ========================================================================= #
  # === initialize
  #
  # The first argument is that gtk-widget that you wish to maximize.
  # ========================================================================= #
  def initialize(
      widget      = nil,
      run_already = true
    )
    super(:vertical, 2)
    @widget = widget
    reset
    run if run_already
  end

  # ========================================================================= #
  # === reset
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    # ======================================================================= #
    # === @is_maximized
    #
    # This variable keeps track of whether we are maximized or not.
    # ======================================================================= #
    @is_maximized = false
    # ======================================================================= #
    # === @use_bold_text
    # ======================================================================= #
    @use_bold_text = true
    # ======================================================================= #
    # === @width
    # ======================================================================= #
    @width = 400
    # ======================================================================= #
    # === @height
    # ======================================================================= #
    @height = 400
    use_gtk_paradise_project_css_file
  end

  # ========================================================================= #
  # === do_maximize_the_main_window
  # ========================================================================= #
  def do_maximize_the_main_window
    # ======================================================================= #
    # Trigger the .maximize functionality next. The inversion is .unmaximize().
    # ======================================================================= #
    case @is_maximized
    when true
      if @widget
        @widget.unmaximize
        @is_maximized = false
      end
    when false
      if @widget
        @widget.maximize
        @is_maximized = true
      end
    end
  end; alias toggle_state do_maximize_the_main_window # === toggle_state

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
    @button = gtk_button('Maximize')
    @button.steelblue
    @button.fancy_tooltips =
      "Clicking on this button will\n"\
      "<b>maximize</b> or <b>unmaximize</b>\n"\
      "this widget."
    @button.make_bold if @use_bold_text
    @button.on_hover_colour { :mediumturquoise }
    @button.on_clicked {
      do_maximize_the_main_window
    }
  end

  # ========================================================================= #
  # === no_relief
  # ========================================================================= #
  def no_relief
    @button.no_relief
  end

  # ========================================================================= #
  # === connect_skeleton
  # ========================================================================= #
  def connect_skeleton
    maximal(@button, 0)
  end

  # ========================================================================= #
  # === set_parent_widget
  # ========================================================================= #
  def set_parent_widget(i)
    @widget = i
  end; alias set_parent     set_parent_widget # === set_parent
       alias set_new_parent set_parent_widget # === set_new_parent

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    create_skeleton_then_connect_skeleton
  end

  # ========================================================================= #
  # === Gtk::MaximizeButtonModule.run
  # ========================================================================= #
  def self.run
    require 'gtk_paradise/run'
    r = ::Gtk.run
    _ = Gtk::MaximizeButton.new(r)
    r.automatic_size
    r.automatic_title
    r.signal_connect(:delete_event) { |_widget| Gtk.main_quit }
    r.enable_simple_exit
    r.modify_background(:normal, :white)
    r << _
    r.top_left_then_run
  end

end; end