#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::PidDisplayerModule
#
# The purpose of this widget is to display all PIDs on the given system,
# usually on a linux host system.
#
# Written in ruby-gtk, this widget also allows the user to kill off a PID
# from a given process. The user can use combinations such as "Alt+1" to
# jump to a specific PID entry.
# =========================================================================== #
# === TODOS AND TO-IMPROVES for this class (in german, as it is ancient):
#
# -  Man soll rechte Maustaste klicken können und da ein
#    weiteres Kontextmenu bekommen.
#
# -  Farbliche Unterschiede so das man die verschiedenen User
#    leichter unterscheiden kann.
#
# -  Warum geht das nit das man nur die Prozesse erhält beim greppen
#    die man erhalten sollte ..
#
# -  Make use of "pkill" as well!
#
# Aliased to:
#
#   rpid
#
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/pid_displayer/pid_displayer_module.rb'
# include Gtk::PidDisplayerModule
# =========================================================================== #
module Gtk

module PidDisplayerModule

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Pid-Displayer'

  # ========================================================================= #
  # === SLEEP_FOR_HOW_MANY_SECONDS
  # ========================================================================= #
  SLEEP_FOR_HOW_MANY_SECONDS = 16

  # ========================================================================= #
  # === DEFAULT_PID_NUMBER
  # ========================================================================= #
  DEFAULT_PID_NUMBER = 1001

  # ========================================================================= #
  # === FONT_TO_USE
  # ========================================================================= #
  FONT_TO_USE = 'Sans 20'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = 1400

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 50

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      commandline_arguments = nil,
      run_already           = true
    )
    super(:vertical)
    reset
    set_commandline_arguments(
      commandline_arguments
    )
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, infer_the_namespace]
    title_width_height_font(TITLE, WIDTH, HEIGHT, FONT_TO_USE)
    # ======================================================================= #
    # === @processes
    # ======================================================================= #
    @processes = '' # stores our processes
    # ======================================================================= #
    # === @array_commands
    # ======================================================================= #
    @array_commands = [] # this array stores our commands
    # ======================================================================= #
    # === @n_rows
    # ======================================================================= #
    @n_rows = 0 # this will tell us how many rows we have
    use_gtk_paradise_project_css_file
  end

  # ========================================================================= #
  # === report_n_rows
  # ========================================================================= #
  def report_n_rows
    e
    efancy "#{@n_rows} Reihen."
    e
  end

  # ========================================================================= #
  # === create_buttons
  #
  # creates all buttons in use. buttons tag button tag
  # ========================================================================= #
  def create_buttons
    # ======================================================================= #
    # The Quit button.
    # ======================================================================= #
    @button_quit = button('_Quit')
    @button_quit.clear_background
    @button_quit.modify_background(:active,   :coral)
    @button_quit.modify_background(:normal,   :chocolate)
    @button_quit.modify_background(:prelight, :yellow)
    @button_quit.on_clicked { ::Gtk.main_quit }

    # ======================================================================= #
    # The kill-PID button comes next.
    # ======================================================================= #
    @button_kill_pid = button('_Kill Pid')
    @button_kill_pid.clear_background
    @button_kill_pid.hint = 'This will kill the PID (process) '\
      'via .kill_id(), and then update the listing via .update_listing().'
    @button_kill_pid.modify_background(:active,   :coral)
    @button_kill_pid.modify_background(:normal,   :chocolate)
    @button_kill_pid.modify_background(:prelight, :mediumorchid)
    @button_kill_pid.on_clicked { 
      efancy 'Killing PID Number:'
      e @entry_pid_number.text.to_i
      kill_id(@entry_pid_number.text.to_i)
      update_listing # after we killed a pid, we must update the listing again
    }

    # ======================================================================= #
    # === @button_clear_list
    # ======================================================================= #
    @button_clear_list = button('_Clear Listing')
    @button_clear_list.clear_background
    @button_clear_list.modify_background(:active,   :coral)
    @button_clear_list.modify_background(:normal,   :chocolate)
    @button_clear_list.modify_background(:prelight, :mediumorchid)
    @button_clear_list.on_clicked { 
      efancy 'Clearing list:'
      report_n_rows
      clear_liststore
    }

    # ======================================================================= #
    # === Update listing
    # ======================================================================= #
    @button_update_listing = button('_Update Listing')
    @button_update_listing.clear_background
    @button_update_listing.modify_background(:active, :coral)
    @button_update_listing.modify_background(:normal, :chocolate)
    @button_update_listing.modify_background(:prelight, :mediumorchid)
    @button_update_listing.on_clicked { 
      efancy 'Updating listing:'
      update_listing
    }
    # ======================================================================= #
    # === @button_grep
    # ======================================================================= #
    @button_grep = button('_Grep')
    @button_grep.clear_background
    @button_grep.modify_background(:active,   :coral)
    @button_grep.modify_background(:normal,   :chocolate)
    @button_grep.modify_background(:prelight, :mediumorchid)
    @button_grep.on_clicked {
      grep_for
      update_listing # after we grepped, we will update
    }
  end

  # ========================================================================= #
  # === set_geometry
  #
  # Set geometry hints.
  # ========================================================================= #
  def set_geometry
    @geometry = Gdk::Geometry.new
    min =  430
    max = 1100
    @geometry.set_min_width(min+600) if @geometry.respond_to? :set_min_width
    @geometry.set_min_height(min) if @geometry.respond_to? :set_min_height
    @geometry.set_max_width(max) if @geometry.respond_to? :set_max_width
    @geometry.set_max_height(max) if @geometry.respond_to? :set_max_height
    @geometry.set_width_inc(20) if @geometry.respond_to? :set_width_inc
    @geometry.set_height_inc(20) if @geometry.respond_to? :set_height_inc
  end

  # ========================================================================= #
  # === create_liststore
  #
  # 12 entries.
  # ========================================================================= #
  def create_liststore
    @list_store = ::Gtk::ListStore.new(* ([String] * 12) )
  end

  # ========================================================================= #
  # === connect_skeleton
  #
  # Connects for example buttons with main window.
  #
  # Does NOT cover the various Gtk::Entries.
  # ========================================================================= #
  def connect_skeleton
    @scrolled_window.add(@tree_view)
    @scrolled_window.set_size_request(1000, 500)
    # Listing. Rows kommen als letzte Params.
    @main_table.attach_defaults(@scrolled_window,0,2, 3,5)
    @v_box.add(@main_table)
    add(@v_box)
    show_all
  end

  # ========================================================================= #
  # === create_vbox
  # ========================================================================= #
  def create_vbox
    @v_box = gtk_vbox
  end

  # ========================================================================= #
  # === clear_liststore
  #
  # Used to clear our liststore entries. See for Gtk::ListStore clear.
  #
  # I believe thats not the same as emptying it.
  # clear_liststore
  # ========================================================================= #
  def clear_liststore
    @list_store.clear
  end

  # ========================================================================= #
  # === create_treeview
  #
  # Creates our treeview, by using our @list_store
  # ========================================================================= #
  def create_treeview
    # ================================================================== #
    # === @tree_view
    # ================================================================== #
    @tree_view = gtk_tree_view(@list_store)
    @tree_view.set_size_request(800, 300)
    @tree_view.resizable_headers
    @tree_view.the_rows_are_sortable
  end

  # ========================================================================= #
  # === create_renderer
  # ========================================================================= # 
  def create_renderer
    @renderer = gtk_cell_renderer_text
    @renderer.set_property('foreground', 'black')
    @renderer.set_property('background', 'white')
  end

  # ========================================================================= #
  # === create_scrolled_window
  # ========================================================================= #
  def create_scrolled_window
    @scrolled_window = gtk_scrolled_window
  end

  # ========================================================================= #
  # === populate_treeview
  #
  # Fill our treeview with entries, info retrieved by doing:
  #   ps ax
  #  
  # This only needs to be done once, on startup of the program.
  #
  #    PID TTY STAT TIME COMMAND
  #                                                                                                                          
  # ========================================================================= #
  def populate_treeview
    # @tree_view.insert_column(-1, 'PID',      @renderer,  :text =>  0)
    # @tree_view.insert_column(-1, 'TTY',      @renderer,  :text =>  1)
    # @tree_view.insert_column(-1, 'STAT',     @renderer,  :text =>  2)
    # @tree_view.insert_column(-1, 'TIME',     @renderer,  :text =>  3)
    # @tree_view.insert_column(-1, 'COMMAND',  @renderer,  :text =>  4)

    column_pid     = Gtk::TreeViewColumn.new('PID', gtk_cell_renderer_text,     { text: 0 })
    column_tty     = Gtk::TreeViewColumn.new('TTY', gtk_cell_renderer_text,     { text: 1 })
    column_stat    = Gtk::TreeViewColumn.new('STAT', gtk_cell_renderer_text,    { text: 2 })
    column_time    = Gtk::TreeViewColumn.new('TIME', gtk_cell_renderer_text,    { text: 3 })
    column_command = Gtk::TreeViewColumn.new('COMMAND', gtk_cell_renderer_text, { text: 4 })

    @tree_view.append_column(column_pid)
    @tree_view.append_column(column_tty)
    @tree_view.append_column(column_stat)
    @tree_view.append_column(column_time)
    @tree_view.append_column(column_command)
  end

  # ========================================================================= #
  # === update_listing
  #
  # Updates the listing. If no array is passed, we run 'ps aux', else we
  # treat the passed array as our field which we want to display to
  # the user.
  # ========================================================================= #
  def update_listing(use_this_array = [])
    clear_liststore # Clear liststore first.
    unless use_this_array.empty?
      clear_liststore
      @n_rows = use_this_array.size
      use_this_array.each {|process|
        iter = @list_store.append
        # Go through each process detail creating a column for it.
        process.each_with_index {|the_process, index|
          # index is zb 10, process zb "ruby myserver_control.rb start"
          iter[index] = the_process
        }
      }
    else
      # ===================================================================== #
      # if you want sort by PID.
      # ===================================================================== #
      @processes = `ps ax`.split(N)[1..-1].map { |entry| 
        # entry zb:
        # 201 ?        S<     0:00 [khubd]
        # Zerst kommt die wichtige PID
        _ = entry.strip.split(/\s+/, 5)
      }
      @n_rows = @processes.size 
      @array_commands = [] # empty it again
      # Go through each process creating a row for it.  
      @processes.each { |process|
        iter = @list_store.append
        # Go through each process detail creating a column for it.
        process.each_with_index { |the_process, index|
          iter[index] = the_process
        }
      }
    end
  end

  # ========================================================================= #
  # === create_entries
  #
  # entries tag. Handles all various entries.
  # ========================================================================= #
  def create_entries
    @entry_pid_number = gtk_entry
    @entry_pid_number.set_max_length(20) # How long can a pid be ? hmm
    # @entry_url.text # zum abfragen des inhalts
    @entry_pid_number.set_text(DEFAULT_PID_NUMBER.to_s)
    @entry_pid_number.on_button_press_event { |widget, event|
      @entry_pid_number.set_focus(true)
      @entry_pid_number.select_region(0, -1)
    }
    @entry_pid_number.on_key_press_event { |widget, event|
      ensure_that_only_numbers_are_input(event)
    }
    # ======================================================================= #
    # Next, we add the entry which allows us to choose a PID.
    # ======================================================================= #
    @entry_choose_pid = gtk_entry('ruby') {{ max_length: 50 }}
    #@entry_choose_pid.set_text('ruby') # #greppe nach ruby
  end

  # ========================================================================= #
  # === ensure_that_only_numbers_are_input
  #
  # We may only input numbers here.
  # ========================================================================= #
  def ensure_that_only_numbers_are_input(event)
    _ = ''
    case event.class
    when Gdk::EventKey
      my_key = Gdk::Keyval.to_name(event.keyval)
    else
      my_key = event
    end
    old_text = @entry_pid_number.text.dup # stores old text
    if my_key =~ /^\d+/
      _ << my_key
    end
    case my_key
    when 'BackSpace','Delete'
      @entry_pid_number.set_text('')
    else
      @entry_pid_number.set_text(old_text << _)
    end
  end
  
  # ========================================================================= #
  # === grepping_for?
  # ========================================================================= #
  def grepping_for?
    @entry_choose_pid.text
  end

  # ========================================================================= #
  # === grep_for
  #
  # This greps for all names; note that @processes must have been set
  # properly.
  # ========================================================================= # 
  def grep_for(
      input = grepping_for?
    )
    pp @processes
    e "Grepping for #{grepping_for?}"
    @processes.each { |entry|
      if entry.last.include? input
        @array_commands << entry
        e "Yes, #{entry.last} does include #{::Colours.steelblue(input)}."
      end
    }
    unless @array_commands.empty?
      update_listing(@array_commands) 
    end
    report_n_rows
  end

  # ========================================================================= #
  # === create_and_connect_tables
  #
  # creates and handles all of our tables. table tag
  # ========================================================================= #
  def create_and_connect_tables
    @main_table = gtk_table(5,2,false) # 4 reihen

    h_box_left_top = gtk_hbox
    h_box_left_top.minimal(@button_clear_list, 2)
    h_box_left_top.maximal(@button_kill_pid,   2)
    @main_table.attach_defaults(h_box_left_top, 0,1, 0, 1)

    @h_box_right_top = gtk_hbox
    @h_box_right_top.maximal(@button_update_listing, 2)
    @h_box_right_top.maximal(@button_quit,           2)
    @main_table.attach_defaults(@h_box_right_top,1,2, 0,1)

    @main_table.attach_defaults(
      modify_label('PID to kill: ','darkblue'),0,1, 1,2
    )
    # @main_table.attach_defaults(@entry_pid_number,1,2, 1,2)
    @main_table.attach(@entry_pid_number, 1, 2, 1, 2, nil, nil, 0, 0)
    # ======================================================================= #
    # Add some tooltips to @entry_pid:number.
    # ======================================================================= #
    name_tip = 'This input field accept numbers only. The '\
               '<b>number</b> should be the PID that you '\
               'wish to kill.'
    @entry_pid_number.hint = name_tip
    hbox = modify_label('Grep for this:','darkblue')
    @h_box_small = gtk_hbox
    @h_box_small.minimal(hbox)
    @h_box_small.add(@button_grep)
    @main_table.attach_defaults(@h_box_small,     0,1, 2,3)
    @main_table.attach_defaults(@entry_choose_pid,1,2, 2,3)

    # ======================================================================= #
    # Set spacing
    # ======================================================================= #
    @main_table.set_column_spacings(3)
  end

  # ========================================================================= #
  # === focus_on
  #
  # This method specifically allows us to jump to the correct gtk-entry.
  # ========================================================================= #
  def focus_on(i)
    case i
    when '@entry_choose_pid'
      @entry_choose_pid.set_focus(true)
    when '@entry_pid_number'
      @entry_pid_number.set_focus(true)
    end
  end

  # ========================================================================= #
  # === kill_id
  #
  # Kills a specific id.
  #
  # Usage Example:
  #
  #   kill_id(22)
  #
  # ========================================================================= #
  def kill_id(which_id)
    which_id = which_id.to_s
    the_cmd  = 'kill -9 '+which_id
    esystem the_cmd
  end

  # ========================================================================= #
  # === start_thread
  #
  # Starts a thread.
  #
  # We sleep in here as well. (If you sleep in a thread, the whole program
  # isnt delayed.)
  # ========================================================================= #
  def start_thread
    efancy('Starting thread next:')
    Thread.new {
      loop {
        efancy "Updating listing now, then sleeping for "\
               "#{SLEEP_FOR_HOW_MANY_SECONDS} seconds"
        update_listing
        # Refresh process list every CONSTANT seconds.
        sleep SLEEP_FOR_HOW_MANY_SECONDS
      }
    }
  end

  # ========================================================================= #
  # === style_all_buttons_in_a_uniform_manner
  # ========================================================================= #
  def style_all_buttons_in_a_uniform_manner
    return_all_buttons.each {|this_button| this_button.bblack1 }
  end

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
    create_vbox               # creates our vbox
    create_buttons            # create our buttons
    create_liststore          # create entries for use in treeview
    create_treeview           # stores our liststore    
    create_renderer           # create our renderer
    create_scrolled_window    #
    create_entries            # all entries
    create_and_connect_tables # we want a table and will connect it
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    # ======================================================================= #
    # Determine the size:
    # ======================================================================= #
    set_size_request(width?, height?)
    set_geometry
    create_skeleton
    populate_treeview         # fill our treeview with entries.
    start_thread              # Supposed to make a thread
    connect_skeleton          # connect the skeleton parts, except for tables
    update_listing            # init it for the first time
    style_all_buttons_in_a_uniform_manner
  end

  # ========================================================================= #
  # === Gtk::PidDisplayer[]
  # ========================================================================= #
  def self.[](i = '')
    new(i)
  end

  # ========================================================================= #
  # === Gtk::PidDisplayer.run
  # ======================================================================== #
  def self.run
    require 'gtk_paradise/run.rb'
    _ = Gtk::PidDisplayer.new
    r = ::Gtk.run
    r.set_border_width(4)
    r.shortcut('alt+1') { 'focus_on(@entry_pid_number)' }
    r.shortcut('alt+2') { 'focus_on(@entry_choose_pid)' }
    r << _
    r.enable_quick_exit_then_top_left_then_run
  end

end; end

if __FILE__ == $PROGRAM_NAME
  Gtk::PidDisplayer.run
end