#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::RandomImageModule
#
# The purpose of this class is to randomly select an image from a given
# local directory. It is not that useful, but it can be used for
# demonstration-purposes.
#
# The class is deliberately working only on random images so that we
# don't have to think about how to select an image specifically, for
# the time being. At a later time this may change.
# =========================================================================== #
# Original creation date of this file: 09.12.2005
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/random_image/random_image_module.rb'
# include ::Gtk::RandomImageModule
# =========================================================================== #
module Gtk

module RandomImageModule

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  begin
    require 'open'
  rescue LoadError; end

  begin
    require 'colours'
    include ::Colours
  rescue LoadError; end

  begin
    require 'opn'
  rescue LoadError; end

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === NAMESPACE
  # ========================================================================= #
  NAMESPACE = inspect

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = 1200

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 400

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Random Image Viewer'

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      optional_file = nil,
      run_already   = true
    )
    super(:vertical)
    reset
    set_background_colour(:black)
    consider_adding_this_image(optional_file)
    determine_the_main_image
    run if run_already
  end

  # ========================================================================= #
  # === determine_the_main_image
  # ========================================================================= #
  def determine_the_main_image(i = image?)
    @main_image = gtk_image(i)
  end

  # ========================================================================= #
  # === reset
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, infer_the_namespace]
    title_width_height(TITLE, WIDTH, HEIGHT)
    # ======================================================================= #
    # === @vpaned
    # ======================================================================= #
    @vpaned = gtk_vpaned
    increase_font_size
    use_gtk_paradise_project_css_file
    infer_the_size_automatically
  end
  
  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    2
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    4
  end

  # ========================================================================= #
  # === favicon?
  # ========================================================================= #
  def favicon?
    :tabble
  end

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
    create_event_box
    create_gtk_entry_on_bottom
    create_buttons
  end

  # ========================================================================= #
  # === consider_adding_this_image
  # ========================================================================= #
  def consider_adding_this_image(i)
    unless i.nil?
      get_image(i)
    else
      get_image(:rand)
    end
  end

  # ========================================================================= #
  # === image?
  # ========================================================================= #
  def image?
    @image
  end

  # ========================================================================= #
  # === reload_new_random_image                                    (load tag)
  #
  # This method will try to reload a new image. Whenever this is triggered,
  # it also reloads the entry and sets the focus on it.
  # ========================================================================= #
  def reload_new_random_image
    @main_image.clear
    get_image(:rand)
    opn; e "The image was: #{sfile('  '+image?)}"
    @entry_image_location.set_text(image?)
    @entry_image_location.set_focus(true)
    @entry_image_location.select_region(0, -1)
    begin
      # ===================================================================== #
      # Next we instantiate a new GdkPixbuf instance.
      # ===================================================================== #
      tmp = return_pixbuf(image?)
      # tmp = Gdk::Pixbuf.new(@image, 10,10,40,40)
      # small_image = Gdk::Pixbuf.new( tmp, 0, 0, 
      #   575, 355 
      # )
      small_image = tmp.subpixbuf( 0, 0, 575, 355)
      if ::Gtk.use_gtk2?
        @main_image.set(small_image)
      else
        @main_image.set_pixbuf(small_image)
      end
      tmp = small_image = nil
      GC.start
    rescue Exception => error
      opn; e error.class
      opn; ewarn error
      opn; e @image
    end
  end

  # ========================================================================= #
  # === create_event_box
  # ========================================================================= #
  def create_event_box
    @event_box_containing_the_image = gtk_eventbox(@main_image)
    # ======================================================================= #
    # Reload a new random image upon a mouse-click event.
    # ======================================================================= #
    @event_box_containing_the_image.on_button_press_event { reload_new_random_image }
    @event_box_containing_the_image.modify_background(:normal, :black)
    @event_box_containing_the_image.set_size_request(1200, 800)
    @main_image.set_size_request(1200, 800)
  end

  # ========================================================================= #
  # === return_pixbuf
  # ========================================================================= #
  def return_pixbuf(i = image?)
    GdkPixbuf::Pixbuf.new(file: i)
  end

  # ========================================================================= #
  # === create_buttons
  # ========================================================================= #
  def create_buttons
    @button_on_the_right_side_for_loading_a_new_image = gtk_bold_button('_Load New Image')
    @button_on_the_right_side_for_loading_a_new_image.hint =
      '<span foreground="gold" weight="ultrabold">Click</span> on this '\
      'button in order to load up a new (random) image.'
    @button_on_the_right_side_for_loading_a_new_image.on_clicked { reload_new_random_image }
  end
  
  # ========================================================================= #
  # === get_image
  #
  # This is the method that will grab a random image.
  # ========================================================================= #
  def get_image(which_image)
    case which_image
    # ======================================================================= #
    # === :random
    # ======================================================================= #
    when :random,
         :rand
      @image = Dir[ENV['IMG_RPG'].to_s+'/ADOPTION/*'].sample
    else
      @image = which_image
    end
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    abort_on_exception
    scrolled_window = gtk_scrolled_window(@event_box_containing_the_image) { :always }
    add(scrolled_window)
    scrolled_window.set_size_request(1200, 800)
    mini_hbox = create_hbox
    mini_hbox.maximal(@entry_image_location)
    mini_hbox.minimal(@button_on_the_right_side_for_loading_a_new_image)
    mini_hbox.minimal(return_widget_containing_the_open_in_browser_functionality)
    add(mini_hbox)
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    super()
  end

  # ========================================================================= #
  # === GtkParadise::GUI::Gtk::RandomImageModule.run
  # ========================================================================= #
  def self.run(
      i = ARGV
    )
    _ = ::Gtk::RandomImage.new(ARGV.first)
    r = ::Gtk.run
    r << _
    r.set_size_request(_.width?, _.height?)
    r.top_left_then_run
  end

  # ========================================================================= #
  # === create_gtk_entry_on_bottom
  #
  # This is the gtk-entry for the box on the very bottom.
  # ========================================================================= #
  def create_gtk_entry_on_bottom
    # ======================================================================= #
    # === @entry_image_location
    # ======================================================================= #
    @entry_image_location = create_entry
    @entry_image_location.set_max_length(100)
    @entry_image_location.set_size_request(100, 50)
    @entry_image_location.set_text(image?)
    @entry_image_location.on_key_press_event { |widget, event|
      _ = Gdk::Keyval.to_name(event.keyval)
      case _
      when 'Up','KP_Up','Down','KP_Down',
           'Left','KP_Left','Right','KP_Right'
        reload_new_random_image
      end
    }
  end

  # ========================================================================= #
  # === return_widget_containing_the_open_in_browser_functionality
  # ========================================================================= #
  def return_widget_containing_the_open_in_browser_functionality
    _ = event_box(image_web_browser)
    _.on_clicked {
      if Object.const_defined? :Open
        Open.in_browser(@entry_image_location.text?)
      else
        e 'Please install the "open" gem first, via: gem install open'
      end
    }
    return _
  end

end; end