#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::ShowIconThemeModule
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/show_icon_theme/show_icon_theme_module.rb'
# include ::Gtk::ShowIconThemeModule
# =========================================================================== #
module Gtk

module ShowIconThemeModule # === Gtk::ShowIconThemeModule

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === NAMESPACE
  # ========================================================================= #
  NAMESPACE = inspect

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Show Icon Theme'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = '90%'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = '80%'

  # ========================================================================= #
  # === TARGET_DIRECTORY
  #
  # Keep this aimed at the most important directory.
  # ========================================================================= #
  TARGET_DIRECTORY =
    '/usr/share/icons/Adwaita/32x32/'
    # '/usr/share/icons/Papirus/32x32/'
    # '/usr/share/icons/oxygen/32x32/'

  # ========================================================================= #
  # === DEFAULT_ICON_THEME
  # ========================================================================= #
  DEFAULT_ICON_THEME = Gtk::IconTheme.default

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :hack_21

  # ========================================================================= #
  # === ALPHABET
  # ========================================================================= #
  ALPHABET = ('a' .. 'z').to_a

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      run_already = true
    )
    super(:horizontal)
    reset
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    # ======================================================================= #
    # === title, width, height and font
    # ======================================================================= #
    title_width_height_font(TITLE, WIDTH, HEIGHT, USE_THIS_FONT)
    # ======================================================================= #
    # === @array_all_entries
    # ======================================================================= #
    @array_all_entries = []
    # ======================================================================= #
    # === @target_directory
    # ======================================================================= #
    @target_directory = TARGET_DIRECTORY
    # ======================================================================= #
    # === @context_menu
    # ======================================================================= #
    @context_menu = create_context_menu(self) {{
      # n_entries: 12,
      actions: {
        'Quit':                               :exit_the_application,
        'Show Statistics on the commandline': :show_stats
      }
    }}
    @context_menu.css_class(
      'bblack3_and_padding'
    )
    use_gtk_paradise_project_css_file
    infer_the_size_automatically
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    0
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    0
  end

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
    @search_widget = gtk_vbox
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    create_skeleton
    vbox1 = gtk_vbox
    @sorted_array = Dir[@target_directory+'**/**'].sort_by {|entry|
      File.basename(entry.to_s)
    }
    # ======================================================================= #
    # Next, we have to remove entries that are essentially identical.
    # For instance, these two entries are identical:
    #
    #   "/usr/share/icons/Adwaita/32x32/actions/zoom-out.png"
    #   "/usr/share/icons/Adwaita/32x32/legacy/zoom-out.png"
    #
    # So we will weed away on the last part.
    # ======================================================================= #
    _ = []
    @sorted_array.each {|entry|
      _ << entry unless _.any? {|inner_entry|
        File.basename(inner_entry) == (File.basename(entry))
      }
    }
    @sorted_array = _
    @sorted_array.each_with_index {|this_icon, index|
      # ===================================================================== #
      # The entries may look like this:
      #
      #   /usr/share/icons/Adwaita/32x32/mimetypes/x-office-document-template.png
      #
      # ===================================================================== #
      basename = File.basename(this_icon) # Now we have: "x-office-document-template.png"
      basename = basename.sub(/#{File.extname(basename)}$/,'')
      begin
        # =================================================================== #
        # The next Array shall only be filled under the begin/rescue
        # clause, so that we can skip the count if an error occurred.
        # =================================================================== #
        @array_all_entries << basename
        icon = DEFAULT_ICON_THEME.load_icon(basename, 32, 0) # Use the window-close example.
        # image = ::Gtk::Image.new(pixbuf: icon) # Use the icon here.
        image = gtk_pixbuf(icon)
        label = gtk_label(" #{basename}")
        # Build a hbox that will be added next.
        hbox = gtk_hbox
        # =================================================================== #
        # Set up the index label next. This will simply count how many
        # entries were added so far.
        # =================================================================== #
        index_label = gtk_label(
          (index+1).to_s.rjust(3,' ')
        )
        index_label.bold
        event_box_for_the_index = gtk_event_box(index_label)
        event_box_for_the_index.on_clicked {
          set_text_for_the_search_bar.set_text(basename)
          sync_the_text_of_the_search_bar_onto_the_icon_below_the_search_bar
        }
        hbox.minimal(event_box_for_the_index, 12)
        hbox.minimal(image, 1)
        event_box_for_the_last_label = gtk_eventbox(label)
        event_box_for_the_last_label.on_clicked {
          set_text_for_the_search_bar.set_text(basename)
          sync_the_text_of_the_search_bar_onto_the_icon_below_the_search_bar
        }
        # label.is_selectable # Can not do this if we use an event box.
        hbox.minimal(event_box_for_the_last_label, 1)
        vbox1.minimal(hbox)
        vbox1.minimal(gtk_hspacer)
      rescue Gtk::IconThemeError::NotFound => error
        pp error
      end
    }
    vbox2 = create_and_return_the_search_widget
    @scrolled_window = gtk_scrolled_window(vbox1) { :always }
    @scrolled_window.set_name('scrollbar2')
    hbox = draggable_vertical_pane(@scrolled_window, vbox2)
    hbox.position = 920
    maximal(hbox)
    Thread.new {
      sleep 0.005
      main_entry?.deselect
      @label_search.do_focus
      @label_search.deselect
    }
  end

  # ========================================================================= #
  # === scroll_down_by
  # ========================================================================= #
  def scroll_down_by(
      i = 5000
    )
    vadjustment = @scrolled_window.vadjustment
    vadjustment.set_value(i) # Scroll down.
  end

  # ========================================================================= #
  # === context_menu?
  # ========================================================================= #
  def context_menu?
    @context_menu
  end

  # ========================================================================= #
  # === exit_the_application
  # ========================================================================= #
  def exit_the_application
    gtk_main_exit
  end

  # ========================================================================= #
  # === show_stats
  # ========================================================================= #
  def show_stats
    e
    e 'There are '+@sorted_array.size.to_s+' available emojis/entries.'
    e
  end

  # ========================================================================= #
  # === main_entry?
  # ========================================================================= #
  def main_entry?
    @entry_for_the_search_bar
  end

  # ========================================================================= #
  # === which_theme_is_in_use?
  #
  # This will return the current theme in use, based on the directory.
  # ========================================================================= #
  def which_theme_is_in_use?
    @target_directory.split('/')[4].to_s.downcase
  end

  # ========================================================================= #
  # === Gtk::ShowIconTheme.run
  # ========================================================================= #
  def self.run
    require 'gtk_paradise/run'
    r = ::Gtk.run
    _ = ::Gtk::ShowIconTheme.new
    r << _
    r.enable_context_menu
    r.set_my_title(
      _.title?+' ('+_.which_theme_is_in_use?+' theme is in use)'
    )
    r.automatic_size
    r.signal_connect(:button_press_event) { |widget, event|
      if ::Gtk.right_mouse_click?(event) # right mouse click event.
        _.context_menu?.popup_based_on_this_event(event)
      end
    }
    r.top_left_then_run
  end

  # ========================================================================= #
  # === create_and_return_the_search_widget
  #
  # This method will return the search-widget, which is normally on the
  # right-hand side of this widget.
  # ========================================================================= #
  def create_and_return_the_search_widget
    @label_search = left_aligned_text(' Search: ')
    @label_search.make_selectable
    @label_search.make_bold
    @label_search.hint =
      'In the entry-field below you can search for a specific name.'
    @search_widget.minimal(@label_search, 8)
    results_widget = gtk_label
    # ======================================================================= #
    # Next add the search-entry.
    # ======================================================================= #
    @entry_for_the_search_bar = create_entry
    @entry_for_the_search_bar.set_size_request(150, 50)
    @entry_for_the_search_bar.css_class('pad8px')
    @entry_for_the_search_bar.css_class('mar48px')
    @entry_for_the_search_bar.css_class('BG_light_yellow')
    @entry_for_the_search_bar.css_class('border_3px_royalblue')
    @entry_for_the_search_bar.align_to_the_middle
    @entry_for_the_search_bar.enable_search_icon
    # The user can use the "enter" key to trigger a specific search for
    # a gtk-icon, such as "batter".
    @entry_for_the_search_bar.on_enter_key_clicked {
      search_term = @entry_for_the_search_bar.text?
      array = @array_all_entries
      results = array.select {|entry|
        entry.to_s.include? search_term
      }
      if results.empty?
        results_widget.set_text('No results could be found.')
      else
        # =================================================================== #
        # In this case we found a result.
        # =================================================================== #
        results_widget.set_text(
          results.each_with_index.map {|entry, index|
            padded_index = (
              '(<b>'+(index+1).to_s+'</b>)'
            ).to_s.rjust(6)
            "#{padded_index} #{entry}"
          }.join("\n")
        )
        results_widget.enable_markup
        first_char = results.first[0,1].downcase
        position = ALPHABET.index(first_char)
        new_position = (position+1) * 1_650
        scroll_down_by(new_position)
      end
    }
    @search_widget.minimal(@entry_for_the_search_bar, 10)
    sample = File.basename(@sorted_array.sample)
    basename = sample.delete_suffix(File.extname(sample))
    icon = DEFAULT_ICON_THEME.load_icon(basename, 32, 0) # Use the window-close example.
    @widget_showing_which_icon_is_used = gdk_pixbuf(icon)
    @widget_showing_which_icon_is_used.hint = 'This area '\
      'will show which icon is currently selected as the '\
      'default.'
    @search_widget.minimal(@widget_showing_which_icon_is_used, 10)
    @search_widget.minimal(results_widget,            10)
    @entry_for_the_search_bar.ghost_text('Input your search term here')
    return @search_widget
  end

  # ========================================================================= #
  # === set_text_for_the_search_bar
  # ========================================================================= #
  def set_text_for_the_search_bar(i = '')
    @entry_for_the_search_bar.set_text(
      i.to_s
    )
  end

  # ========================================================================= #
  # === sync_the_text_of_the_search_bar_onto_the_icon_below_the_search_bar
  # ========================================================================= #
  def sync_the_text_of_the_search_bar_onto_the_icon_below_the_search_bar
    _ = @entry_for_the_search_bar.text?
    icon = DEFAULT_ICON_THEME.load_icon(_, 32, 0)
    @widget_showing_which_icon_is_used.set_pixbuf(icon)
  end

end; end