#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::UnicodeClockModule
#
# This widget will show a small unicode-clock, but without any details,
# such as the exact time. This is on purpose, at the least for the time
# being. (Optionally we could show the real time on a pop-over event.)
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/unicode_clock/unicode_clock_module.rb'
# include ::Gtk::UnicodeClockModule
# =========================================================================== #
module Gtk

module UnicodeClockModule

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === NAMESPACE
  # ========================================================================= #
  NAMESPACE = inspect

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Gtk::UnicodeClockModule'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = 1000

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 100

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      optional_file = nil,
      run_already   = true
    )
    super(:horizontal)
    reset
    run if run_already
  end

  # ========================================================================= #
  # === reset
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, NAMESPACE]
    # ======================================================================= #
    # === @current_time
    #
    # We need to store the current time on startup.
    # ======================================================================= #
    @current_time = Time.now
    set_use_this_font(:dejavu_condensed_25)
    # ======================================================================= #
    # === @title
    # ======================================================================= #
    @title  = TITLE
    # ======================================================================= #
    # === @width
    # ======================================================================= #
    set_width(WIDTH)
    # ======================================================================= #
    # === @height
    # ======================================================================= #
    set_height(HEIGHT)
    use_gtk_paradise_project_css_file
    increase_font_size
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    2
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    2
  end

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
    # ======================================================================= #
    # === @entry_showing_the_clock
    # ======================================================================= #
    @entry_showing_the_clock = gtk_label(map_half_hour_to_the_corret_clock_emoji)
    @event_box = gtk_eventbox(@entry_showing_the_clock)
    @event_box.fancy_tooltips = return_full_current_time_string
    @event_box.on_clicked {
      # ===================================================================== #
      # We set a new tooltip text here, as well as change the hour.
      # ===================================================================== #
      @event_box.fancy_tooltips = return_full_current_time_string
      @entry_showing_the_clock.set_text(map_half_hour_to_the_corret_clock_emoji)
    }
    @entry_showing_the_clock.set_font(:dejavu_condensed_50)
  end

  # ========================================================================= #
  # === return_full_current_time_string
  # ========================================================================= #
  def return_full_current_time_string
    'The current time is: <b>'+
     return_current_time_in_hh_mm_ss+'</b>'
  end

  # ========================================================================= #
  # === connect_skeleton
  # ========================================================================= #
  def connect_skeleton
    abort_on_exception
    maximal(@event_box)
  end

  # ========================================================================= #
  # === return_current_half_hour
  #
  # The "half" hour is the 24 hours notation, divided by 2.
  # ========================================================================= #
  def return_current_half_hour
    @current_time.hour / 2
  end

  # ========================================================================= #
  # === map_half_hour_to_the_corret_clock_emoji
  # ========================================================================= #
  def map_half_hour_to_the_corret_clock_emoji
    _ = return_current_half_hour # Capture it to simplify working with it lateron.
    case _
    when  1    then map_emoji(:clock_1)
    when  2    then map_emoji(:clock_2)
    when  3    then map_emoji(:clock_3)
    when  4    then map_emoji(:clock_4)
    when  5    then map_emoji(:clock_5)
    when  6    then map_emoji(:clock_6)
    when  7    then map_emoji(:clock_7)
    when  8    then map_emoji(:clock_8)
    when  9    then map_emoji(:clock_9)
    when 10    then map_emoji(:clock_10)
    when 11    then map_emoji(:clock_11)
    when 0, 12 then map_emoji(:clock_12)
    else
      e 'Not found: '+_.to_s
    end
  end

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    create_skeleton
    connect_skeleton
    show_all
  end

  # ========================================================================= #
  # === Gtk::UnicodeCodeClockModule.run
  # ========================================================================= #
  def self.run
    require 'gtk_paradise/run'
    unicode_clock = ::Gtk::UnicodeClock.new
    r = ::Gtk.run
    r << unicode_clock
    r.set_size_request(
      unicode_clock.width?,
      unicode_clock.height?
    )
    r.enable_quick_exit
    r.top_left_then_run
  end

end; end