/*
 * Decompiled with CFR 0.152.
 */
package edu.stanford.nlp.io;

import edu.stanford.nlp.io.IOUtils;
import edu.stanford.nlp.util.StringUtils;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStreamWriter;
import java.io.StringReader;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import junit.framework.TestCase;
import org.junit.Assert;

public class IOUtilsTest
extends TestCase {
    private String dirPath;
    private File dir;

    protected void setUp() throws Exception {
        super.setUp();
        this.dir = File.createTempFile("IOUtilsTest", ".dir");
        IOUtilsTest.assertTrue((boolean)this.dir.delete());
        IOUtilsTest.assertTrue((boolean)this.dir.mkdir());
        this.dirPath = this.dir.getAbsolutePath();
    }

    protected void tearDown() throws Exception {
        super.tearDown();
        this.delete(this.dir);
    }

    public void testReadWriteStreamFromString() throws IOException, ClassNotFoundException {
        ObjectOutputStream oos = IOUtils.writeStreamFromString(this.dirPath + "/objs.obj");
        oos.writeObject(42);
        oos.writeObject("forty two");
        oos.close();
        ObjectInputStream ois = IOUtils.readStreamFromString(this.dirPath + "/objs.obj");
        Object i = ois.readObject();
        Object s = ois.readObject();
        Assert.assertTrue((boolean)Integer.valueOf(42).equals(i));
        Assert.assertTrue((boolean)"forty two".equals(s));
        ois.close();
    }

    public void testReadLines() throws Exception {
        File file = new File(this.dir, "lines.txt");
        IOUtilsTest.write("abc", file);
        Iterable<String> iterable = IOUtils.readLines(file);
        Assert.assertEquals((Object)"abc", (Object)StringUtils.join(iterable, "!"));
        Assert.assertEquals((Object)"abc", (Object)StringUtils.join(iterable, "!"));
        IOUtilsTest.write("abc\ndef\n", file);
        iterable = IOUtils.readLines(file);
        Assert.assertEquals((Object)"abc!def", (Object)StringUtils.join(iterable, "!"));
        Assert.assertEquals((Object)"abc!def", (Object)StringUtils.join(iterable, "!"));
        IOUtilsTest.write("\na\nb\n", file);
        iterable = IOUtils.readLines(file.getPath());
        Assert.assertEquals((Object)"!a!b", (Object)StringUtils.join(iterable, "!"));
        Assert.assertEquals((Object)"!a!b", (Object)StringUtils.join(iterable, "!"));
        IOUtilsTest.write("", file);
        iterable = IOUtils.readLines(file);
        Assert.assertFalse((boolean)iterable.iterator().hasNext());
        IOUtilsTest.write("\n", file);
        iterable = IOUtils.readLines(file.getPath());
        Iterator<String> iterator = iterable.iterator();
        Assert.assertTrue((boolean)iterator.hasNext());
        iterator.next();
        BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(new GZIPOutputStream(new FileOutputStream(file))));
        writer.write("\nzipped\ntext\n");
        writer.close();
        iterable = IOUtils.readLines(file, GZIPInputStream.class);
        Assert.assertEquals((Object)"!zipped!text", (Object)StringUtils.join(iterable, "!"));
        Assert.assertEquals((Object)"!zipped!text", (Object)StringUtils.join(iterable, "!"));
    }

    private static void checkLineIterable(boolean includeEol) throws IOException {
        String[] expected = new String[]{"abcdefhij\r\n", "klnm\r\n", "opqrst\n", "uvwxyz\r", "I am a longer line than the rest\n", "12345"};
        String testString = StringUtils.join(expected, "");
        StringReader reader = new StringReader(testString);
        int i = 0;
        Iterable<String> iterable = IOUtils.getLineIterable(reader, 10, includeEol);
        for (String line : iterable) {
            String expLine = expected[i];
            if (!includeEol) {
                expLine = expLine.replaceAll("\\r|\\n", "");
            }
            IOUtilsTest.assertEquals((String)("Checking line " + i), (String)expLine, (String)line);
            ++i;
        }
        IOUtilsTest.assertEquals((String)"Check got all lines", (int)expected.length, (int)i);
        IOUtils.closeIgnoringExceptions(reader);
    }

    public void testLineIterableWithEol() throws IOException {
        IOUtilsTest.checkLineIterable(true);
    }

    public void testLineIterableWithoutEol() throws IOException {
        IOUtilsTest.checkLineIterable(false);
    }

    public void testIterFilesRecursive() throws IOException {
        File dir = new File(this.dir, "recursive");
        File a = new File(dir, "x/a");
        File b = new File(dir, "x/y/b.txt");
        File c = new File(dir, "c.txt");
        File d = new File(dir, "dtxt");
        IOUtilsTest.write("A", a);
        IOUtilsTest.write("B", b);
        IOUtilsTest.write("C", c);
        IOUtilsTest.write("D", d);
        Set<File> actual = IOUtilsTest.toSet(IOUtils.iterFilesRecursive(dir));
        Assert.assertEquals(IOUtilsTest.toSet(Arrays.asList(a, b, c, d)), actual);
        actual = IOUtilsTest.toSet(IOUtils.iterFilesRecursive(dir, ".txt"));
        Assert.assertEquals(IOUtilsTest.toSet(Arrays.asList(b, c)), actual);
        actual = IOUtilsTest.toSet(IOUtils.iterFilesRecursive(dir, Pattern.compile(".txt")));
        Assert.assertEquals(IOUtilsTest.toSet(Arrays.asList(b, c, d)), actual);
    }

    protected void delete(File file) {
        File[] children;
        if (file.isDirectory() && (children = file.listFiles()) != null) {
            for (File child : children) {
                this.delete(child);
            }
        }
        Assert.assertTrue((boolean)file.delete());
    }

    protected static void write(String text, File file) throws IOException {
        if (!file.getParentFile().exists()) {
            file.getParentFile().mkdirs();
        }
        FileWriter writer = new FileWriter(file);
        writer.write(text);
        writer.close();
    }

    private static <E> Set<E> toSet(Iterable<E> iter) {
        HashSet<E> set = new HashSet<E>();
        for (E item : iter) {
            set.add(item);
        }
        return set;
    }

    public void testCpSourceFileTargetNotExists() throws IOException {
        File source = File.createTempFile("foo", ".file");
        IOUtils.writeStringToFile("foo", source.getPath(), "utf-8");
        File dst = File.createTempFile("foo", ".file");
        IOUtilsTest.assertTrue((boolean)dst.delete());
        IOUtils.cp(source, dst);
        IOUtilsTest.assertEquals((String)"foo", (String)IOUtils.slurpFile(dst));
        IOUtilsTest.assertTrue((boolean)source.delete());
        IOUtilsTest.assertTrue((boolean)dst.delete());
    }

    public void testCpSourceFileTargetExists() throws IOException {
        File source = File.createTempFile("foo", ".file");
        IOUtils.writeStringToFile("foo", source.getPath(), "utf-8");
        File dst = File.createTempFile("foo", ".file");
        IOUtils.cp(source, dst);
        IOUtilsTest.assertEquals((String)"foo", (String)IOUtils.slurpFile(dst));
        IOUtilsTest.assertTrue((boolean)source.delete());
        IOUtilsTest.assertTrue((boolean)dst.delete());
    }

    public void testCpSourceFileTargetIsDir() throws IOException {
        File source = File.createTempFile("foo", ".file");
        IOUtils.writeStringToFile("foo", source.getPath(), "utf-8");
        File dst = File.createTempFile("foo", ".file");
        IOUtilsTest.assertTrue((boolean)dst.delete());
        IOUtilsTest.assertTrue((boolean)dst.mkdir());
        IOUtils.cp(source, dst);
        IOUtilsTest.assertEquals((String)"foo", (String)IOUtils.slurpFile(dst.getPath() + File.separator + source.getName()));
        IOUtilsTest.assertTrue((boolean)source.delete());
        IOUtilsTest.assertTrue((boolean)new File(dst.getPath() + File.separator + source.getName()).delete());
        IOUtilsTest.assertTrue((boolean)dst.delete());
    }

    public void testCpSourceDirTargetNotExists() throws IOException {
        File sourceDir = File.createTempFile("foo", ".file");
        IOUtilsTest.assertTrue((boolean)sourceDir.delete());
        IOUtilsTest.assertTrue((boolean)sourceDir.mkdir());
        File foo = new File(sourceDir + File.separator + "foo");
        IOUtils.writeStringToFile("foo", foo.getPath(), "utf-8");
        File dst = File.createTempFile("foo", ".file");
        IOUtilsTest.assertTrue((boolean)dst.delete());
        IOUtils.cp(sourceDir, dst, true);
        IOUtilsTest.assertEquals((String)"foo", (String)IOUtils.slurpFile(dst.getPath() + File.separator + "foo"));
        IOUtilsTest.assertTrue((boolean)foo.delete());
        IOUtilsTest.assertTrue((boolean)sourceDir.delete());
        IOUtilsTest.assertTrue((boolean)new File(dst.getPath() + File.separator + "foo").delete());
        IOUtilsTest.assertTrue((boolean)dst.delete());
    }

    public void testCpSourceDirTargetIsDir() throws IOException {
        File sourceDir = File.createTempFile("foo", ".file");
        IOUtilsTest.assertTrue((boolean)sourceDir.delete());
        IOUtilsTest.assertTrue((boolean)sourceDir.mkdir());
        File foo = new File(sourceDir + File.separator + "foo");
        IOUtils.writeStringToFile("foo", foo.getPath(), "utf-8");
        File dst = File.createTempFile("foo", ".file");
        IOUtilsTest.assertTrue((boolean)dst.delete());
        IOUtilsTest.assertTrue((boolean)dst.mkdir());
        IOUtils.cp(sourceDir, dst, true);
        IOUtilsTest.assertEquals((String)"foo", (String)IOUtils.slurpFile(dst.getPath() + File.separator + sourceDir.getName() + File.separator + "foo"));
        IOUtilsTest.assertTrue((boolean)foo.delete());
        IOUtilsTest.assertTrue((boolean)sourceDir.delete());
        IOUtilsTest.assertTrue((boolean)new File(dst.getPath() + File.separator + sourceDir.getName() + File.separator + "foo").delete());
        IOUtilsTest.assertTrue((boolean)new File(dst.getPath() + File.separator + sourceDir.getName()).delete());
        IOUtilsTest.assertTrue((boolean)dst.delete());
    }

    public void testCpRecursive() throws IOException {
        File sourceDir = File.createTempFile("directory", ".file");
        IOUtilsTest.assertTrue((boolean)sourceDir.delete());
        IOUtilsTest.assertTrue((boolean)sourceDir.mkdir());
        File sourceSubDir = new File(sourceDir + File.separator + "d2");
        IOUtilsTest.assertTrue((boolean)sourceSubDir.mkdir());
        File foo = new File(sourceSubDir + File.separator + "foo");
        IOUtils.writeStringToFile("foo", foo.getPath(), "utf-8");
        File bar = new File(sourceDir + File.separator + "bar");
        IOUtils.writeStringToFile("bar", bar.getPath(), "utf-8");
        File dst = File.createTempFile("dst", ".file");
        IOUtilsTest.assertTrue((boolean)dst.delete());
        IOUtils.cp(sourceDir, dst, true);
        IOUtilsTest.assertEquals((String)"foo", (String)IOUtils.slurpFile(dst + File.separator + "d2" + File.separator + "foo"));
        IOUtilsTest.assertEquals((String)"bar", (String)IOUtils.slurpFile(dst + File.separator + "bar"));
        IOUtilsTest.assertTrue((boolean)foo.delete());
        IOUtilsTest.assertTrue((boolean)bar.delete());
        IOUtilsTest.assertTrue((boolean)sourceSubDir.delete());
        IOUtilsTest.assertTrue((boolean)sourceDir.delete());
        IOUtilsTest.assertTrue((boolean)new File(dst + File.separator + "d2" + File.separator + "foo").delete());
        IOUtilsTest.assertTrue((boolean)new File(dst + File.separator + "d2").delete());
        IOUtilsTest.assertTrue((boolean)new File(dst + File.separator + "bar").delete());
        IOUtilsTest.assertTrue((boolean)dst.delete());
    }

    public void testTail() throws IOException {
        File f = File.createTempFile("totail", ".file");
        IOUtils.writeStringToFile("line 1\nline 2\nline 3\nline 4\nline 5\nline 6\nline 7", f.getPath(), "utf-8");
        IOUtilsTest.assertEquals((String)"line 7", (String)IOUtils.tail(f, 1)[0]);
        IOUtilsTest.assertEquals((String)"line 6", (String)IOUtils.tail(f, 2)[0]);
        IOUtilsTest.assertEquals((String)"line 7", (String)IOUtils.tail(f, 2)[1]);
        IOUtils.writeStringToFile("line 1\nline 2\n\nline 3\n", f.getPath(), "utf-8");
        IOUtilsTest.assertEquals((String)"", (String)IOUtils.tail(f, 1)[0]);
        IOUtilsTest.assertEquals((String)"", (String)IOUtils.tail(f, 3)[0]);
        IOUtilsTest.assertEquals((String)"line 3", (String)IOUtils.tail(f, 3)[1]);
        IOUtilsTest.assertEquals((String)"", (String)IOUtils.tail(f, 3)[2]);
        IOUtils.writeStringToFile("line 1\nline 2", f.getPath(), "utf-8");
        IOUtilsTest.assertEquals((int)0, (int)IOUtils.tail(f, 0).length);
        IOUtilsTest.assertEquals((int)1, (int)IOUtils.tail(f, 1).length);
        IOUtilsTest.assertEquals((int)2, (int)IOUtils.tail(f, 3).length);
        IOUtilsTest.assertEquals((int)2, (int)IOUtils.tail(f, 2).length);
        IOUtils.writeStringToFile("\u21b9\u219d\n\u06dd\u00e6", f.getPath(), "utf-8");
        IOUtilsTest.assertEquals((String)"\u21b9\u219d", (String)IOUtils.tail(f, 2)[0]);
        IOUtilsTest.assertEquals((String)"\u06dd\u00e6", (String)IOUtils.tail(f, 2)[1]);
        IOUtilsTest.assertTrue((boolean)f.delete());
    }
}

