/*
 * Decompiled with CFR 0.152.
 */
package edu.stanford.nlp.trees.tregex;

import edu.stanford.nlp.trees.LabeledScoredTreeFactory;
import edu.stanford.nlp.trees.PennTreeReader;
import edu.stanford.nlp.trees.Tree;
import edu.stanford.nlp.trees.tregex.TregexMatcher;
import edu.stanford.nlp.trees.tregex.TregexParseException;
import edu.stanford.nlp.trees.tregex.TregexPattern;
import edu.stanford.nlp.trees.tregex.TregexPatternCompiler;
import java.io.IOException;
import java.io.StringReader;
import java.util.IdentityHashMap;
import java.util.Set;
import java.util.function.Function;
import junit.framework.AssertionFailedError;
import junit.framework.TestCase;

public class TregexTest
extends TestCase {
    public static Tree treeFromString(String s) {
        try {
            PennTreeReader tr = new PennTreeReader(new StringReader(s), new LabeledScoredTreeFactory());
            return tr.readTree();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static Tree[] treesFromString(String ... s) {
        Tree[] trees = new Tree[s.length];
        for (int i = 0; i < s.length; ++i) {
            trees[i] = TregexTest.treeFromString(s[i]);
        }
        return trees;
    }

    public void testJo\u00e3oSilva() {
        TregexPattern tregex1 = TregexPattern.compile("PNT=p >>- (__=l >, (__=t <- (__=r <, __=m <- (__ <, CONJ <- __=z))))");
        TregexPattern tregex2 = TregexPattern.compile("PNT=p >>- (/(.+)/#1%var=l >, (__=t <- (__=r <, /(.+)/#1%var=m <- (__ <, CONJ <- /(.+)/#1%var=z))))");
        TregexPattern tregex3 = TregexPattern.compile("PNT=p >>- (__=l >, (__=t <- (__=r <, ~l <- (__ <, CONJ <- ~l))))");
        Tree tree = TregexTest.treeFromString("(T (X (N (N Moe (PNT ,)))) (NP (X (N Curly)) (NP (CONJ and) (X (N Larry)))))");
        TregexMatcher matcher1 = tregex1.matcher(tree);
        TregexTest.assertTrue((boolean)matcher1.find());
        TregexMatcher matcher2 = tregex2.matcher(tree);
        TregexTest.assertTrue((boolean)matcher2.find());
        TregexMatcher matcher3 = tregex3.matcher(tree);
        TregexTest.assertTrue((boolean)matcher3.find());
    }

    public void testNoResults() {
        TregexPattern pMWE = TregexPattern.compile("/^MW/");
        Tree tree = TregexTest.treeFromString("(Foo)");
        TregexMatcher matcher = pMWE.matcher(tree);
        TregexTest.assertFalse((boolean)matcher.find());
    }

    public void testOneResult() {
        TregexPattern pMWE = TregexPattern.compile("/^MW/");
        Tree tree = TregexTest.treeFromString("(ROOT (MWE (N 1) (N 2) (N 3)))");
        TregexMatcher matcher = pMWE.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        Tree match = matcher.getMatch();
        TregexTest.assertEquals((String)"(MWE (N 1) (N 2) (N 3))", (String)match.toString());
        TregexTest.assertFalse((boolean)matcher.find());
    }

    public void testTwoResults() {
        TregexPattern pMWE = TregexPattern.compile("/^MW/");
        Tree tree = TregexTest.treeFromString("(ROOT (MWE (N 1) (N 2) (N 3)) (MWV (A B)))");
        TregexMatcher matcher = pMWE.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        Tree match = matcher.getMatch();
        TregexTest.assertEquals((String)"(MWE (N 1) (N 2) (N 3))", (String)match.toString());
        TregexTest.assertTrue((boolean)matcher.find());
        match = matcher.getMatch();
        TregexTest.assertEquals((String)"(MWV (A B))", (String)match.toString());
        TregexTest.assertFalse((boolean)matcher.find());
    }

    public void testReuse() {
        TregexPattern pMWE = TregexPattern.compile("/^MW/");
        Tree tree = TregexTest.treeFromString("(ROOT (MWE (N 1) (N 2) (N 3)) (MWV (A B)))");
        TregexMatcher matcher = pMWE.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertFalse((boolean)matcher.find());
        tree = TregexTest.treeFromString("(ROOT (MWE (N 1) (N 2) (N 3)))");
        matcher = pMWE.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertFalse((boolean)matcher.find());
        tree = TregexTest.treeFromString("(Foo)");
        matcher = pMWE.matcher(tree);
        TregexTest.assertFalse((boolean)matcher.find());
    }

    public void testTest() {
        TregexTest.runTest("/^MW/", "(ROOT (MWE (N 1) (N 2) (N 3)) (MWV (A B)))", "(MWE (N 1) (N 2) (N 3))", "(MWV (A B))");
    }

    public void testWordDisjunction() {
        TregexPattern pattern = TregexPattern.compile("a|b|c << bar");
        TregexTest.runTest(pattern, "(a (bar 1))", "(a (bar 1))");
        TregexTest.runTest(pattern, "(b (bar 1))", "(b (bar 1))");
        TregexTest.runTest(pattern, "(c (bar 1))", "(c (bar 1))");
        TregexTest.runTest(pattern, "(d (bar 1))", new String[0]);
        TregexTest.runTest(pattern, "(e (bar 1))", new String[0]);
        TregexTest.runTest(pattern, "(f (bar 1))", new String[0]);
        TregexTest.runTest(pattern, "(g (bar 1))", new String[0]);
        pattern = TregexPattern.compile("a|b|c|d|e|f << bar");
        TregexTest.runTest(pattern, "(a (bar 1))", "(a (bar 1))");
        TregexTest.runTest(pattern, "(b (bar 1))", "(b (bar 1))");
        TregexTest.runTest(pattern, "(c (bar 1))", "(c (bar 1))");
        TregexTest.runTest(pattern, "(d (bar 1))", "(d (bar 1))");
        TregexTest.runTest(pattern, "(e (bar 1))", "(e (bar 1))");
        TregexTest.runTest(pattern, "(f (bar 1))", "(f (bar 1))");
        TregexTest.runTest(pattern, "(g (bar 1))", new String[0]);
    }

    public void testDominates() {
        TregexPattern dominatesPattern = TregexPattern.compile("foo << bar");
        TregexTest.runTest(dominatesPattern, "(foo (bar 1))", "(foo (bar 1))");
        TregexTest.runTest(dominatesPattern, "(foo (a (bar 1)))", "(foo (a (bar 1)))");
        TregexTest.runTest(dominatesPattern, "(foo (a (b (bar 1))))", "(foo (a (b (bar 1))))");
        TregexTest.runTest(dominatesPattern, "(foo (a (b 1) (bar 2)))", "(foo (a (b 1) (bar 2)))");
        TregexTest.runTest(dominatesPattern, "(foo (a (b 1) (c 2) (bar 3)))", "(foo (a (b 1) (c 2) (bar 3)))");
        TregexTest.runTest(dominatesPattern, "(foo (baz 1))", new String[0]);
        TregexTest.runTest(dominatesPattern, "(a (foo (bar 1)))", "(foo (bar 1))");
        TregexTest.runTest(dominatesPattern, "(a (foo (baz (bar 1))))", "(foo (baz (bar 1)))");
        TregexTest.runTest(dominatesPattern, "(a (foo (bar 1)) (foo (bar 2)))", "(foo (bar 1))", "(foo (bar 2))");
        TregexPattern dominatedPattern = TregexPattern.compile("foo >> bar");
        TregexTest.runTest(dominatedPattern, "(foo (bar 1))", new String[0]);
        TregexTest.runTest(dominatedPattern, "(foo (a (bar 1)))", new String[0]);
        TregexTest.runTest(dominatedPattern, "(foo (a (b (bar 1))))", new String[0]);
        TregexTest.runTest(dominatedPattern, "(foo (a (b 1) (bar 2)))", new String[0]);
        TregexTest.runTest(dominatedPattern, "(foo (a (b 1) (c 2) (bar 3)))", new String[0]);
        TregexTest.runTest(dominatedPattern, "(bar (foo 1))", "(foo 1)");
        TregexTest.runTest(dominatedPattern, "(bar (a (foo 1)))", "(foo 1)");
        TregexTest.runTest(dominatedPattern, "(bar (a (foo (b 1))))", "(foo (b 1))");
        TregexTest.runTest(dominatedPattern, "(bar (a (foo 1) (foo 2)))", "(foo 1)", "(foo 2)");
        TregexTest.runTest(dominatedPattern, "(bar (foo (foo 1)))", "(foo (foo 1))", "(foo 1)");
        TregexTest.runTest(dominatedPattern, "(a (bar (foo 1)))", "(foo 1)");
    }

    public void testImmediatelyDominates() {
        TregexPattern dominatesPattern = TregexPattern.compile("foo < bar");
        TregexTest.runTest(dominatesPattern, "(foo (bar 1))", "(foo (bar 1))");
        TregexTest.runTest(dominatesPattern, "(foo (a (bar 1)))", new String[0]);
        TregexTest.runTest(dominatesPattern, "(a (foo (bar 1)))", "(foo (bar 1))");
        TregexTest.runTest(dominatesPattern, "(a (foo (baz 1) (bar 2)))", "(foo (baz 1) (bar 2))");
        TregexTest.runTest(dominatesPattern, "(a (foo (bar 1)) (foo (bar 2)))", "(foo (bar 1))", "(foo (bar 2))");
        TregexPattern dominatedPattern = TregexPattern.compile("foo > bar");
        TregexTest.runTest(dominatedPattern, "(foo (bar 1))", new String[0]);
        TregexTest.runTest(dominatedPattern, "(foo (a (bar 1)))", new String[0]);
        TregexTest.runTest(dominatedPattern, "(foo (a (b (bar 1))))", new String[0]);
        TregexTest.runTest(dominatedPattern, "(foo (a (b 1) (bar 2)))", new String[0]);
        TregexTest.runTest(dominatedPattern, "(foo (a (b 1) (c 2) (bar 3)))", new String[0]);
        TregexTest.runTest(dominatedPattern, "(bar (foo 1))", "(foo 1)");
        TregexTest.runTest(dominatedPattern, "(bar (a (foo 1)))", new String[0]);
        TregexTest.runTest(dominatedPattern, "(bar (foo 1) (foo 2))", "(foo 1)", "(foo 2)");
        TregexTest.runTest(dominatedPattern, "(bar (foo (foo 1)))", "(foo (foo 1))");
        TregexTest.runTest(dominatedPattern, "(a (bar (foo 1)))", "(foo 1)");
    }

    public void testSister() {
        TregexPattern pattern = TregexPattern.compile("/.*/ $ foo");
        TregexTest.runTest(pattern, "(a (foo 1) (bar 2))", "(bar 2)");
        TregexTest.runTest(pattern, "(a (bar 1) (foo 2))", "(bar 1)");
        TregexTest.runTest(pattern, "(a (foo 1) (bar 2) (baz 3))", "(bar 2)", "(baz 3)");
        TregexTest.runTest(pattern, "(a (foo (bar 2)) (baz 3))", "(baz 3)");
        TregexTest.runTest(pattern, "(a (foo (bar 2)) (baz (bif 3)))", "(baz (bif 3))");
        TregexTest.runTest(pattern, "(a (foo (bar 2)))", new String[0]);
        TregexTest.runTest(pattern, "(a (foo 1))", new String[0]);
        pattern = TregexPattern.compile("bar|baz $ foo");
        TregexTest.runTest(pattern, "(a (foo 1) (bar 2))", "(bar 2)");
        TregexTest.runTest(pattern, "(a (bar 1) (foo 2))", "(bar 1)");
        TregexTest.runTest(pattern, "(a (foo 1) (bar 2) (baz 3))", "(bar 2)", "(baz 3)");
        TregexTest.runTest(pattern, "(a (foo (bar 2)) (baz 3))", "(baz 3)");
        TregexTest.runTest(pattern, "(a (foo (bar 2)) (baz (bif 3)))", "(baz (bif 3))");
        TregexTest.runTest(pattern, "(a (foo (bar 2)))", new String[0]);
        TregexTest.runTest(pattern, "(a (foo 1))", new String[0]);
        pattern = TregexPattern.compile("/.*/ $ foo");
        TregexTest.runTest(pattern, "(a (foo 1) (foo 2))", "(foo 1)", "(foo 2)");
        TregexTest.runTest(pattern, "(a (foo 1))", new String[0]);
        pattern = TregexPattern.compile("foo $ foo");
        TregexTest.runTest(pattern, "(a (foo 1) (foo 2))", "(foo 1)", "(foo 2)");
        TregexTest.runTest(pattern, "(a (foo 1))", new String[0]);
        pattern = TregexPattern.compile("foo $ foo=a");
        Tree tree = TregexTest.treeFromString("(a (foo 1) (foo 2) (foo 3))");
        TregexMatcher matcher = pattern.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)"(foo 1)", (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"(foo 2)", (String)matcher.getNode("a").toString());
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)"(foo 1)", (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"(foo 3)", (String)matcher.getNode("a").toString());
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)"(foo 2)", (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"(foo 1)", (String)matcher.getNode("a").toString());
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)"(foo 2)", (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"(foo 3)", (String)matcher.getNode("a").toString());
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)"(foo 3)", (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"(foo 1)", (String)matcher.getNode("a").toString());
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)"(foo 3)", (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"(foo 2)", (String)matcher.getNode("a").toString());
        TregexTest.assertFalse((boolean)matcher.find());
        TregexTest.runTest("foo $ foo", "(a (foo 1))", new String[0]);
    }

    public void testPrecedesFollows() {
        TregexPattern pattern = TregexPattern.compile("/.*/ .. foo");
        TregexTest.runTest(pattern, "(a (foo 1) (bar 2))", new String[0]);
        TregexTest.runTest(pattern, "(a (bar 1) (foo 2))", "(bar 1)", "(1)");
        TregexTest.runTest(pattern, "(a (bar 1) (baz 2) (foo 3))", "(bar 1)", "(1)", "(baz 2)", "(2)");
        TregexTest.runTest(pattern, "(a (foo 1) (baz 2) (bar 3))", new String[0]);
        TregexTest.runTest(pattern, "(a (bar (foo 1)) (baz 2))", new String[0]);
        TregexTest.runTest(pattern, "(a (bar 1) (baz (foo 2)))", "(bar 1)", "(1)");
        TregexTest.runTest(pattern, "(a (bar 1) (baz 2) (bif (foo 3)))", "(bar 1)", "(1)", "(baz 2)", "(2)");
        TregexTest.runTest(pattern, "(a (bar (foo 1)) (baz 2) (bif 3))", new String[0]);
        TregexTest.runTest(pattern, "(a (bar 1) (baz 2) (foo (bif 3)))", "(bar 1)", "(1)", "(baz 2)", "(2)");
        TregexTest.runTest(pattern, "(a (bar 1) (foo (bif 2)) (baz 3))", "(bar 1)", "(1)");
        pattern = TregexPattern.compile("/.*/ ,, foo");
        TregexTest.runTest(pattern, "(a (foo 1) (bar 2))", "(bar 2)", "(2)");
        TregexTest.runTest(pattern, "(a (bar 1) (foo 2))", new String[0]);
        TregexTest.runTest(pattern, "(a (bar 1) (baz 2) (foo 3))", new String[0]);
        TregexTest.runTest(pattern, "(a (foo 1) (baz 2) (bar 3))", "(baz 2)", "(2)", "(bar 3)", "(3)");
        TregexTest.runTest(pattern, "(a (bar (foo 1)) (baz 2))", "(baz 2)", "(2)");
        TregexTest.runTest(pattern, "(a (bar 1) (baz (foo 2)))", new String[0]);
        TregexTest.runTest(pattern, "(a (bar 1) (baz 2) (bif (foo 3)))", new String[0]);
        TregexTest.runTest(pattern, "(a (bar (foo 1)) (baz 2) (bif 3))", "(baz 2)", "(2)", "(bif 3)", "(3)");
        TregexTest.runTest(pattern, "(a (bar 1) (baz 2) (foo (bif 3)))", new String[0]);
        TregexTest.runTest(pattern, "(a (foo (bif 1)) (bar 2) (baz 3))", "(bar 2)", "(2)", "(baz 3)", "(3)");
        TregexTest.runTest(pattern, "(a (bar 1) (foo (bif 2)) (baz 3))", "(baz 3)", "(3)");
    }

    public void testImmediatePrecedesFollows() {
        TregexPattern pattern = TregexPattern.compile("/.*/ . foo");
        TregexTest.runTest(pattern, "(a (foo 1) (bar 2))", new String[0]);
        TregexTest.runTest(pattern, "(a (bar 1) (foo 2))", "(bar 1)", "(1)");
        TregexTest.runTest(pattern, "(a (bar 1) (baz 2) (foo 3))", "(baz 2)", "(2)");
        TregexTest.runTest(pattern, "(a (foo 1) (baz 2) (bar 3))", new String[0]);
        TregexTest.runTest(pattern, "(a (bar (foo 1)) (baz 2))", new String[0]);
        TregexTest.runTest(pattern, "(a (bar 1) (baz (foo 2)))", "(bar 1)", "(1)");
        TregexTest.runTest(pattern, "(a (bar 1) (baz 2) (bif (foo 3)))", "(baz 2)", "(2)");
        TregexTest.runTest(pattern, "(a (bar (foo 1)) (baz 2) (bif 3))", new String[0]);
        TregexTest.runTest(pattern, "(a (bar 1) (baz 2) (foo (bif 3)))", "(baz 2)", "(2)");
        TregexTest.runTest(pattern, "(a (bar 1) (foo (bif 2)) (baz 3))", "(bar 1)", "(1)");
        TregexTest.runTest(pattern, "(a (bar 1) (foo 2) (baz 3) (foo 4) (bif 5))", "(bar 1)", "(1)", "(baz 3)", "(3)");
        TregexTest.runTest(pattern, "(a (bar 1) (foo 2) (foo 3) (baz 4))", "(bar 1)", "(1)", "(foo 2)", "(2)");
        TregexTest.runTest(pattern, "(a (b (c 1) (d 2)) (foo))", "(b (c 1) (d 2))", "(d 2)", "(2)");
        TregexTest.runTest(pattern, "(a (b (c 1) (d 2)) (bar (foo 3)))", "(b (c 1) (d 2))", "(d 2)", "(2)");
        TregexTest.runTest(pattern, "(a (b (c 1) (d 2)) (bar (baz 3) (foo 4)))", "(baz 3)", "(3)");
        TregexTest.runTest(pattern, "(a (b (c 1) (d 2)) (bar (baz 2 3) (foo 4)))", "(baz 2 3)", "(3)");
        pattern = TregexPattern.compile("/.*/ , foo");
        TregexTest.runTest(pattern, "(a (foo 1) (bar 2))", "(bar 2)", "(2)");
        TregexTest.runTest(pattern, "(a (bar 1) (foo 2))", new String[0]);
        TregexTest.runTest(pattern, "(a (bar 1) (baz 2) (foo 3))", new String[0]);
        TregexTest.runTest(pattern, "(a (foo 1) (baz 2) (bar 3))", "(baz 2)", "(2)");
        TregexTest.runTest(pattern, "(a (bar (foo 1)) (baz 2))", "(baz 2)", "(2)");
        TregexTest.runTest(pattern, "(a (bar 1) (baz (foo 2)))", new String[0]);
        TregexTest.runTest(pattern, "(a (bar 1) (baz 2) (bif (foo 3)))", new String[0]);
        TregexTest.runTest(pattern, "(a (bar (foo 1)) (baz 2) (bif 3))", "(baz 2)", "(2)");
        TregexTest.runTest(pattern, "(a (bar 1) (baz 2) (foo (bif 3)))", new String[0]);
        TregexTest.runTest(pattern, "(a (foo (bif 1)) (bar 2) (baz 3))", "(bar 2)", "(2)");
        TregexTest.runTest(pattern, "(a (bar 1) (foo (bif 2)) (baz 3))", "(baz 3)", "(3)");
        TregexTest.runTest(pattern, "(a (bar 1) (foo 2) (baz 3) (foo 4) (bif 5))", "(baz 3)", "(3)", "(bif 5)", "(5)");
        TregexTest.runTest(pattern, "(a (bar 1) (foo 2) (foo 3) (baz 4))", "(foo 3)", "(3)", "(baz 4)", "(4)");
        TregexTest.runTest(pattern, "(a (foo) (b (c 1) (d 2)))", "(b (c 1) (d 2))", "(c 1)", "(1)");
        TregexTest.runTest(pattern, "(a (bar (foo 3)) (b (c 1) (d 2)))", "(b (c 1) (d 2))", "(c 1)", "(1)");
        TregexTest.runTest(pattern, "(a (bar (baz 3) (foo 4)) (b (c 1) (d 2)))", "(b (c 1) (d 2))", "(c 1)", "(1)");
        TregexTest.runTest(pattern, "(a (bar (foo 4) (baz 3)) (b (c 1) (d 2)))", "(baz 3)", "(3)");
    }

    public void testLeftRightMostDescendant() {
        TregexTest.runTest("/.*/ <<, /1/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(a (foo 1 2) (bar 3 4))", "(foo 1 2)");
        TregexTest.runTest("/.*/ <<, /2/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", new String[0]);
        TregexTest.runTest("/.*/ <<, foo", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(a (foo 1 2) (bar 3 4))");
        TregexTest.runTest("/.*/ <<, baz", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(b (baz 5))");
        TregexTest.runTest("/.*/ <<- /1/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", new String[0]);
        TregexTest.runTest("/.*/ <<- /2/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(foo 1 2)");
        TregexTest.runTest("/.*/ <<- /4/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(a (foo 1 2) (bar 3 4))", "(bar 3 4)");
        TregexTest.runTest("/.*/ >>, root", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(a (foo 1 2) (bar 3 4))", "(foo 1 2)", "(1)");
        TregexTest.runTest("/.*/ >>, a", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(foo 1 2)", "(1)");
        TregexTest.runTest("/.*/ >>, bar", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(3)");
        TregexTest.runTest("/.*/ >>- root", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(b (baz 5))", "(baz 5)", "(5)");
        TregexTest.runTest("/.*/ >>- a", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(bar 3 4)", "(4)");
        TregexTest.runTest("/.*/ >>- /1/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", new String[0]);
    }

    public void testFirstLastChild() {
        TregexTest.runTest("/.*/ >, root", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(a (foo 1 2) (bar 3 4))");
        TregexTest.runTest("/.*/ >, a", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(foo 1 2)");
        TregexTest.runTest("/.*/ >, foo", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(1)");
        TregexTest.runTest("/.*/ >, bar", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(3)");
        TregexTest.runTest("/.*/ >- root", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(b (baz 5))");
        TregexTest.runTest("/.*/ >- a", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(bar 3 4)");
        TregexTest.runTest("/.*/ >- foo", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(2)");
        TregexTest.runTest("/.*/ >- bar", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(4)");
        TregexTest.runTest("/.*/ >- b", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(baz 5)");
        TregexTest.runTest("/.*/ <, root", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", new String[0]);
        TregexTest.runTest("/.*/ <, a", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))");
        TregexTest.runTest("/.*/ <, /1/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(foo 1 2)");
        TregexTest.runTest("/.*/ <, /2/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", new String[0]);
        TregexTest.runTest("/.*/ <, bar", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", new String[0]);
        TregexTest.runTest("/.*/ <, /3/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(bar 3 4)");
        TregexTest.runTest("/.*/ <, /4/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", new String[0]);
        TregexTest.runTest("/.*/ <- root", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", new String[0]);
        TregexTest.runTest("/.*/ <- a", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", new String[0]);
        TregexTest.runTest("/.*/ <- /1/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", new String[0]);
        TregexTest.runTest("/.*/ <- /2/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(foo 1 2)");
        TregexTest.runTest("/.*/ <- bar", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(a (foo 1 2) (bar 3 4))");
        TregexTest.runTest("/.*/ <- /3/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", new String[0]);
        TregexTest.runTest("/.*/ <- /4/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(bar 3 4)");
    }

    public void testIthChild() {
        TregexTest.runTest("/.*/ >1 root", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(a (foo 1 2) (bar 3 4))");
        TregexTest.runTest("/.*/ >1 a", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(foo 1 2)");
        TregexTest.runTest("/.*/ >2 a", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(bar 3 4)");
        TregexTest.runTest("/.*/ >1 foo", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(1)");
        TregexTest.runTest("/.*/ >2 foo", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(2)");
        TregexTest.runTest("/.*/ >1 bar", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(3)");
        TregexTest.runTest("/.*/ >2 bar", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(4)");
        TregexTest.runTest("/.*/ >-1 root", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(b (baz 5))");
        TregexTest.runTest("/.*/ >-1 a", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(bar 3 4)");
        TregexTest.runTest("/.*/ >-2 a", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(foo 1 2)");
        TregexTest.runTest("/.*/ >-1 foo", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(2)");
        TregexTest.runTest("/.*/ >-2 bar", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(3)");
        TregexTest.runTest("/.*/ >-1 bar", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(4)");
        TregexTest.runTest("/.*/ >-1 b", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(baz 5)");
        TregexTest.runTest("/.*/ >-2 b", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", new String[0]);
        TregexTest.runTest("/.*/ <1 root", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", new String[0]);
        TregexTest.runTest("/.*/ <1 a", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))");
        TregexTest.runTest("/.*/ <1 /1/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(foo 1 2)");
        TregexTest.runTest("/.*/ <1 /2/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", new String[0]);
        TregexTest.runTest("/.*/ <1 bar", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", new String[0]);
        TregexTest.runTest("/.*/ <2 bar", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(a (foo 1 2) (bar 3 4))");
        TregexTest.runTest("/.*/ <3 bar", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", new String[0]);
        TregexTest.runTest("/.*/ <1 /3/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(bar 3 4)");
        TregexTest.runTest("/.*/ <1 /4/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", new String[0]);
        TregexTest.runTest("/.*/ <2 /4/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(bar 3 4)");
        TregexTest.runTest("/.*/ <-1 root", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", new String[0]);
        TregexTest.runTest("/.*/ <-1 a", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", new String[0]);
        TregexTest.runTest("/.*/ <-2 a", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))");
        TregexTest.runTest("/.*/ <-1 /1/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", new String[0]);
        TregexTest.runTest("/.*/ <-2 /1/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(foo 1 2)");
        TregexTest.runTest("/.*/ <-1 /2/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(foo 1 2)");
        TregexTest.runTest("/.*/ <-2 /2/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", new String[0]);
        TregexTest.runTest("/.*/ <-1 bar", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(a (foo 1 2) (bar 3 4))");
        TregexTest.runTest("/.*/ <-1 /3/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", new String[0]);
        TregexTest.runTest("/.*/ <-2 /3/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(bar 3 4)");
        TregexTest.runTest("/.*/ <-1 /4/", "(root (a (foo 1 2) (bar 3 4)) (b (baz 5)))", "(bar 3 4)");
    }

    public void testOnlyChild() {
        TregexTest.runTest("foo <: bar", "(foo (bar 1))", "(foo (bar 1))");
        TregexTest.runTest("foo <: bar", "(foo (bar 1) (bar 2))", new String[0]);
        TregexTest.runTest("foo <: bar", "(foo)", new String[0]);
        TregexTest.runTest("foo <: bar", "(foo (baz (bar))))", new String[0]);
        TregexTest.runTest("foo <: bar", "(foo 1)", new String[0]);
        TregexTest.runTest("bar >: foo", "(foo (bar 1))", "(bar 1)");
        TregexTest.runTest("bar >: foo", "(foo (bar 1) (bar 2))", new String[0]);
        TregexTest.runTest("bar >: foo", "(foo)", new String[0]);
        TregexTest.runTest("bar >: foo", "(foo (baz (bar))))", new String[0]);
        TregexTest.runTest("bar >: foo", "(bar (foo 1))", new String[0]);
        TregexTest.runTest("/.*/ >: foo", "(a (foo (bar 1)) (foo (baz 2)))", "(bar 1)", "(baz 2)");
    }

    public void testDominateUnaryChain() {
        TregexTest.runTest("foo <<: bar", "(a (foo (b (c (d (bar))))))", "(foo (b (c (d (bar)))))");
        TregexTest.runTest("foo <<: bar", "(a (foo (b (c (d (bar) (baz))))))", new String[0]);
        TregexTest.runTest("foo <<: bar", "(a (foo (b (c (d (bar)) (baz)))))", new String[0]);
        TregexTest.runTest("foo <<: bar", "(a (foo (b (c (d (bar))) (baz))))", new String[0]);
        TregexTest.runTest("foo <<: bar", "(a (foo (b (c (d (bar)))) (baz)))", new String[0]);
        TregexTest.runTest("foo <<: bar", "(a (foo (b (c (d (bar))))) (baz))", "(foo (b (c (d (bar)))))");
        TregexTest.runTest("foo <<: bar", "(a (foo (b (c (bar)))))", "(foo (b (c (bar))))");
        TregexTest.runTest("foo <<: bar", "(a (foo (b (bar))))", "(foo (b (bar)))");
        TregexTest.runTest("foo <<: bar", "(a (foo (bar)))", "(foo (bar))");
        TregexTest.runTest("bar >>: foo", "(a (foo (b (c (d (bar))))))", "(bar)");
        TregexTest.runTest("bar >>: foo", "(a (foo (b (c (d (bar) (baz))))))", new String[0]);
        TregexTest.runTest("bar >>: foo", "(a (foo (b (c (d (bar)) (baz)))))", new String[0]);
        TregexTest.runTest("bar >>: foo", "(a (foo (b (c (d (bar))) (baz))))", new String[0]);
        TregexTest.runTest("bar >>: foo", "(a (foo (b (c (d (bar)))) (baz)))", new String[0]);
        TregexTest.runTest("bar >>: foo", "(a (foo (b (c (d (bar))))) (baz))", "(bar)");
        TregexTest.runTest("bar >>: foo", "(a (foo (b (c (bar)))))", "(bar)");
        TregexTest.runTest("bar >>: foo", "(a (foo (b (bar))))", "(bar)");
        TregexTest.runTest("bar >>: foo", "(a (foo (bar)))", "(bar)");
    }

    public void testPrecedingFollowingSister() {
        TregexPattern preceding = TregexPattern.compile("/.*/ $.. baz");
        TregexTest.runTest(preceding, "(a (foo 1) (bar 2) (baz 3))", "(foo 1)", "(bar 2)");
        TregexTest.runTest(preceding, "(root (b (foo 1)) (a (foo 1) (bar 2) (baz 3)))", "(foo 1)", "(bar 2)");
        TregexTest.runTest(preceding, "(root (a (foo 1) (bar 2) (baz 3)) (b (foo 1)))", "(foo 1)", "(bar 2)");
        TregexTest.runTest(preceding, "(a (foo 1) (baz 2) (bar 3))", "(foo 1)");
        TregexTest.runTest(preceding, "(a (baz 1) (foo 2) (bar 3))", new String[0]);
        TregexPattern impreceding = TregexPattern.compile("/.*/ $. baz");
        TregexTest.runTest(impreceding, "(a (foo 1) (bar 2) (baz 3))", "(bar 2)");
        TregexTest.runTest(impreceding, "(root (b (foo 1)) (a (foo 1) (bar 2) (baz 3)))", "(bar 2)");
        TregexTest.runTest(impreceding, "(root (a (foo 1) (bar 2) (baz 3)) (b (foo 1)))", "(bar 2)");
        TregexTest.runTest(impreceding, "(a (foo 1) (baz 2) (bar 3))", "(foo 1)");
        TregexTest.runTest(impreceding, "(a (baz 1) (foo 2) (bar 3))", new String[0]);
        TregexPattern following = TregexPattern.compile("/.*/ $,, baz");
        TregexTest.runTest(following, "(a (foo 1) (bar 2) (baz 3))", new String[0]);
        TregexTest.runTest(following, "(root (b (foo 1)) (a (foo 1) (bar 2) (baz 3)))", new String[0]);
        TregexTest.runTest(following, "(root (a (foo 1) (bar 2) (baz 3)) (b (foo 1)))", new String[0]);
        TregexTest.runTest(following, "(root (a (baz 1) (bar 2) (foo 3)) (b (foo 1)))", "(bar 2)", "(foo 3)");
        TregexTest.runTest(following, "(a (foo 1) (baz 2) (bar 3))", "(bar 3)");
        TregexTest.runTest(following, "(a (baz 1) (foo 2) (bar 3))", "(foo 2)", "(bar 3)");
        TregexPattern imfollowing = TregexPattern.compile("/.*/ $, baz");
        TregexTest.runTest(imfollowing, "(a (foo 1) (bar 2) (baz 3))", new String[0]);
        TregexTest.runTest(imfollowing, "(root (b (foo 1)) (a (foo 1) (bar 2) (baz 3)))", new String[0]);
        TregexTest.runTest(imfollowing, "(root (a (foo 1) (bar 2) (baz 3)) (b (foo 1)))", new String[0]);
        TregexTest.runTest(imfollowing, "(root (a (baz 1) (bar 2) (foo 3)) (b (foo 1)))", "(bar 2)");
        TregexTest.runTest(imfollowing, "(a (foo 1) (baz 2) (bar 3))", "(bar 3)");
        TregexTest.runTest(imfollowing, "(a (baz 1) (foo 2) (bar 3))", "(foo 2)");
    }

    public void testCategoryFunctions() {
        Function<String, String> fooCategory = new Function<String, String>(){

            @Override
            public String apply(String label) {
                if (label == null) {
                    return label;
                }
                if (label.equals("bar")) {
                    return "foo";
                }
                return label;
            }
        };
        TregexPatternCompiler fooCompiler = new TregexPatternCompiler(fooCategory);
        TregexPattern fooTregex = fooCompiler.compile("@foo > bar");
        TregexTest.runTest(fooTregex, "(bar (foo 0))", "(foo 0)");
        TregexTest.runTest(fooTregex, "(bar (bar 0))", "(bar 0)");
        TregexTest.runTest(fooTregex, "(foo (foo 0))", new String[0]);
        TregexTest.runTest(fooTregex, "(foo (bar 0))", new String[0]);
        Function<String, String> barCategory = new Function<String, String>(){

            @Override
            public String apply(String label) {
                if (label == null) {
                    return label;
                }
                if (label.equals("foo")) {
                    return "bar";
                }
                return label;
            }
        };
        TregexPatternCompiler barCompiler = new TregexPatternCompiler(barCategory);
        TregexPattern barTregex = barCompiler.compile("@bar > foo");
        TregexTest.runTest(barTregex, "(bar (foo 0))", new String[0]);
        TregexTest.runTest(barTregex, "(bar (bar 0))", new String[0]);
        TregexTest.runTest(barTregex, "(foo (foo 0))", "(foo 0)");
        TregexTest.runTest(barTregex, "(foo (bar 0))", "(bar 0)");
        TregexTest.runTest(fooTregex, "(bar (foo 0))", "(foo 0)");
        TregexTest.runTest(fooTregex, "(bar (bar 0))", "(bar 0)");
        TregexTest.runTest(fooTregex, "(foo (foo 0))", new String[0]);
        TregexTest.runTest(fooTregex, "(foo (bar 0))", new String[0]);
    }

    public void testCategoryDisjunction() {
        Function<String, String> abCategory = new Function<String, String>(){

            @Override
            public String apply(String label) {
                if (label == null) {
                    return label;
                }
                if (label.startsWith("a")) {
                    return "aaa";
                }
                if (label.startsWith("b")) {
                    return "bbb";
                }
                return label;
            }
        };
        TregexPatternCompiler abCompiler = new TregexPatternCompiler(abCategory);
        TregexPattern aaaTregex = abCompiler.compile("foo > @aaa");
        TregexTest.runTest(aaaTregex, "(aaa (foo 0))", "(foo 0)");
        TregexTest.runTest(aaaTregex, "(abc (foo 0))", "(foo 0)");
        TregexTest.runTest(aaaTregex, "(bbb (foo 0))", new String[0]);
        TregexTest.runTest(aaaTregex, "(bcd (foo 0))", new String[0]);
        TregexTest.runTest(aaaTregex, "(ccc (foo 0))", new String[0]);
        TregexPattern bbbTregex = abCompiler.compile("foo > @bbb");
        TregexTest.runTest(bbbTregex, "(aaa (foo 0))", new String[0]);
        TregexTest.runTest(bbbTregex, "(abc (foo 0))", new String[0]);
        TregexTest.runTest(bbbTregex, "(bbb (foo 0))", "(foo 0)");
        TregexTest.runTest(bbbTregex, "(bcd (foo 0))", "(foo 0)");
        TregexTest.runTest(bbbTregex, "(ccc (foo 0))", new String[0]);
        TregexPattern bothTregex = abCompiler.compile("foo > @aaa|bbb");
        TregexTest.runTest(bothTregex, "(aaa (foo 0))", "(foo 0)");
        TregexTest.runTest(bothTregex, "(abc (foo 0))", "(foo 0)");
        TregexTest.runTest(bothTregex, "(bbb (foo 0))", "(foo 0)");
        TregexTest.runTest(bothTregex, "(bcd (foo 0))", "(foo 0)");
        TregexTest.runTest(bothTregex, "(ccc (foo 0))", new String[0]);
    }

    public void testPrecedesDescribedChain() {
        TregexTest.runTest("DT .+(JJ) NN", "(NP (DT the) (JJ large) (JJ green) (NN house))", "(DT the)");
        TregexTest.runTest("DT .+(@JJ) /^NN/", "(NP (PDT both) (DT the) (JJ-SIZE large) (JJ-COLOUR green) (NNS houses))", "(DT the)");
        TregexTest.runTest("NN ,+(JJ) DT", "(NP (DT the) (JJ large) (JJ green) (NN house))", "(NN house)");
        TregexTest.runTest("NNS ,+(@JJ) /^DT/", "(NP (PDT both) (DT the) (JJ-SIZE large) (JJ-COLOUR green) (NNS houses))", "(NNS houses)");
        TregexTest.runTest("NNS ,+(/^(JJ|DT).*$/) PDT", "(NP (PDT both) (DT the) (JJ-SIZE large) (JJ-COLOUR green) (NNS houses))", "(NNS houses)");
        TregexTest.runTest("NNS ,+(@JJ) JJ", "(NP (PDT both) (DT the) (JJ large) (JJ-COLOUR green) (NNS houses))", "(NNS houses)");
    }

    public void testDominateDescribedChain() {
        TregexTest.runTest("foo <+(bar) baz", "(a (foo (baz)))", "(foo (baz))");
        TregexTest.runTest("foo <+(bar) baz", "(a (foo (bar (baz))))", "(foo (bar (baz)))");
        TregexTest.runTest("foo <+(bar) baz", "(a (foo (bar (bar (baz)))))", "(foo (bar (bar (baz))))");
        TregexTest.runTest("foo <+(bar) baz", "(a (foo (bif (baz))))", new String[0]);
        TregexTest.runTest("foo <+(!bif) baz", "(a (foo (bif (baz))))", new String[0]);
        TregexTest.runTest("foo <+(!bif) baz", "(a (foo (bar (baz))))", "(foo (bar (baz)))");
        TregexTest.runTest("foo <+(/b/) baz", "(a (foo (bif (baz))))", "(foo (bif (baz)))");
        TregexTest.runTest("foo <+(/b/) baz", "(a (foo (bar (bif (baz)))))", "(foo (bar (bif (baz))))");
        TregexTest.runTest("foo <+(bar) baz", "(a (foo (bar (blah 1) (bar (baz)))))", "(foo (bar (blah 1) (bar (baz))))");
        TregexTest.runTest("baz >+(bar) foo", "(a (foo (baz)))", "(baz)");
        TregexTest.runTest("baz >+(bar) foo", "(a (foo (bar (baz))))", "(baz)");
        TregexTest.runTest("baz >+(bar) foo", "(a (foo (bar (bar (baz)))))", "(baz)");
        TregexTest.runTest("baz >+(bar) foo", "(a (foo (bif (baz))))", new String[0]);
        TregexTest.runTest("baz >+(!bif) foo", "(a (foo (bif (baz))))", new String[0]);
        TregexTest.runTest("baz >+(!bif) foo", "(a (foo (bar (baz))))", "(baz)");
        TregexTest.runTest("baz >+(/b/) foo", "(a (foo (bif (baz))))", "(baz)");
        TregexTest.runTest("baz >+(/b/) foo", "(a (foo (bar (bif (baz)))))", "(baz)");
        TregexTest.runTest("baz >+(bar) foo", "(a (foo (bar (blah 1) (bar (baz)))))", "(baz)");
    }

    public void testSegmentedAndEqualsExpressions() {
        TregexTest.runTest("foo : bar", "(a (foo) (bar))", "(foo)");
        TregexTest.runTest("foo : bar", "(a (foo))", new String[0]);
        TregexTest.runTest("(foo << bar) : (foo << baz)", "(a (foo (bar 1)) (foo (baz 2)))", "(foo (bar 1))");
        TregexTest.runTest("(foo << bar) : (foo << baz)", "(a (foo (bar 1)) (foo (baz 2)))", "(foo (bar 1))");
        TregexTest.runTest("(foo << bar) == (foo << baz)", "(a (foo (bar)) (foo (baz)))", new String[0]);
        TregexTest.runTest("(foo << bar) : (foo << baz)", "(a (foo (bar) (baz)))", "(foo (bar) (baz))");
        TregexTest.runTest("(foo << bar) == (foo << baz)", "(a (foo (bar) (baz)))", "(foo (bar) (baz))");
        TregexTest.runTest("(foo << bar) : (baz >> a)", "(a (foo (bar) (baz)))", "(foo (bar) (baz))");
        TregexTest.runTest("(foo << bar) == (baz >> a)", "(a (foo (bar) (baz)))", new String[0]);
        TregexTest.runTest("foo == foo", "(a (foo (bar)))", "(foo (bar))");
        TregexTest.runTest("foo << bar == foo", "(a (foo (bar)) (foo (baz)))", "(foo (bar))");
        TregexTest.runTest("foo << bar == foo", "(a (foo (bar) (baz)))", "(foo (bar) (baz))");
        TregexTest.runTest("foo << bar == foo << baz", "(a (foo (bar) (baz)))", "(foo (bar) (baz))");
        TregexTest.runTest("foo << bar : (foo << baz)", "(a (foo (bar)) (foo (baz)))", "(foo (bar))");
    }

    public void testTwoChildren() {
        TregexTest.runTest("foo << bar << baz", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))");
        TregexTest.runTest("foo << __ << baz", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))");
        TregexTest.runTest("foo << bar << __", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))");
        TregexTest.runTest("foo << __ << __", "(foo (bar 1))", "(foo (bar 1))", "(foo (bar 1))", "(foo (bar 1))", "(foo (bar 1))");
        TregexTest.runTest("foo << __=a << __=b", "(foo (bar 1))", "(foo (bar 1))", "(foo (bar 1))", "(foo (bar 1))", "(foo (bar 1))");
        TregexTest.runTest("foo << __ << __", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))");
        TregexTest.runTest("(foo << __=a << __=b) : (=a !== =b)", "(foo (bar 1))", "(foo (bar 1))", "(foo (bar 1))");
        TregexTest.runTest("(foo < __=a < __=b) : (=a !== =b)", "(foo (bar 1))", new String[0]);
        TregexTest.runTest("(foo << __=a << __=b) : (=a !== =b)", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))", "(foo (bar 1) (baz 2))");
        TregexTest.runTest("(foo << __=a << __=b << __=c) : (=a !== =b) : (=a !== =c) : (=b !== =c)", "(foo (bar 1))", new String[0]);
    }

    public void testDocExamples() {
        TregexTest.runTest("S < VP < NP", "(S (VP) (NP))", "(S (VP) (NP))");
        TregexTest.runTest("S < VP < NP", "(a (S (VP) (NP)) (S (NP) (VP)))", "(S (VP) (NP))", "(S (NP) (VP))");
        TregexTest.runTest("S < VP < NP", "(S (VP (NP)))", new String[0]);
        TregexTest.runTest("S < VP & < NP", "(S (VP) (NP))", "(S (VP) (NP))");
        TregexTest.runTest("S < VP & < NP", "(a (S (VP) (NP)) (S (NP) (VP)))", "(S (VP) (NP))", "(S (NP) (VP))");
        TregexTest.runTest("S < VP & < NP", "(S (VP (NP)))", new String[0]);
        TregexTest.runTest("S < VP << NP", "(S (VP (NP)))", "(S (VP (NP)))");
        TregexTest.runTest("S < VP << NP", "(S (VP) (foo NP))", "(S (VP) (foo NP))");
        TregexTest.runTest("S < (VP < NP)", "(S (VP (NP)))", "(S (VP (NP)))");
        TregexTest.runTest("S < (NP $++ VP)", "(S (NP) (VP))", "(S (NP) (VP))");
        TregexTest.runTest("S < (NP $++ VP)", "(S (NP VP))", new String[0]);
        TregexTest.runTest("(NP < NN | < NNS)", "((NP NN) (NP foo) (NP NNS))", "(NP NN)", "(NP NNS)");
        TregexTest.runTest("(NP (< NN | < NNS) & > S)", "(foo (S (NP NN) (NP foo) (NP NNS)) (NP NNS))", "(NP NN)", "(NP NNS)");
        TregexTest.runTest("(NP [< NN | < NNS] & > S)", "(foo (S (NP NN) (NP foo) (NP NNS)) (NP NNS))", "(NP NN)", "(NP NNS)");
    }

    public void testMonthDayYear() {
        String MONTH_REGEX = "January|February|March|April|May|June|July|August|September|October|November|December|Jan\\.|Feb\\.|Mar\\.|Apr\\.|Aug\\.|Sep\\.|Sept\\.|Oct\\.|Nov\\.|Dec\\.";
        String testPattern = "NP=root <1 (NP=monthdayroot <1 (NNP=month <: /" + MONTH_REGEX + "/) <2 (CD=day <: __)) <2 (/^,$/=comma <: /^,$/) <3 (NP=yearroot <: (CD=year <: __)) : (=root <- =yearroot) : (=monthdayroot <- =day)";
        TregexTest.runTest(testPattern, "(ROOT (S (NP (NNP Mr.) (NNP Good)) (VP (VBZ devotes) (NP (RB much) (JJ serious) (NN space)) (PP (TO to) (NP (NP (DT the) (NNS events)) (PP (IN of) (NP (NP (NP (NNP Feb.) (CD 25)) (, ,) (NP (CD 1942))) (, ,) (SBAR (WHADVP (WRB when)) (S (NP (JJ American) (NNS gunners)) (VP (VBD spotted) (NP (NP (JJ strange) (NNS lights)) (PP (IN in) (NP (NP (DT the) (NN sky)) (PP (IN above) (NP (NNP Los) (NNP Angeles)))))))))))))) (. .)))", "(NP (NP (NNP Feb.) (CD 25)) (, ,) (NP (CD 1942)))");
        TregexTest.runTest(testPattern, "(ROOT (S (NP (DT The) (JJ preferred) (NNS shares)) (VP (MD will) (VP (VB carry) (NP (NP (DT a) (JJ floating) (JJ annual) (NN dividend)) (ADJP (JJ equal) (PP (TO to) (NP (NP (CD 72) (NN %)) (PP (IN of) (NP (NP (DT the) (JJ 30-day) (NNS bankers) (POS ')) (NN acceptance) (NN rate))))))) (PP (IN until) (NP (NP (NNP Dec.) (CD 31)) (, ,) (NP (CD 1994)))))) (. .)))", "(NP (NP (NNP Dec.) (CD 31)) (, ,) (NP (CD 1994)))");
        TregexTest.runTest(testPattern, "(ROOT (S (NP (PRP It)) (VP (VBD said) (SBAR (S (NP (NN debt)) (VP (VBD remained) (PP (IN at) (NP (NP (DT the) (QP ($ $) (CD 1.22) (CD billion))) (SBAR (WHNP (DT that)) (S (VP (VBZ has) (VP (VBD prevailed) (PP (IN since) (NP (JJ early) (CD 1989))))))))) (, ,) (SBAR (IN although) (S (NP (IN that)) (VP (VBN compared) (PP (IN with) (NP (NP (QP ($ $) (CD 911) (CD million))) (PP (IN at) (NP (NP (NNP Sept.) (CD 30)) (, ,) (NP (CD 1988))))))))))))) (. .)))", "(NP (NP (NNP Sept.) (CD 30)) (, ,) (NP (CD 1988)))");
        TregexTest.runTest(testPattern, "(ROOT (S (NP (DT The) (JJ new) (NNS notes)) (VP (MD will) (VP (VB bear) (NP (NN interest)) (PP (PP (IN at) (NP (NP (CD 5.5) (NN %)) (PP (IN through) (NP (NP (NNP July) (CD 31)) (, ,) (NP (CD 1991)))))) (, ,) (CC and) (ADVP (RB thereafter)) (PP (IN at) (NP (CD 10) (NN %)))))) (. .)))", "(NP (NP (NNP July) (CD 31)) (, ,) (NP (CD 1991)))");
        TregexTest.runTest(testPattern, "(ROOT (S (NP (NP (NNP Francis) (NNP M.) (NNP Wheat)) (, ,) (NP (NP (DT a) (JJ former) (NNPS Securities)) (CC and) (NP (NNP Exchange) (NNP Commission) (NN member))) (, ,)) (VP (VBD headed) (NP (NP (DT the) (NN panel)) (SBAR (WHNP (WDT that)) (S (VP (VBD had) (VP (VP (VBN studied) (NP (DT the) (NNS issues)) (PP (IN for) (NP (DT a) (NN year)))) (CC and) (VP (VBD proposed) (NP (DT the) (NNP FASB)) (PP (IN on) (NP (NP (NNP March) (CD 30)) (, ,) (NP (CD 1972))))))))))) (. .)))", "(NP (NP (NNP March) (CD 30)) (, ,) (NP (CD 1972)))");
        TregexTest.runTest(testPattern, "(ROOT (S (NP (DT The) (NNP FASB)) (VP (VBD had) (NP (PRP$ its) (JJ initial) (NN meeting)) (PP (IN on) (NP (NP (NNP March) (CD 28)) (, ,) (NP (CD 1973))))) (. .)))", "(NP (NP (NNP March) (CD 28)) (, ,) (NP (CD 1973)))");
        TregexTest.runTest(testPattern, "(ROOT (S (S (PP (IN On) (NP (NP (NNP Dec.) (CD 13)) (, ,) (NP (CD 1973)))) (, ,) (NP (PRP it)) (VP (VBD issued) (NP (PRP$ its) (JJ first) (NN rule)))) (: ;) (S (NP (PRP it)) (VP (VBD required) (S (NP (NNS companies)) (VP (TO to) (VP (VB disclose) (NP (NP (JJ foreign) (NN currency) (NNS translations)) (PP (IN in) (NP (NNP U.S.) (NNS dollars))))))))) (. .)))", "(NP (NP (NNP Dec.) (CD 13)) (, ,) (NP (CD 1973)))");
        TregexTest.runTest(testPattern, "(ROOT (S (NP (NP (NNP Fidelity) (NNPS Investments)) (, ,) (NP (NP (DT the) (NN nation) (POS 's)) (JJS largest) (NN fund) (NN company)) (, ,)) (VP (VBD said) (SBAR (S (NP (NN phone) (NN volume)) (VP (VBD was) (NP (NP (QP (RBR more) (IN than) (JJ double)) (PRP$ its) (JJ typical) (NN level)) (, ,) (CC but) (ADVP (RB still)) (NP (NP (NN half) (DT that)) (PP (IN of) (NP (NP (NNP Oct.) (CD 19)) (, ,) (NP (CD 1987)))))))))) (. .)))", "(NP (NP (NNP Oct.) (CD 19)) (, ,) (NP (CD 1987)))");
        TregexTest.runTest(testPattern, "(ROOT (S (NP (JJ SOFT) (NN CONTACT) (NNS LENSES)) (VP (VP (VBP WON) (NP (JJ federal) (NN blessing)) (PP (IN on) (NP (NP (NNP March) (CD 18)) (, ,) (NP (CD 1971))))) (, ,) (CC and) (VP (ADVP (RB quickly)) (VBD became) (NP (NN eye) (NNS openers)) (PP (IN for) (NP (PRP$ their) (NNS makers))))) (. .)))", "(NP (NP (NNP March) (CD 18)) (, ,) (NP (CD 1971)))");
        TregexTest.runTest(testPattern, "(ROOT (NP (NP (NP (VBN Annualized) (NN interest) (NNS rates)) (PP (IN on) (NP (JJ certain) (NNS investments))) (SBAR (IN as) (S (VP (VBN reported) (PP (IN by) (NP (DT the) (NNP Federal) (NNP Reserve) (NNP Board))) (PP (IN on) (NP (DT a) (JJ weekly-average) (NN basis))))))) (: :) (NP-TMP (NP (CD 1989)) (CC and) (NP (NP (NNP Wednesday)) (NP (NP (NNP October) (CD 4)) (, ,) (NP (CD 1989))))) (. .)))", "(NP (NP (NNP October) (CD 4)) (, ,) (NP (CD 1989)))");
        TregexTest.runTest(testPattern, "(ROOT (S (S (ADVP (RB Together))) (, ,) (NP (DT the) (CD two) (NNS stocks)) (VP (VP (VBD wreaked) (NP (NN havoc)) (PP (IN among) (NP (NN takeover) (NN stock) (NNS traders)))) (, ,) (CC and) (VP (VBD caused) (NP (NP (DT a) (ADJP (CD 7.3) (NN %)) (NN drop)) (PP (IN in) (NP (DT the) (NNP Dow) (NNP Jones) (NNP Transportation) (NNP Average))) (, ,) (ADJP (JJ second) (PP (IN in) (NP (NN size))) (PP (RB only) (TO to) (NP (NP (DT the) (NN stock-market) (NN crash)) (PP (IN of) (NP (NP (NNP Oct.) (CD 19)) (, ,) (NP (CD 1987)))))))))) (. .)))", "(NP (NP (NNP Oct.) (CD 19)) (, ,) (NP (CD 1987)))");
    }

    public void testComplex() {
        String testPattern = "S < (NP=m1 $.. (VP < ((/VB/ < /^(am|are|is|was|were|'m|'re|'s|be)$/) $.. NP=m2)))";
        String testTree = "(S (NP (NP (DT The) (JJ next) (NN stop)) (PP (IN on) (NP (DT the) (NN itinerary)))) (VP (VBD was) (NP (NP (NNP Chad)) (, ,) (SBAR (WHADVP (WRB where)) (S (NP (NNP Chen)) (VP (VBD dined) (PP (IN with) (NP (NP (NNP Chad) (POS 's)) (NNP President) (NNP Idris) (NNP Debi)))))))) (. .))";
        TregexTest.runTest(testPattern, "(ROOT " + testTree + ")", testTree);
        testTree = "(S (NP (NNP Chen) (NNP Shui) (HYPH -) (NNP bian)) (VP (VBZ is) (NP (NP (DT the) (JJ first) (NML (NNP ROC) (NN president))) (SBAR (S (ADVP (RB ever)) (VP (TO to) (VP (VB travel) (PP (IN to) (NP (JJ western) (NNP Africa))))))))) (. .))";
        TregexTest.runTest(testPattern, "(ROOT " + testTree + ")", testTree);
        testTree = "(ROOT (S (NP (PRP$ My) (NN dog)) (VP (VBZ is) (VP (VBG eating) (NP (DT a) (NN sausage)))) (. .)))";
        TregexTest.runTest(testPattern, testTree, new String[0]);
        testTree = "(ROOT (S (NP (PRP He)) (VP (MD will) (VP (VB be) (ADVP (RB here) (RB soon)))) (. .)))";
        TregexTest.runTest(testPattern, testTree, new String[0]);
        testPattern = "/^NP(?:-TMP|-ADV)?$/=m1 < (NP=m2 $- /^,$/ $-- NP=m3 !$ CC|CONJP)";
        testTree = "(ROOT (S (NP (NP (NP (NP (DT The) (NNP ROC) (POS 's)) (NN ambassador)) (PP (IN to) (NP (NNP Nicaragua)))) (, ,) (NP (NNP Antonio) (NNP Tsai)) (, ,)) (ADVP (RB bluntly)) (VP (VBD argued) (PP (IN in) (NP (NP (DT a) (NN briefing)) (PP (IN with) (NP (NNP Chen))))) (SBAR (IN that) (S (NP (NP (NP (NNP Taiwan) (POS 's)) (JJ foreign) (NN assistance)) (PP (IN to) (NP (NNP Nicaragua)))) (VP (VBD was) (VP (VBG being) (ADJP (JJ misused))))))) (. .)))";
        String expectedResult = "(NP (NP (NP (NP (DT The) (NNP ROC) (POS 's)) (NN ambassador)) (PP (IN to) (NP (NNP Nicaragua)))) (, ,) (NP (NNP Antonio) (NNP Tsai)) (, ,))";
        TregexTest.runTest(testPattern, testTree, expectedResult);
        testTree = "(ROOT (S (PP (IN In) (NP (NP (DT the) (NN opinion)) (PP (IN of) (NP (NP (NNP Norman) (NNP Hsu)) (, ,) (NP (NP (NN vice) (NN president)) (PP (IN of) (NP (NP (DT a) (NNS foods) (NN company)) (SBAR (WHNP (WHNP (WP$ whose) (NN family)) (PP (IN of) (NP (CD four)))) (S (VP (VBD had) (VP (VBN spent) (NP (QP (DT a) (JJ few)) (NNS years)) (PP (IN in) (NP (NNP New) (NNP Zealand))) (PP (IN before) (S (VP (VBG moving) (PP (IN to) (NP (NNP Dongguan))))))))))))))))) (, ,) (`` \") (NP (NP (DT The) (JJ first) (NN thing)) (VP (TO to) (VP (VB do)))) (VP (VBZ is) (S (VP (VB ask) (NP (DT the) (NNS children)) (NP (PRP$ their) (NN reason)) (PP (IN for) (S (VP (VBG saying) (NP (JJ such) (NNS things)))))))) (. .)))";
        expectedResult = "(NP (NP (NNP Norman) (NNP Hsu)) (, ,) (NP (NP (NN vice) (NN president)) (PP (IN of) (NP (NP (DT a) (NNS foods) (NN company)) (SBAR (WHNP (WHNP (WP$ whose) (NN family)) (PP (IN of) (NP (CD four)))) (S (VP (VBD had) (VP (VBN spent) (NP (QP (DT a) (JJ few)) (NNS years)) (PP (IN in) (NP (NNP New) (NNP Zealand))) (PP (IN before) (S (VP (VBG moving) (PP (IN to) (NP (NNP Dongguan))))))))))))))";
        TregexTest.runTest(testPattern, testTree, expectedResult);
        testTree = "(ROOT (S (NP (NP (NNP Banana)) (, ,) (NP (NN orange)) (, ,) (CC and) (NP (NN apple))) (VP (VBP are) (NP (NNS fruits))) (. .)))";
        TregexTest.runTest(testPattern, testTree, new String[0]);
        testTree = "(ROOT (S (NP (PRP He)) (, ,) (ADVP (RB however)) (, ,) (VP (VBZ does) (RB not) (VP (VB look) (ADJP (JJ fine)))) (. .)))";
        TregexTest.runTest(testPattern, testTree, new String[0]);
    }

    public void testComplex2() {
        String[] inputTrees = new String[]{"(ROOT (S (NP (PRP You)) (VP (VBD did) (VP (VB go) (WHADVP (WRB How) (JJ long)) (PP (IN for)))) (. .)))", "(ROOT (S (NP (NNS Raccoons)) (VP (VBP do) (VP (VB come) (WHADVP (WRB When)) (PRT (RP out)))) (. .)))", "(ROOT (S (NP (PRP She)) (VP (VBZ is) (VP (WHADVP (WRB Where)) (VBG working))) (. .)))", "(ROOT (S (NP (PRP You)) (VP (VBD did) (VP (WHNP (WP What)) (VB do))) (. .)))", "(ROOT (S (NP (PRP You)) (VP (VBD did) (VP (VB do) (PP (IN in) (NP (NNP Australia))) (WHNP (WP What)))) (. .)))"};
        String pattern = "WHADVP=whadvp > VP $+ /[A-Z]*/=last ![$++ (PP < NP)]";
        TregexTest.runTest(pattern, inputTrees[0], "(WHADVP (WRB How) (JJ long))");
        TregexTest.runTest(pattern, inputTrees[1], "(WHADVP (WRB When))");
        TregexTest.runTest(pattern, inputTrees[2], "(WHADVP (WRB Where))");
        TregexTest.runTest(pattern, inputTrees[3], new String[0]);
        TregexTest.runTest(pattern, inputTrees[4], new String[0]);
        pattern = "VP < (/^WH/=wh $++ /^VB/=vb)";
        TregexTest.runTest(pattern, inputTrees[0], new String[0]);
        TregexTest.runTest(pattern, inputTrees[1], new String[0]);
        TregexTest.runTest(pattern, inputTrees[2], "(VP (WHADVP (WRB Where)) (VBG working))");
        TregexTest.runTest(pattern, inputTrees[3], "(VP (WHNP (WP What)) (VB do))");
        TregexTest.runTest(pattern, inputTrees[4], new String[0]);
        pattern = "PP=pp > VP $+ WHNP=whnp";
        TregexTest.runTest(pattern, inputTrees[0], new String[0]);
        TregexTest.runTest(pattern, inputTrees[1], new String[0]);
        TregexTest.runTest(pattern, inputTrees[2], new String[0]);
        TregexTest.runTest(pattern, inputTrees[3], new String[0]);
        TregexTest.runTest(pattern, inputTrees[4], "(PP (IN in) (NP (NNP Australia)))");
    }

    public void testNamed() {
        Tree tree = TregexTest.treeFromString("(a (foo 1) (bar 2) (bar 3))");
        TregexPattern pattern = TregexPattern.compile("foo=a $ bar=b");
        TregexMatcher matcher = pattern.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)"(foo 1)", (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"(foo 1)", (String)matcher.getNode("a").toString());
        TregexTest.assertEquals((String)"(bar 2)", (String)matcher.getNode("b").toString());
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)"(foo 1)", (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"(foo 1)", (String)matcher.getNode("a").toString());
        TregexTest.assertEquals((String)"(bar 3)", (String)matcher.getNode("b").toString());
        TregexTest.assertFalse((boolean)matcher.find());
    }

    public void testLink() {
        TregexTest.runTest("bar $- (bar $- foo)", "(a (foo 1) (bar 2) (bar 3))", "(bar 3)");
        TregexTest.runTest("bar=a $- (~a $- foo)", "(a (foo 1) (bar 2) (bar 3))", "(bar 3)");
        TregexTest.runTest("bar=a $- (=a $- foo)", "(a (foo 1) (bar 2) (bar 3))", new String[0]);
        TregexTest.runTest("bar=a $- (~a=b $- foo)", "(a (foo 1) (bar 2) (bar 3))", "(bar 3)");
        Tree tree = TregexTest.treeFromString("(a (foo 1) (bar 2) (bar 3))");
        TregexPattern pattern = TregexPattern.compile("bar=a $- (~a $- foo)");
        TregexMatcher matcher = pattern.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)"(bar 3)", (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"(bar 3)", (String)matcher.getNode("a").toString());
        TregexTest.assertFalse((boolean)matcher.find());
        tree = TregexTest.treeFromString("(a (foo 1) (bar 2) (bar 3))");
        pattern = TregexPattern.compile("bar=a $- (~a=b $- foo)");
        matcher = pattern.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)"(bar 3)", (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"(bar 3)", (String)matcher.getNode("a").toString());
        TregexTest.assertEquals((String)"(bar 2)", (String)matcher.getNode("b").toString());
        TregexTest.assertFalse((boolean)matcher.find());
        tree = TregexTest.treeFromString("(a (foo 1) (bar 2) (bar 3))");
        pattern = TregexPattern.compile("bar=a $- (~a=b $- foo=c)");
        matcher = pattern.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)"(bar 3)", (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"(bar 3)", (String)matcher.getNode("a").toString());
        TregexTest.assertEquals((String)"(bar 2)", (String)matcher.getNode("b").toString());
        TregexTest.assertEquals((String)"(foo 1)", (String)matcher.getNode("c").toString());
        TregexTest.assertFalse((boolean)matcher.find());
    }

    public void testNonsense() {
        try {
            TregexPattern pattern = TregexPattern.compile("foo=a $ bar=a");
            throw new RuntimeException("Expected a parse exception");
        }
        catch (TregexParseException pattern) {
            try {
                TregexPattern pattern2 = TregexPattern.compile("foo=a > bar=b $ ~a=b");
                throw new RuntimeException("Expected a parse exception");
            }
            catch (TregexParseException pattern2) {
                TregexPattern.compile("foo=a > bar=b $ ~a");
                try {
                    TregexPattern pattern3 = TregexPattern.compile("~a $- (bar=a $- foo)");
                    throw new RuntimeException("Expected a parse exception");
                }
                catch (TregexParseException pattern3) {
                    try {
                        TregexPattern pattern4 = TregexPattern.compile("=a $- (bar=a $- foo)");
                        throw new RuntimeException("Expected a parse exception");
                    }
                    catch (TregexParseException pattern4) {
                        try {
                            TregexPattern pattern522 = TregexPattern.compile("~a=a $- (bar=b $- foo)");
                            throw new RuntimeException("Expected a parse exception");
                        }
                        catch (TregexParseException pattern522) {
                            TregexPattern.compile("foo=a : ~a");
                            TregexPattern.compile("a < foo=a | < bar=a");
                            try {
                                TregexPattern.compile("a < foo=a | < ~a");
                                throw new RuntimeException("Expected a parse exception");
                            }
                            catch (TregexParseException pattern522) {
                                try {
                                    TregexPattern.compile("a < foo=a | < =a");
                                    throw new RuntimeException("Expected a parse exception");
                                }
                                catch (TregexParseException pattern522) {
                                    try {
                                        TregexPattern pattern6 = TregexPattern.compile("__ ! > __=a");
                                        throw new RuntimeException("Expected a parse exception");
                                    }
                                    catch (TregexParseException tregexParseException) {
                                        return;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    public void testHeadOfPhrase() {
        TregexTest.runTest("NP <# NNS", "(NP (NN work) (NNS practices))", "(NP (NN work) (NNS practices))");
        TregexTest.runTest("NP <# NN", "(NP (NN work) (NNS practices))", new String[0]);
        TregexTest.runTest("NP <<# NNS", "(NP (NP (NN work) (NNS practices)) (PP (IN in) (NP (DT the) (JJ former) (NNP Soviet) (NNP Union))))", "(NP (NP (NN work) (NNS practices)) (PP (IN in) (NP (DT the) (JJ former) (NNP Soviet) (NNP Union))))", "(NP (NN work) (NNS practices))");
        TregexTest.runTest("NP !<# NNS <<# NNS", "(NP (NP (NN work) (NNS practices)) (PP (IN in) (NP (DT the) (JJ former) (NNP Soviet) (NNP Union))))", "(NP (NP (NN work) (NNS practices)) (PP (IN in) (NP (DT the) (JJ former) (NNP Soviet) (NNP Union))))");
        TregexTest.runTest("NP !<# NNP <<# NNP", "(NP (NP (NN work) (NNS practices)) (PP (IN in) (NP (DT the) (JJ former) (NNP Soviet) (NNP Union))))", new String[0]);
        TregexTest.runTest("NNS ># NP", "(NP (NP (NN work) (NNS practices)) (PP (IN in) (NP (DT the) (JJ former) (NNP Soviet) (NNP Union))))", "(NNS practices)");
        TregexTest.runTest("NNS ># (NP < PP)", "(NP (NP (NN work) (NNS practices)) (PP (IN in) (NP (DT the) (JJ former) (NNP Soviet) (NNP Union))))", new String[0]);
        TregexTest.runTest("NNS >># (NP < PP)", "(NP (NP (NN work) (NNS practices)) (PP (IN in) (NP (DT the) (JJ former) (NNP Soviet) (NNP Union))))", "(NNS practices)");
        TregexTest.runTest("NP <<# /^NN/", "(NP (NP (NN work) (NNS practices)) (PP (IN in) (NP (DT the) (JJ former) (NNP Soviet) (NNP Union))))", "(NP (NP (NN work) (NNS practices)) (PP (IN in) (NP (DT the) (JJ former) (NNP Soviet) (NNP Union))))", "(NP (NN work) (NNS practices))", "(NP (DT the) (JJ former) (NNP Soviet) (NNP Union)))");
    }

    public void testOnlyMatchRoot() {
        String treeString = "(a (foo 1) (bar 2))";
        Tree tree = TregexTest.treeFromString(treeString);
        TregexPattern pattern = TregexPattern.compile("__=a ! > __");
        TregexMatcher matcher = pattern.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)treeString, (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)treeString, (String)matcher.getNode("a").toString());
        TregexTest.assertFalse((boolean)matcher.find());
    }

    public void testRepeatedVariables() {
        Tree tree = TregexTest.treeFromString("(root (a (foo 1)) (a (bar 2)))");
        TregexPattern pattern = TregexPattern.compile("a < foo=a | < bar=a");
        TregexMatcher matcher = pattern.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)"(a (foo 1))", (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"(foo 1)", (String)matcher.getNode("a").toString());
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)"(a (bar 2))", (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"(bar 2)", (String)matcher.getNode("a").toString());
        TregexTest.assertFalse((boolean)matcher.find());
    }

    public void testMoeCurlyLarry() {
        String testString = "(T (X (N (N Moe (PNT ,)))) (NP (X (N Curly)) (NP (CONJ and) (X (N Larry)))))";
        Tree tree = TregexTest.treeFromString(testString);
        TregexPattern pattern = TregexPattern.compile("PNT=p >>- (__=l >, (__=t <- (__=r <, __=m <- (__ <, CONJ <- __=z))))");
        TregexMatcher matcher = pattern.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)"(PNT ,)", (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"(PNT ,)", (String)matcher.getNode("p").toString());
        TregexTest.assertEquals((String)"(X (N (N Moe (PNT ,))))", (String)matcher.getNode("l").toString());
        TregexTest.assertEquals((String)testString, (String)matcher.getNode("t").toString());
        TregexTest.assertEquals((String)"(NP (X (N Curly)) (NP (CONJ and) (X (N Larry))))", (String)matcher.getNode("r").toString());
        TregexTest.assertEquals((String)"(X (N Curly))", (String)matcher.getNode("m").toString());
        TregexTest.assertEquals((String)"(X (N Larry))", (String)matcher.getNode("z").toString());
        TregexTest.assertFalse((boolean)matcher.find());
        pattern = TregexPattern.compile("PNT=p >>- (/(.+)/#1%var=l >, (__=t <- (__=r <, /(.+)/#1%var=m <- (__ <, CONJ <- /(.+)/#1%var=z))))");
        matcher = pattern.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)"(PNT ,)", (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"(PNT ,)", (String)matcher.getNode("p").toString());
        TregexTest.assertEquals((String)"(X (N (N Moe (PNT ,))))", (String)matcher.getNode("l").toString());
        TregexTest.assertEquals((String)testString, (String)matcher.getNode("t").toString());
        TregexTest.assertEquals((String)"(NP (X (N Curly)) (NP (CONJ and) (X (N Larry))))", (String)matcher.getNode("r").toString());
        TregexTest.assertEquals((String)"(X (N Curly))", (String)matcher.getNode("m").toString());
        TregexTest.assertEquals((String)"(X (N Larry))", (String)matcher.getNode("z").toString());
        TregexTest.assertFalse((boolean)matcher.find());
        pattern = TregexPattern.compile("PNT=p >>- (__=l >, (__=t <- (__=r <, ~l <- (__ <, CONJ <- ~l))))");
        matcher = pattern.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)"(PNT ,)", (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"(PNT ,)", (String)matcher.getNode("p").toString());
        TregexTest.assertEquals((String)"(X (N (N Moe (PNT ,))))", (String)matcher.getNode("l").toString());
        TregexTest.assertEquals((String)testString, (String)matcher.getNode("t").toString());
        TregexTest.assertEquals((String)"(NP (X (N Curly)) (NP (CONJ and) (X (N Larry))))", (String)matcher.getNode("r").toString());
        TregexTest.assertFalse((boolean)matcher.find());
        pattern = TregexPattern.compile("PNT=p >>- (__=l >, (__=t <- (__=r <, ~l=m <- (__ <, CONJ <- ~l=z))))");
        matcher = pattern.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)"(PNT ,)", (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"(PNT ,)", (String)matcher.getNode("p").toString());
        TregexTest.assertEquals((String)"(X (N (N Moe (PNT ,))))", (String)matcher.getNode("l").toString());
        TregexTest.assertEquals((String)testString, (String)matcher.getNode("t").toString());
        TregexTest.assertEquals((String)"(NP (X (N Curly)) (NP (CONJ and) (X (N Larry))))", (String)matcher.getNode("r").toString());
        TregexTest.assertEquals((String)"(X (N Curly))", (String)matcher.getNode("m").toString());
        TregexTest.assertEquals((String)"(X (N Larry))", (String)matcher.getNode("z").toString());
        TregexTest.assertFalse((boolean)matcher.find());
    }

    public void testChinese() {
        TregexPattern pattern = TregexPattern.compile("DEG|DEC < \u7684");
        TregexTest.runTest("DEG|DEC < \u7684", "(DEG (\u7684 1))", "(DEG (\u7684 1))");
    }

    public void testImmediateSister() {
        TregexTest.runTest("@NP < (/^,/=comma $+ CC)", "((NP NP , NP , NP , CC NP))", "(NP NP , NP , NP , CC NP)");
        TregexTest.runTest("@NP < (/^,/=comma $++ CC)", "((NP NP , NP , NP , CC NP))", "(NP NP , NP , NP , CC NP)", "(NP NP , NP , NP , CC NP)", "(NP NP , NP , NP , CC NP)");
        TregexTest.runTest("@NP < (@/^,/=comma $+ @CC)", "((NP NP , NP , NP , CC NP))", "(NP NP , NP , NP , CC NP)");
        TregexPattern pattern = TregexPattern.compile("@NP < (/^,/=comma $+ CC)");
        String treeString = "(NP NP (, 1) NP (, 2) NP (, 3) CC NP)";
        Tree tree = TregexTest.treeFromString(treeString);
        TregexMatcher matcher = pattern.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)treeString, (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"(, 3)", (String)matcher.getNode("comma").toString());
        TregexTest.assertFalse((boolean)matcher.find());
        treeString = "(NP NP , NP , NP , CC NP)";
        tree = TregexTest.treeFromString(treeString);
        matcher = pattern.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)treeString, (String)matcher.getMatch().toString());
        Tree node = matcher.getNode("comma");
        TregexTest.assertEquals((String)",", (String)node.toString());
        TregexTest.assertSame((Object)tree.children()[5], (Object)node);
        TregexTest.assertNotSame((Object)tree.children()[3], (Object)node);
        TregexTest.assertFalse((boolean)matcher.find());
    }

    public void testVariableGroups() {
        String treeString = "(albatross (foo 1) (bar 2))";
        Tree tree = TregexTest.treeFromString(treeString);
        TregexPattern pattern = TregexPattern.compile("/(.*)/#1%name < foo");
        TregexMatcher matcher = pattern.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)treeString, (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"albatross", (String)matcher.getVariableString("name"));
        TregexTest.assertFalse((boolean)matcher.find());
        pattern = TregexPattern.compile("/(.*)/#1%name < /foo(.*)/#1%blah");
        matcher = pattern.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)treeString, (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"albatross", (String)matcher.getVariableString("name"));
        TregexTest.assertEquals((String)"", (String)matcher.getVariableString("blah"));
        TregexTest.assertFalse((boolean)matcher.find());
        pattern = TregexPattern.compile("/(.*)/#1%name < (/(.*)/#1%blah < __)");
        matcher = pattern.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)treeString, (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"albatross", (String)matcher.getVariableString("name"));
        TregexTest.assertEquals((String)"foo", (String)matcher.getVariableString("blah"));
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)treeString, (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"albatross", (String)matcher.getVariableString("name"));
        TregexTest.assertEquals((String)"bar", (String)matcher.getVariableString("blah"));
        TregexTest.assertFalse((boolean)matcher.find());
        treeString = "(albatross (foo foo_albatross) (bar foo_albatross))";
        tree = TregexTest.treeFromString(treeString);
        pattern = TregexPattern.compile("/(.*)/#1%name < (/(.*)/#1%blah < /(.*)_(.*)/#2%name)");
        matcher = pattern.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)treeString, (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"albatross", (String)matcher.getVariableString("name"));
        TregexTest.assertEquals((String)"foo", (String)matcher.getVariableString("blah"));
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)treeString, (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"albatross", (String)matcher.getVariableString("name"));
        TregexTest.assertEquals((String)"bar", (String)matcher.getVariableString("blah"));
        TregexTest.assertFalse((boolean)matcher.find());
        pattern = TregexPattern.compile("/(.*)/#1%name < (/(.*)/#1%blah < /(.*)_(.*)/#1%blah#2%name)");
        matcher = pattern.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)treeString, (String)matcher.getMatch().toString());
        TregexTest.assertEquals((String)"albatross", (String)matcher.getVariableString("name"));
        TregexTest.assertEquals((String)"foo", (String)matcher.getVariableString("blah"));
        TregexTest.assertFalse((boolean)matcher.find());
    }

    public void testParenthesizedExpressions() {
        String[] treeStrings = new String[]{"( (S (S (PP (IN In) (NP (CD 1941) )) (, ,) (NP (NP (NNP Raeder) ) (CC and) (NP (DT the) (JJ German) (NN navy) )) (VP (VBD threatened) (S (VP (TO to) (VP (VB attack) (NP (DT the) (NNP Panama) (NNP Canal) )))))) (, ,) (RB so) (S (NP (PRP we) ) (VP (VBD created) (NP (NP (DT the) (NNP Southern) (NNP Command) ) (PP-LOC (IN in) (NP (NNP Panama) ))))) (. .) ))", "(S (S (NP-SBJ (NNP Japan) ) (VP (MD can) (VP (VP (VB grow) ) (CC and) (VP (RB not) (VB cut) (PRT (RB back) ))))) (, ,) (CC and) (RB so) (S (ADVP (RB too) ) (, ,) (NP (NP (NNP New) (NNP Zealand) )) ))))", "( (S (S (NP-SBJ (PRP You) ) (VP (VBP make) (NP (DT a) (NN forecast) ))) (, ,) (CC and) (RB then) (S (NP-SBJ (PRP you) ) (VP (VBP become) (NP-PRD (PRP$ its) (NN prisoner) ))) (. .)))"};
        Tree[] trees = TregexTest.treesFromString(treeStrings);
        TregexPattern pattern = TregexPattern.compile("/^S/ < (/^S/ $++ (/^[,]|CC|CONJP$/ $+ (RB=adv $+ /^S/)))");
        TregexMatcher matcher = pattern.matcher(trees[0]);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertFalse((boolean)matcher.find());
        matcher = pattern.matcher(trees[1]);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertFalse((boolean)matcher.find());
        matcher = pattern.matcher(trees[2]);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertFalse((boolean)matcher.find());
        pattern = TregexPattern.compile("/^S/ < (/^S/ $++ (/^[,]|CC|CONJP$/ (< and) $+ (RB=adv $+ /^S/)))");
        matcher = pattern.matcher(trees[0]);
        TregexTest.assertFalse((boolean)matcher.find());
        matcher = pattern.matcher(trees[1]);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertFalse((boolean)matcher.find());
        matcher = pattern.matcher(trees[2]);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertFalse((boolean)matcher.find());
        pattern = TregexPattern.compile("/^S/ < (/^S/ $++ (/^[,]|CC|CONJP$/ !(< and) $+ (RB=adv $+ /^S/)))");
        matcher = pattern.matcher(trees[0]);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertFalse((boolean)matcher.find());
        matcher = pattern.matcher(trees[1]);
        TregexTest.assertFalse((boolean)matcher.find());
        matcher = pattern.matcher(trees[2]);
        TregexTest.assertFalse((boolean)matcher.find());
        pattern = TregexPattern.compile("/^S/ < (/^S/ $++ (/^[,]|CC|CONJP$/ (< and $+ RB) $+ (RB=adv $+ /^S/)))");
        matcher = pattern.matcher(trees[0]);
        TregexTest.assertFalse((boolean)matcher.find());
        matcher = pattern.matcher(trees[1]);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertFalse((boolean)matcher.find());
        matcher = pattern.matcher(trees[2]);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertFalse((boolean)matcher.find());
        pattern = TregexPattern.compile("/^S/ < (/^S/ $++ (/^[,]|CC|CONJP$/ !(< and $+ RB) $+ (RB=adv $+ /^S/)))");
        matcher = pattern.matcher(trees[0]);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertFalse((boolean)matcher.find());
        matcher = pattern.matcher(trees[1]);
        TregexTest.assertFalse((boolean)matcher.find());
        matcher = pattern.matcher(trees[2]);
        TregexTest.assertFalse((boolean)matcher.find());
        pattern = TregexPattern.compile("/^S/ < (/^S/ $++ (/^[,]|CC|CONJP$/ !(< and $+ (RB < then)) $+ (RB=adv $+ /^S/)))");
        matcher = pattern.matcher(trees[0]);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertFalse((boolean)matcher.find());
        matcher = pattern.matcher(trees[1]);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertFalse((boolean)matcher.find());
        matcher = pattern.matcher(trees[2]);
        TregexTest.assertFalse((boolean)matcher.find());
    }

    public void testParentEquals() {
        TregexTest.runTest("A <= B", "(A (B 1))", "(A (B 1))");
        TregexTest.runTest("A <= A", "(A (A 1) (B 2))", "(A (A 1) (B 2))", "(A (A 1) (B 2))", "(A 1)");
        TregexTest.runTest("A <= (A < B)", "(A (A (B 1)))", "(A (A (B 1)))", "(A (B 1))");
        TregexTest.runTest("A <= (A < B)", "(A (A (B 1)) (A (C 2)))", "(A (A (B 1)) (A (C 2)))", "(A (B 1))");
        TregexTest.runTest("A <= (A < B)", "(A (A (C 2)))", new String[0]);
    }

    public void testRootDisjunction() {
        TregexTest.runTest("A | B", "(A (B 1))", "(A (B 1))", "(B 1)");
        TregexTest.runTest("(A) | (B)", "(A (B 1))", "(A (B 1))", "(B 1)");
        TregexTest.runTest("A < B | A < C", "(A (B 1) (C 2))", "(A (B 1) (C 2))", "(A (B 1) (C 2))");
        TregexTest.runTest("A < B | B < C", "(A (B 1) (C 2))", "(A (B 1) (C 2))");
        TregexTest.runTest("A < B | B < C", "(A (B (C 1)) (C 2))", "(A (B (C 1)) (C 2))", "(B (C 1))");
        TregexTest.runTest("A | B | C", "(A (B (C 1)) (C 2))", "(A (B (C 1)) (C 2))", "(B (C 1))", "(C 1)", "(C 2)");
        TregexTest.runTest("A < B | < C", "(A (B 1))", "(A (B 1))");
        TregexTest.runTest("A < B | < C", "(A (B 1) (C 2))", "(A (B 1) (C 2))", "(A (B 1) (C 2))");
        TregexTest.runTest("A < B | < C", "(B (C 1))", new String[0]);
    }

    public void testSubtreePattern() {
        TregexTest.runTest("A <... { B ; C ; D }", "(A (B 1) (C 2) (D 3))", "(A (B 1) (C 2) (D 3))");
        TregexTest.runTest("A <... { B ; C ; D }", "(Z (A (B 1) (C 2) (D 3)))", "(A (B 1) (C 2) (D 3))");
        TregexTest.runTest("A <... { B ; C ; D }", "(A (B 1) (C 2) (D 3) (E 4))", new String[0]);
        TregexTest.runTest("A <... { B ; C ; D }", "(A (E 4) (B 1) (C 2) (D 3))", new String[0]);
        TregexTest.runTest("A <... { B ; C ; D }", "(A (B 1) (C 2) (E 4) (D 3))", new String[0]);
        TregexTest.runTest("A <... { B ; C ; D }", "(A (B 1) (C 2))", new String[0]);
        TregexTest.runTest("A !<... { B ; C ; D }", "(A (B 1) (C 2) (D 3))", new String[0]);
        TregexTest.runTest("A !<... { B ; C ; D }", "(Z (A (B 1) (C 2) (D 3)))", new String[0]);
        TregexTest.runTest("A !<... { B ; C ; D }", "(A (B 1) (C 2) (D 3) (E 4))", "(A (B 1) (C 2) (D 3) (E 4))");
        TregexTest.runTest("A !<... { B ; C ; D }", "(A (E 4) (B 1) (C 2) (D 3))", "(A (E 4) (B 1) (C 2) (D 3))");
        TregexTest.runTest("A !<... { B ; C ; D }", "(A (B 1) (C 2) (E 4) (D 3))", "(A (B 1) (C 2) (E 4) (D 3))");
        TregexTest.runTest("A !<... { B ; C ; D }", "(A (B 1) (C 2))", "(A (B 1) (C 2))");
        TregexTest.runTest("A <... { (B < C) ; D }", "(A (B (C 2)) (D 3))", "(A (B (C 2)) (D 3))");
        TregexTest.runTest("A <... { (B <... { C ; D }) ; E }", "(A (B (C 2) (D 3)) (E 4))", "(A (B (C 2) (D 3)) (E 4))");
        TregexTest.runTest("A <... { (B !< C) ; D }", "(A (B (C 2)) (D 3))", new String[0]);
    }

    public void testDisjunctionVariableAssignments() {
        Tree tree = TregexTest.treeFromString("(NP (UCP (NNP U.S.) (CC and) (ADJP (JJ northern) (JJ European))) (NNS diplomats))");
        TregexPattern pattern = TregexPattern.compile("UCP [ <- (ADJP=adjp < JJR) | <, NNP=np ]");
        TregexMatcher matcher = pattern.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)"(NNP U.S.)", (String)matcher.getNode("np").toString());
        TregexTest.assertFalse((boolean)matcher.find());
    }

    public void testOptional() {
        Tree tree = TregexTest.treeFromString("(A (B (C 1)) (B 2))");
        TregexPattern pattern = TregexPattern.compile("B ? < C=c");
        TregexMatcher matcher = pattern.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)"(C 1)", (String)matcher.getNode("c").toString());
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals(null, (Object)matcher.getNode("c"));
        TregexTest.assertFalse((boolean)matcher.find());
        tree = TregexTest.treeFromString("(ROOT (INTJ (CC But) (S (NP (DT the) (NNP RTC)) (ADVP (RB also)) (VP (VBZ requires) (`` ``) (S (FRAG (VBG working) ('' '') (NP (NP (NN capital)) (S (VP (TO to) (VP (VB maintain) (SBAR (S (NP (NP (DT the) (JJ bad) (NNS assets)) (PP (IN of) (NP (NP (NNS thrifts)) (SBAR (WHNP (WDT that)) (S (VP (VBP are) (VBN sold) (, ,) (PP (IN until) (NP (DT the) (NNS assets))))))))) (VP (MD can) (VP (VB be) (VP (VBN sold) (ADVP (RB separately))))))))))))))) (S (VP (. .)))))");
        pattern = TregexPattern.compile("__ !> __ <- (__=top <- (__ <<- (/[.]|PU/=punc < /[.!?\u3002\uff01\uff1f]/ ?> (__=single <: =punc))))");
        matcher = pattern.matcher(tree);
        TregexTest.assertTrue((boolean)matcher.find());
        TregexTest.assertEquals((String)"(. .)", (String)matcher.getNode("punc").toString());
        TregexTest.assertEquals((String)"(VP (. .))", (String)matcher.getNode("single").toString());
        TregexTest.assertFalse((boolean)matcher.find());
    }

    public static void runTest(String pattern, String tree, String ... expectedResults) {
        TregexTest.runTest(TregexPattern.compile(pattern), tree, expectedResults);
    }

    public static void runTest(TregexPattern pattern, String tree, String ... expectedResults) {
        TreeTestExample test = new TreeTestExample(tree, expectedResults);
        test.runTest(pattern);
    }

    public static void outputResults(TregexPattern pattern, TreeTestExample ... tests) {
        for (TreeTestExample test : tests) {
            test.outputResults(pattern);
        }
    }

    public static void outputResults(String pattern, String ... trees) {
        for (String tree : trees) {
            TreeTestExample test = new TreeTestExample(tree, new String[0]);
            test.outputResults(TregexPattern.compile(pattern));
        }
    }

    public static class TreeTestExample {
        Tree input;
        Tree[] expectedOutput;

        public TreeTestExample(String input, String ... expectedOutput) {
            this.input = TregexTest.treeFromString(input);
            this.expectedOutput = new Tree[expectedOutput.length];
            for (int i = 0; i < expectedOutput.length; ++i) {
                this.expectedOutput[i] = TregexTest.treeFromString(expectedOutput[i]);
            }
        }

        public void outputResults(TregexPattern pattern) {
            System.out.println(pattern + " found the following matches on input " + this.input);
            TregexMatcher matcher = pattern.matcher(this.input);
            boolean output = false;
            while (matcher.find()) {
                output = true;
                System.out.println("  " + matcher.getMatch());
                Set<String> namesToNodes = matcher.getNodeNames();
                for (String name : namesToNodes) {
                    System.out.println("    " + name + ": " + matcher.getNode(name));
                }
            }
            if (!output) {
                System.out.println("  Nothing!  Absolutely nothing!");
            }
        }

        public void runTest(TregexPattern pattern) {
            IdentityHashMap matchedTrees = new IdentityHashMap();
            TregexMatcher matcher = pattern.matcher(this.input);
            for (int i = 0; i < this.expectedOutput.length; ++i) {
                try {
                    TestCase.assertTrue((boolean)matcher.find());
                }
                catch (AssertionFailedError e) {
                    throw new RuntimeException("Pattern " + pattern + " failed on input " + this.input.toString() + " [expected " + this.expectedOutput.length + " results, got " + i + "]", e);
                }
                Tree match = matcher.getMatch();
                String result = match.toString();
                String expectedResult = this.expectedOutput[i].toString();
                try {
                    TestCase.assertEquals((String)expectedResult, (String)result);
                }
                catch (AssertionFailedError e) {
                    throw new RuntimeException("Pattern " + pattern + " matched the wrong tree on input " + this.input.toString() + " [expected " + this.expectedOutput[i] + " got " + matcher.getMatch() + "]", e);
                }
                matchedTrees.put(match, null);
            }
            try {
                TestCase.assertFalse((boolean)matcher.find());
            }
            catch (AssertionFailedError e) {
                throw new RuntimeException("Pattern " + pattern + " failed on input " + this.input.toString() + " [expected " + this.expectedOutput.length + " results, got more than that]", e);
            }
            for (Tree subtree : this.input) {
                if (matchedTrees.containsKey(subtree)) {
                    TestCase.assertTrue((boolean)matcher.matchesAt(subtree));
                    continue;
                }
                TestCase.assertFalse((boolean)matcher.matchesAt(subtree));
            }
        }
    }
}

