<?php
// (c) Copyright by authors of the Tiki Wiki CMS Groupware Project
//
// All Rights Reserved. See copyright.txt for details and a complete list of authors.
// Licensed under the GNU LESSER GENERAL PUBLIC LICENSE. See license.txt for details.
// $Id: TikiFilter.php 66298 2018-05-07 23:30:07Z xorti $

/**
 * Class TikiFilter
 *
 * Just offers a get method to obtain an instance of a Zend\Filter\FilterInterface implementation, either stock (Zend) or custom.
 * The objects are "filters" in an extended sense. Data is not necessarily just filtered, but can be otherwise altered.
 * For example, special characters can be escaped.
 *
 * FIXME: The filter() method may perform lossy data alteration quietly, which complicates debugging. See https://github.com/zendframework/zend-filter/issues/63
 */
class TikiFilter
{
	/**
	 * Provides an object implementing Zend\Filter\FilterInterface based on the input
	 *
	 * @param \Zend\Filter\FilterInterface|string $filter		A filter shortcut name, or the filter itself.
	 * @return \Zend\Filter\FilterInterface 					The filter object requested
	 *
	 * @link https://dev.tiki.org/Filtering+Best+Practices
	 * @link https://zendframework.github.io/zend-filter/
	 */
	public static function get($filter)
	{
		if ($filter instanceof \Zend\Filter\FilterInterface) {
			return $filter;
		}

		switch ($filter) {
			case 'alpha':
				// Removes all but alphabetic characters. Unicode support.
				return new TikiFilter_Alpha;
			case 'alphaspace':
				// Removes all but alphabetic characters and spaces
				return new TikiFilter_Alpha(true);
			case 'word':
				// Strips everything but digit and alpha and underscore characters. Unicode support. eg. "g.4h&#Δ δ🍗_🍘コン" evaluates to "ghΔδ_コン"
				return new Zend\Filter\PregReplace('/\W+/', '');
			case 'wordspace':
				// Words and spaces only (no trimming)
				return new Zend\Filter\PregReplace('/[^\p{L}\p{M}\p{N}_\p{Zs}]*/u', '');
			case 'alnum':
				// Only alphabetic characters and digits. All other characters are suppressed. Unicode support.
				return new TikiFilter_Alnum;
			case 'alnumspace':
				// Only alphabetic characters, digits and spaces. All other characters are suppressed. Unicode support
				return new TikiFilter_Alnum(true);
			case 'alnumdash':
				// Removes everything except alphabetic characters, digits, dashes and underscores. Could be used for
				// class names, sortmode values, etc.
				return new Zend\Filter\PregReplace('/[^\p{L}\p{N}\p{Pc}\p{Pd}]*/', '');
			case 'digits':
				// Removes everything except digits eg. ' 12345 to 67890' returns '1234567890', while '-5' returns '5'
				// return type: (string)
				return new Zend\Filter\Digits;
			case 'digitscolons':
				// Removes everything except digits and colons, e.g., for colon-separated ID numbers.
				// Only characters matched, not patterns - eg 'x75::xx44:' will return '75::44:'
				return new Zend\Filter\PregReplace('/[^\p{N}:]*/', '');
			case 'digitscommas':
				// Removes everything except digits and commas, e.g., for comma-separated ID numbers.
				// Only characters matched, not patterns - eg 'x75,,xx44,' will return '75,,44,'
				return new Zend\Filter\PregReplace('/[^\p{N},]*/', '');
			case 'digitspipes':
				// Removes everything except digits and pipes, e.g., for pipe-separated ID numbers.
				// Only characters matched, not patterns - eg 'x75||xx44|' will return '75||44|'
				return new Zend\Filter\PregReplace('/[^\p{N}\|]*/', '');
			case 'int':
				// Transforms a phrase into an integer. eg. '-4 is less than 0' returns -4, while '' returns 0
				// return type: (int), if the input is scalar
				return new Zend\Filter\ToInt;
			case 'isodate':
				return new TikiFilter_IsoDate;
			case 'isodatetime':
				return new TikiFilter_IsoDate('Y-m-d H:i:s');
			case 'iso8601':
				return new TikiFilter_IsoDate('Y-m-d\TH:i:s');
			case 'username':
			case 'groupname':
			case 'pagename':
			case 'topicname':
			case 'themename':
			case 'email':
			case 'url':
			case 'text':
			case 'date':
			case 'time':
			case 'datetime':
			case 'striptags':
				// Strips XML and HTML tags
				return new Zend\Filter\StripTags;
			case 'bool':
				// False upon:	false, 0, '0', 0.0, '', array(), null, 'false', 'no', 'n' and php casting equivalent to false.
				// True upon:	Everything else returns true. Case insensitive evaluation.
				return new Zend\Filter\Boolean([
					'type'			=> Zend\Filter\Boolean::TYPE_ALL,
					'translations'	=> ['n' => false, 'N' => false]
					]);
			case 'relativeurl':
				// If formatted as a absolute url, will return the relative portion, also applies striptags
				return new TikiFilter_RelativeURL;
			case 'xss':
				// Leave everything except for potentially malicious HTML
				return new TikiFilter_PreventXss;
			case 'purifier':
				// Strips non-valid HTML and potentially malicious HTML
				return new TikiFilter_HtmlPurifier('temp/cache');
			case 'wikicontent':
				return new TikiFilter_WikiContent;

			// Dummy filter to keep value unchanged
			case 'none':
				return new TikiFilter_None;

			// Exotic filter which may alter the filtered value, for values previously "neutered" by the PreventXss filter
			case 'rawhtml_unsafe':
				return new TikiFilter_RawUnsafe;

			case 'lang':
				// Allows values for languages (such as 'en') available on the site
				return new TikiFilter_Lang;
			case 'imgsize':
				// Allows digits optionally followed by a space and/or certain size units
				return new TikiFilter_PregFilter(
					'/^(\p{N}+)\p{Zs}?(%|cm|em|ex|in|mm|pc|pt|px|vh|vw|vmin)?$/u',
					'$1$2'
				);
			case 'attribute_type':
				return new TikiFilter_AttributeType;
			default:
				trigger_error('Filter not found: ' . $filter, E_USER_WARNING);
				return new TikiFilter_PreventXss;
		}
	}
}
