/***************************************************************
 * Name:      dxoptions.cpp
 * Author:    David Vachulka (arch_dvx@users.sourceforge.net)
 * Copyright: 2020
 * License:   GPL3
 **************************************************************/

#include <wx/gbsizer.h>
#include <wx/filename.h>
#include <ctime>
#include "dxoptions.h"
#include "dxdefs.h"
#include "config.h"
#include "dxutils.h"

IMPLEMENT_DYNAMIC_CLASS(OptionsDialog, wxDialog)

BEGIN_EVENT_TABLE(OptionsDialog, wxDialog)
    EVT_BUTTON(wxID_OK, OptionsDialog::OnOk)
    EVT_BUTTON(wxID_CANCEL, OptionsDialog::OnCancel)
    EVT_BUTTON(wxID_APPLY, OptionsDialog::OnApply)
    EVT_BUTTON(ID_BTN_DEFAULTCOLORS, OptionsDialog::OnDefaultColors)
    EVT_BUTTON(ID_BTN_DEFAULTEVENTS, OptionsDialog::OnDefaultEvents)
    EVT_FONTPICKER_CHANGED(ID_BTN_FONT, OptionsDialog::OnFont)
END_EVENT_TABLE()

OptionsDialog::OptionsDialog(wxWindow *parent, bool ipc)
    : wxDialog(parent, wxID_ANY, _("Preferences")), m_reminders(static_cast<dxremindersFrame*>(parent))
{
    m_date = wxDateTime::Now();
    m_settings = dxsettings;
    m_startOnLogin = nullptr;
    m_startAtTray = nullptr;
    m_hideTrayIcon = nullptr;
    m_notaskbar = nullptr;
    m_silentMode = nullptr;
    m_calMiniFont = nullptr;
    m_calendarToday = nullptr;
    int i=0;
    m_mainSizer = new wxBoxSizer(wxVERTICAL);

    m_book = new wxListbook(this, wxID_ANY, wxDefaultPosition, wxDefaultSize, wxLB_LEFT);

    wxPanel *general = new wxPanel(m_book);
    wxBoxSizer *box1 = new wxBoxSizer(wxHORIZONTAL);
    wxGridBagSizer *gridSizer1 = new wxGridBagSizer(5, 5);
#if !(defined (__WXMAC__))
    if(!ipc)
    {
        gridSizer1->Add(new wxStaticText(general, wxID_ANY, _("Start application on login for reminders")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
        m_startOnLogin = new wxCheckBox(general, wxID_ANY, wxEmptyString);
        gridSizer1->Add(m_startOnLogin, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND);
        i++;
        gridSizer1->Add(new wxStaticText(general, wxID_ANY, _("Start application minimalized at tray")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
        m_startAtTray = new wxCheckBox(general, wxID_ANY, wxEmptyString);
        gridSizer1->Add(m_startAtTray, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND);
        i++;
        gridSizer1->Add(new wxStaticText(general, wxID_ANY, _("Hide tray icon")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
        m_hideTrayIcon = new wxCheckBox(general, wxID_ANY, wxEmptyString);
        gridSizer1->Add(m_hideTrayIcon, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND);
        i++;
    }
    else
    {
        gridSizer1->Add(new wxStaticText(general, wxID_ANY, _("Do not show on taskbar")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
        m_notaskbar = new wxCheckBox(general, wxID_ANY, wxEmptyString);
        gridSizer1->Add(m_notaskbar, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND);
        i++;
    }
    gridSizer1->Add(new wxStaticText(general, wxID_ANY, _("Silent mode")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_silentMode = new wxCheckBox(general, wxID_ANY, wxEmptyString);
    gridSizer1->Add(m_silentMode, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND);
    i++;
#endif
    gridSizer1->Add(new wxStaticText(general, wxID_ANY, _("Encrypt database")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_encrypt = new wxCheckBox(general, wxID_ANY, wxEmptyString);
    gridSizer1->Add(m_encrypt, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND);
    i++;
    gridSizer1->Add(new wxStaticText(general, wxID_ANY, _("Close dialog 'Events for day'\nafter Event dialog loaded")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_closeDayDialog = new wxCheckBox(general, wxID_ANY, wxEmptyString);
    gridSizer1->Add(m_closeDayDialog, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND);
    i++;
    gridSizer1->Add(new wxStaticText(general, wxID_ANY, _("New event dialog on right-click at calendar")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_rightclickNewevent = new wxCheckBox(general, wxID_ANY, wxEmptyString);
    gridSizer1->Add(m_rightclickNewevent, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND);
    box1->Add(gridSizer1, 1, wxEXPAND|wxALL, 5);
    general->SetSizer(box1);
    box1->Fit(general);
    box1->SetSizeHints(general);
    m_book->AddPage(general, _("General"));

    wxPanel *appearance = new wxPanel(m_book);
    wxBoxSizer *box3 = new wxBoxSizer(wxHORIZONTAL);
    wxBoxSizer *box3v = new wxBoxSizer(wxVERTICAL);
    m_look0 = new wxRadioButton(appearance, wxID_ANY, _("Calendar above events list"), wxDefaultPosition, wxDefaultSize, wxRB_GROUP);
    box3v->Add(m_look0);
    m_look1 = new wxRadioButton(appearance, wxID_ANY, _("Calendar under events list"));
    box3v->Add(m_look1);
    m_look2 = new wxRadioButton(appearance, wxID_ANY, _("Only events list"));
    box3v->Add(m_look2);
    m_look3 = new wxRadioButton(appearance, wxID_ANY, _("Only calendar"));
    box3v->Add(m_look3);
    m_resize = new wxCheckBox(appearance, wxID_ANY, _("Resize application window"));
    m_resize->SetValue(dxsettings.resize());
    box3v->Add(m_resize);
    m_showNote = new wxCheckBox(appearance, wxID_ANY, _("Show note"));
    m_showNote->SetValue(dxsettings.showNote());
    box3v->Add(m_showNote);
    box3->Add(box3v, 1, wxEXPAND|wxALL, 5);
    appearance->SetSizer(box3);
    box3->Fit(appearance);
    box3->SetSizeHints(appearance);
    m_book->AddPage(appearance, _("Appearance"));

    wxPanel *eventtime = new wxPanel(m_book);
    wxBoxSizer *box6 = new wxBoxSizer(wxHORIZONTAL);
    wxBoxSizer *box6v = new wxBoxSizer(wxVERTICAL);
    m_reminderTime0 = new wxRadioButton(eventtime, wxID_ANY, _("Current time"), wxDefaultPosition, wxDefaultSize, wxRB_GROUP);
    box6v->Add(m_reminderTime0);
    wxBoxSizer *box6h = new wxBoxSizer(wxHORIZONTAL);
    m_reminderTime1 = new wxRadioButton(eventtime, wxID_ANY, _("Preset time"));
    box6h->Add(m_reminderTime1, 0, wxALIGN_CENTER_VERTICAL, 5);
    m_reminderPresetTime = new dxTimePickerCtrl(eventtime, dxsettings.reminderPresetTime());
    box6h->Add(m_reminderPresetTime, 0, wxEXPAND|wxALL, 5);
    box6v->Add(box6h);
    wxBoxSizer *box6h1 = new wxBoxSizer(wxHORIZONTAL);
    m_reminderTime2 = new wxRadioButton(eventtime, wxID_ANY, _("Minutes ahead from now"));
    box6h1->Add(m_reminderTime2, 0, wxALIGN_CENTER_VERTICAL, 5);
    m_reminderMinutes = new wxSpinCtrl(eventtime);
    box6h1->Add(m_reminderMinutes, 0, wxEXPAND|wxALL, 5);
    box6v->Add(box6h1);
    box6->Add(box6v, 1, wxEXPAND|wxALL, 5);
    eventtime->SetSizer(box6);
    box6->Fit(eventtime);
    box6->SetSizeHints(eventtime);
    m_book->AddPage(eventtime, _("Event time"));

    wxPanel *eventrr = new wxPanel(m_book);
    wxBoxSizer *box7 = new wxBoxSizer(wxHORIZONTAL);
    wxGridBagSizer *gridSizer7 = new wxGridBagSizer(5, 5);
    i=0;
    gridSizer7->Add(new wxStaticText(eventrr, wxID_ANY, _("Reminder")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_defaultReminder = new wxChoice(eventrr, wxID_ANY, wxDefaultPosition, wxDefaultSize, dxutils::reminders());
    m_defaultReminder->SetSelection(dxsettings.defaultReminder());
    gridSizer7->Add(m_defaultReminder, wxGBPosition(i,1), wxDefaultSpan, wxALL, 2);
    i++;
    gridSizer7->Add(new wxStaticText(eventrr, wxID_ANY, _("Reminder")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_defaultRecurrence = new wxChoice(eventrr, wxID_ANY, wxDefaultPosition, wxDefaultSize, dxutils::recurrences());
    m_defaultRecurrence->SetSelection(dxutils::recurrenceTypeToSelection(dxsettings.defaultRecurrence()));
    gridSizer7->Add(m_defaultRecurrence, wxGBPosition(i,1), wxDefaultSpan, wxALL, 2);
    box7->Add(gridSizer7, 1, wxEXPAND|wxALL, 5);
    eventrr->SetSizer(box7);
    box7->Fit(eventrr);
    box7->SetSizeHints(eventrr);
    m_book->AddPage(eventrr, _("Event reminder & recurrence"));

    wxPanel *calendar = new wxPanel(m_book);
    wxBoxSizer *box2 = new wxBoxSizer(wxHORIZONTAL);
    wxGridBagSizer *gridSizer2 = new wxGridBagSizer(5, 5);
    i=0;
    gridSizer2->Add(new wxStaticText(calendar, wxID_ANY, _("Text color")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_calForeground = new wxColourPickerCtrl(calendar, wxID_ANY, dxsettings.calForeground());
    gridSizer2->Add(m_calForeground, wxGBPosition(i,1), wxDefaultSpan, wxALL, 2);
    i++;
    gridSizer2->Add(new wxStaticText(calendar, wxID_ANY, _("Background color")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_calBackground = new wxColourPickerCtrl(calendar, wxID_ANY, dxsettings.calBackground());
    gridSizer2->Add(m_calBackground, wxGBPosition(i,1), wxDefaultSpan, wxALL, 2);
    i++;
    gridSizer2->Add(new wxStaticText(calendar, wxID_ANY, _("Weekend text color")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_calWeekendFg = new wxColourPickerCtrl(calendar, wxID_ANY, dxsettings.calWeekendFg());
    gridSizer2->Add(m_calWeekendFg, wxGBPosition(i,1), wxDefaultSpan, wxALL, 2);
    i++;
    gridSizer2->Add(new wxStaticText(calendar, wxID_ANY, _("Weekend background color")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_calWeekendBg = new wxColourPickerCtrl(calendar, wxID_ANY, dxsettings.calWeekendBg());
    gridSizer2->Add(m_calWeekendBg, wxGBPosition(i,1), wxDefaultSpan, wxALL, 2);
    i++;
    gridSizer2->Add(new wxStaticText(calendar, wxID_ANY, _("Public holiday text color")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_calHolidayFg = new wxColourPickerCtrl(calendar, wxID_ANY, dxsettings.calHolidayFg());
    gridSizer2->Add(m_calHolidayFg, wxGBPosition(i,1), wxDefaultSpan, wxALL, 2);
    i++;
    gridSizer2->Add(new wxStaticText(calendar, wxID_ANY, _("Public holiday background color")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_calHolidayBg = new wxColourPickerCtrl(calendar, wxID_ANY, dxsettings.calHolidayBg());
    gridSizer2->Add(m_calHolidayBg, wxGBPosition(i,1), wxDefaultSpan, wxALL, 2);
    i++;
    gridSizer2->Add(new wxStaticText(calendar, wxID_ANY, _("Selected day text color")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_calHighlightFg = new wxColourPickerCtrl(calendar, wxID_ANY, dxsettings.calHighlightFg());
    gridSizer2->Add(m_calHighlightFg, wxGBPosition(i,1), wxDefaultSpan, wxALL, 2);
    i++;
    gridSizer2->Add(new wxStaticText(calendar, wxID_ANY, _("Selected day background color")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_calHighlightBg = new wxColourPickerCtrl(calendar, wxID_ANY, dxsettings.calHighlightBg());
    gridSizer2->Add(m_calHighlightBg, wxGBPosition(i,1), wxDefaultSpan, wxALL, 2);
    i++;
    gridSizer2->Add(new wxStaticText(calendar, wxID_ANY, _("Reminder circle color")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_calReminder = new wxColourPickerCtrl(calendar, wxID_ANY, dxsettings.calReminder());
    gridSizer2->Add(m_calReminder, wxGBPosition(i,1), wxDefaultSpan, wxALL, 2);
    i++;
    gridSizer2->Add(new wxStaticText(calendar, wxID_ANY, _("Recurrent circle color")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_calRecurrence = new wxColourPickerCtrl(calendar, wxID_ANY, dxsettings.calRecurrence());
    gridSizer2->Add(m_calRecurrence, wxGBPosition(i,1), wxDefaultSpan, wxALL, 2);
    i++;
    m_defaultColors = new wxButton(calendar, ID_BTN_DEFAULTCOLORS, _("Default"));
    gridSizer2->Add(m_defaultColors, wxGBPosition(i,0), wxDefaultSpan, wxALL, 2);
    i++;
    gridSizer2->Add(new wxStaticText(calendar, wxID_ANY, _("Calendar font")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_calFont = new wxFontPickerCtrl(calendar, ID_BTN_FONT, dxutils::fontFromString(dxsettings.calFont()));
    gridSizer2->Add(m_calFont, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND|wxALL, 2);
    i++;
    if(ipc)
    {
        gridSizer2->Add(new wxStaticText(calendar, wxID_ANY, _("Mini-calendar font")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
        m_calMiniFont = new wxFontPickerCtrl(calendar, ID_BTN_FONT, dxutils::fontFromString(dxsettings.calMiniFont()));
        gridSizer2->Add(m_calMiniFont, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND|wxALL, 2);
        i++;
        m_calendarToday = new wxCheckBox(calendar, wxID_ANY, _("Selected day - Always show today's day"));
        m_calendarToday->SetValue(dxsettings.calendarToday());
        gridSizer2->Add(m_calendarToday, wxGBPosition(i,0), wxGBSpan(1,2), wxEXPAND|wxALL, 2);
        i++;
    }
    m_showWeekNumber = new wxCheckBox(calendar, wxID_ANY, _("Show week numbers"));
    m_showWeekNumber->SetValue(dxsettings.showWeekNumber());
    gridSizer2->Add(m_showWeekNumber, wxGBPosition(i,0), wxGBSpan(1,2), wxEXPAND|wxALL, 2);
    box2->Add(gridSizer2, 1, wxEXPAND|wxALL, 5);
    calendar->SetSizer(box2);
    box2->Fit(calendar);
    box2->SetSizeHints(calendar);
    m_book->AddPage(calendar, _("Calendar"));

    wxPanel *eventslist = new wxPanel(m_book);
    wxBoxSizer *box4 = new wxBoxSizer(wxHORIZONTAL);
    wxGridBagSizer *gridSizer4 = new wxGridBagSizer(5, 5);
    i=0;
    gridSizer4->Add(new wxStaticText(eventslist, wxID_ANY, _("Show colums:")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    i++;
    m_colDateShow = new wxCheckBox(eventslist, wxID_ANY, _("Date"));
    m_colDateShow->SetValue(dxsettings.colDateShow());
    gridSizer4->Add(m_colDateShow, wxGBPosition(i,0), wxDefaultSpan, wxEXPAND, 2);
    i++;
    m_colReminderShow = new wxCheckBox(eventslist, wxID_ANY, _("Reminder"));
    m_colReminderShow->SetValue(dxsettings.colReminderShow());
    gridSizer4->Add(m_colReminderShow, wxGBPosition(i,0), wxDefaultSpan, wxEXPAND, 2);
    i++;
    m_colRecurrenceShow = new wxCheckBox(eventslist, wxID_ANY, _("Recurrence"));
    m_colRecurrenceShow->SetValue(dxsettings.colRecurrenceShow());
    gridSizer4->Add(m_colRecurrenceShow, wxGBPosition(i,0), wxDefaultSpan, wxEXPAND, 2);
    i++;
    gridSizer4->Add(new wxStaticText(eventslist, wxID_ANY, _("Odd month color")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_oddMonth = new wxColourPickerCtrl(eventslist, wxID_ANY, dxsettings.oddMonth());
    gridSizer4->Add(m_oddMonth, wxGBPosition(i,1), wxDefaultSpan, wxALL, 2);
    i++;
    gridSizer4->Add(new wxStaticText(eventslist, wxID_ANY, _("Even month color")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_evenMonth = new wxColourPickerCtrl(eventslist, wxID_ANY, dxsettings.evenMonth());
    gridSizer4->Add(m_evenMonth, wxGBPosition(i,1), wxDefaultSpan, wxALL, 2);
    i++;
    gridSizer4->Add(new wxStaticText(eventslist, wxID_ANY, _("Highlight event text color")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_highlightFg = new wxColourPickerCtrl(eventslist, wxID_ANY, dxsettings.highlightFg());
    gridSizer4->Add(m_highlightFg, wxGBPosition(i,1), wxDefaultSpan, wxALL, 2);
    i++;
    gridSizer4->Add(new wxStaticText(eventslist, wxID_ANY, _("Highlight event background color #1")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_highlightBg1 = new wxColourPickerCtrl(eventslist, wxID_ANY, dxsettings.highlightBg1());
    gridSizer4->Add(m_highlightBg1, wxGBPosition(i,1), wxDefaultSpan, wxALL, 2);
    i++;
    gridSizer4->Add(new wxStaticText(eventslist, wxID_ANY, _("Highlight event background color #2")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_highlightBg2 = new wxColourPickerCtrl(eventslist, wxID_ANY, dxsettings.highlightBg2());
    gridSizer4->Add(m_highlightBg2, wxGBPosition(i,1), wxDefaultSpan, wxALL, 2);
    i++;
    gridSizer4->Add(new wxStaticText(eventslist, wxID_ANY, _("Highlight event background color #3")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_highlightBg3 = new wxColourPickerCtrl(eventslist, wxID_ANY, dxsettings.highlightBg3());
    gridSizer4->Add(m_highlightBg3, wxGBPosition(i,1), wxDefaultSpan, wxALL, 2);
    i++;
    gridSizer4->Add(new wxStaticText(eventslist, wxID_ANY, _("Highlight event background color #4")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_highlightBg4 = new wxColourPickerCtrl(eventslist, wxID_ANY, dxsettings.highlightBg4());
    gridSizer4->Add(m_highlightBg4, wxGBPosition(i,1), wxDefaultSpan, wxALL, 2);
    i++;
    gridSizer4->Add(new wxStaticText(eventslist, wxID_ANY, _("Highlight event background color #5")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_highlightBg5 = new wxColourPickerCtrl(eventslist, wxID_ANY, dxsettings.highlightBg5());
    gridSizer4->Add(m_highlightBg5, wxGBPosition(i,1), wxDefaultSpan, wxALL, 2);
    i++;
    wxButton *dbtn = new wxButton(eventslist, ID_BTN_DEFAULTEVENTS, _("Default"));
    gridSizer4->Add(dbtn, wxGBPosition(i,0), wxDefaultSpan, wxALL, 2);
    i++;
    m_disableColors = new wxCheckBox(eventslist, wxID_ANY, _("Disable months colors"));
    m_disableColors->SetValue(dxsettings.disableColors());
    gridSizer4->Add(m_disableColors, wxGBPosition(i,0), wxGBSpan(1,2), wxEXPAND, 2);
    i++;
    gridSizer4->Add(new wxStaticText(eventslist, wxID_ANY, _("Event font")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_eventsFont = new wxFontPickerCtrl(eventslist, ID_BTN_FONT, dxutils::fontFromString(dxsettings.eventsFont()));
    gridSizer4->Add(m_eventsFont, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND|wxALL, 2);
    i++;
    gridSizer4->Add(new wxStaticText(eventslist, wxID_ANY, _("Always show events with recurrence:")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    i++;
    m_alwaysNone = new wxCheckBox(eventslist, wxID_ANY, _("None (Do not delete)"));
    m_alwaysNone->SetValue(dxsettings.alwaysNone());
    m_alwaysNone->SetToolTip(_("Checked: Event will be shown if doesn't match filter"));
    gridSizer4->Add(m_alwaysNone, wxGBPosition(i,0), wxGBSpan(1,2), wxEXPAND, 2);
    i++;
    m_alwaysOncenotdelete = new wxCheckBox(eventslist, wxID_ANY, _("Once (Do not delete)"));
    m_alwaysOncenotdelete->SetValue(dxsettings.alwaysOncenotdelete());
    m_alwaysOncenotdelete->SetToolTip(_("Checked: Event will be shown if doesn't match filter"));
    gridSizer4->Add(m_alwaysOncenotdelete, wxGBPosition(i,0), wxGBSpan(1,2), wxEXPAND, 2);
    box4->Add(gridSizer4, 1, wxEXPAND|wxALL, 5);
    eventslist->SetSizer(box4);
    box4->Fit(eventslist);
    box4->SetSizeHints(eventslist);
    m_book->AddPage(eventslist, _("Events list"));

    m_datePage = new wxPanel(m_book);
    m_datetimeSizer = new wxBoxSizer(wxHORIZONTAL);
    wxGridBagSizer *gridSizer5 = new wxGridBagSizer(5, 5);
    i=0;
    gridSizer5->Add(new wxStaticText(m_datePage, wxID_ANY, _("Layout")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_dateLayout = new wxChoice(m_datePage, wxID_ANY);
    m_dateLayout->Append(_("Date, then time"));
    m_dateLayout->Append(_("Time, then date"));
    m_dateLayout->Append(_("Date only"));
    m_dateLayout->Append(_("Time only"));
    m_dateLayout->SetSelection(dxsettings.dateLayout());
    gridSizer5->Add(m_dateLayout, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND, 2);
    i++;
    gridSizer5->Add(new wxStaticText(m_datePage, wxID_ANY, _("Date")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_dateFormat = new wxChoice(m_datePage, ID_DATE_CHOICE);
    m_dateFormat->Append(dxutils::formatDate(m_date, "%Y-%m-%d"));
    m_dateFormat->Append(dxutils::formatDate(m_date, "%Y %B %d"));
    m_dateFormat->Append(dxutils::formatDate(m_date, "%m/%d/%Y"));
    m_dateFormat->Append(dxutils::formatDate(m_date, "%B %d, %Y"));
    m_dateFormat->Append(dxutils::formatDate(m_date, "%b %d, %Y"));
    m_dateFormat->Append(dxutils::formatDate(m_date, "%A, %B %d, %Y"));
    m_dateFormat->Append(dxutils::formatDate(m_date, "%a, %b %d, %Y"));
    m_dateFormat->Append(dxutils::formatDate(m_date, "%d/%m/%Y"));
    m_dateFormat->Append(dxutils::formatDate(m_date, "%d %B %Y"));
    m_dateFormat->Append(dxutils::formatDate(m_date, "%d %b %Y"));
    m_dateFormat->Append(dxutils::formatDate(m_date, "%A, %d %B %Y"));
    m_dateFormat->Append(dxutils::formatDate(m_date, "%a, %d %b %Y"));
    m_dateFormat->Append(dxutils::formatDate(m_date, "%x"));
    m_dateFormat->Append(_("Custom..."));
    gridSizer5->Add(m_dateFormat, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND, 2);
    i++;
    m_dateFormatText = new wxTextCtrl(m_datePage, wxID_ANY);
    m_dateFormatText->SetValue(dxsettings.dateFormat());
    gridSizer5->Add(m_dateFormatText, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND, 2);
    i++;
    m_dateFormatPreview = new wxStaticText(m_datePage, wxID_ANY, dxutils::formatDate(m_date, dxsettings.dateFormat()));
    gridSizer5->Add(m_dateFormatPreview, wxGBPosition(i,1), wxDefaultSpan, wxALIGN_CENTER_VERTICAL, 2);
    i++;
    gridSizer5->Add(new wxStaticText(m_datePage, wxID_ANY, _("Time")), wxGBPosition(i,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL);
    m_timeFormat = new wxChoice(m_datePage, ID_TIME_CHOICE);
    m_timeFormat->Append(dxutils::formatTime(m_date, "%H:%M"));
    m_timeFormat->Append(dxutils::formatTime(m_date, "%H:%M:%S"));
    m_timeFormat->Append(dxutils::formatTime(m_date, "%l:%M %P"));
    m_timeFormat->Append(dxutils::formatTime(m_date, "%l:%M:%S %P"));
    m_timeFormat->Append(dxutils::formatTime(m_date, "%X"));
    m_timeFormat->Append(_("Custom..."));
    gridSizer5->Add(m_timeFormat, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND, 2);
    i++;
    m_timeFormatText = new wxTextCtrl(m_datePage, wxID_ANY);
    m_timeFormatText->SetValue(dxsettings.timeFormat());
    gridSizer5->Add(m_timeFormatText, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND, 2);
    i++;
    m_timeFormatPreview = new wxStaticText(m_datePage, wxID_ANY, dxutils::formatTime(m_date, dxsettings.timeFormat()));
    gridSizer5->Add(m_timeFormatPreview, wxGBPosition(i,1), wxDefaultSpan, wxALIGN_CENTER_VERTICAL, 2);
    i++;
    m_useAMPM = new wxCheckBox(m_datePage, wxID_ANY, _("12-hour clock"));
    m_useAMPM->SetValue(dxsettings.useAMPM());
    gridSizer5->Add(m_useAMPM, wxGBPosition(i,1), wxDefaultSpan, wxEXPAND, 2);
    gridSizer5->SetEmptyCellSize(wxSize(0,0));
    m_datetimeSizer->Add(gridSizer5, 0, wxEXPAND|wxALL, 5);
    m_datePage->SetSizer(m_datetimeSizer);
    m_datetimeSizer->Fit(m_datePage);
    m_datetimeSizer->SetSizeHints(m_datePage);
    m_book->AddPage(m_datePage, _("Date & Time format"));

    m_mainSizer->Add(m_book);

    wxStdDialogButtonSizer *btnSizer = new wxStdDialogButtonSizer();
    wxButton* okButton = new wxButton(this, wxID_OK, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(okButton);
    wxButton* cancelButton = new wxButton(this, wxID_CANCEL, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(cancelButton);
    wxButton* applyButton = new wxButton(this, wxID_APPLY, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(applyButton);
    btnSizer->Realize();
    m_mainSizer->Add(btnSizer, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);

    this->SetSizer(m_mainSizer);
    m_mainSizer->Fit(this);
    m_mainSizer->SetSizeHints(this);

    if(m_startAtTray) m_startAtTray->Connect(wxEVT_CHECKBOX, wxCommandEventHandler(OptionsDialog::OnCheckBox), nullptr, this);
    if(m_hideTrayIcon) m_hideTrayIcon->Connect(wxEVT_CHECKBOX, wxCommandEventHandler(OptionsDialog::OnCheckBox), nullptr, this);
    m_dateFormatText->Bind(wxEVT_KEY_UP, &OptionsDialog::OnDateText, this);
    m_timeFormatText->Bind(wxEVT_KEY_UP, &OptionsDialog::OnTimeText, this);

    if(m_startOnLogin) m_startOnLogin->SetValue(dxsettings.startOnLogin());
    if(m_startAtTray) m_startAtTray->SetValue(dxsettings.startAtTray());
    if(m_hideTrayIcon) m_hideTrayIcon->SetValue(dxsettings.hideTrayIcon());
    switch (dxsettings.look()) {
    case 0: m_look0->SetValue(true); break;
    case 1: m_look1->SetValue(true); break;
    case 2: m_look2->SetValue(true); break;
    default: m_look3->SetValue(true);
    }
    if(m_notaskbar) m_notaskbar->SetValue(dxsettings.notaskbar());
    if(m_silentMode) m_silentMode->SetValue(dxsettings.silentMode());
    m_encrypt->SetValue(dxsettings.encrypt());
    m_closeDayDialog->SetValue(dxsettings.closeDayDialog());
    m_rightclickNewevent->SetValue(dxsettings.rightclickNewevent());

    m_dateFormat->SetSelection(dateFormatToSelection(dxsettings.dateFormat()));
    m_timeFormat->SetSelection(timeFormatToSelection(dxsettings.timeFormat()));
    m_dateFormat->Connect(wxEVT_CHOICE, wxCommandEventHandler(OptionsDialog::OnChoiceDateFormat), nullptr, this);
    m_timeFormat->Connect(wxEVT_CHOICE, wxCommandEventHandler(OptionsDialog::OnChoiceTimeFormat), nullptr, this);
    showHideDateFormat();
    showHideTimeFormat();

    switch (dxsettings.reminderTime()) {
    case 0: m_reminderTime0->SetValue(true); break;
    case 1: m_reminderTime1->SetValue(true); break;
    default: m_reminderTime2->SetValue(true);
    }
    m_reminderMinutes->SetValue(dxsettings.reminderMinutes());
}

bool OptionsDialog::resizeFrame()
{
    return m_resize->GetValue();
}

void OptionsDialog::OnCheckBox(wxCommandEvent &/*event*/)
{
    if(m_hideTrayIcon && m_hideTrayIcon->GetValue()) m_startAtTray->SetValue(false);
}

void OptionsDialog::OnOk(wxCommandEvent &/*event*/)
{
    saveOptions();
    EndModal(wxID_OK);
}

void OptionsDialog::OnCancel(wxCommandEvent &/*event*/)
{
    dxsettings.setStartOnLogin(m_settings.startOnLogin());
    dxsettings.setStartAtTray(m_settings.startAtTray());
    dxsettings.setHideTrayIcon(m_settings.hideTrayIcon());
    dxsettings.setSilentMode(m_settings.silentMode());
    dxsettings.setNotaskbar(m_settings.notaskbar());
    dxsettings.setEncrypt(m_settings.encrypt());
    dxsettings.setCloseDayDialog(m_settings.closeDayDialog());
    dxsettings.setRightclickNewevent(m_settings.rightclickNewevent());
    dxsettings.setLook(m_settings.look());
    dxsettings.setResize(m_settings.resize());
    dxsettings.setShowNote(m_settings.showNote());
    dxsettings.setCalForeground(m_settings.calForeground());
    dxsettings.setCalBackground(m_settings.calBackground());
    dxsettings.setCalHolidayFg(m_settings.calHolidayFg());
    dxsettings.setCalHolidayBg(m_settings.calHolidayBg());
    dxsettings.setCalHighlightFg(m_settings.calHighlightFg());
    dxsettings.setCalHighlightBg(m_settings.calHighlightBg());
    dxsettings.setCalReminder(m_settings.calReminder());
    dxsettings.setCalRecurrence(m_settings.calRecurrence());
    dxsettings.setCalFont(m_settings.calFont());
    dxsettings.setCalMiniFont(m_settings.calMiniFont());
    dxsettings.setCalendarToday(m_settings.calendarToday());
    dxsettings.setShowWeekNumber(m_settings.showWeekNumber());
    dxsettings.setColDateShow(m_settings.colDateShow());
    dxsettings.setColReminderShow(m_settings.colReminderShow());
    dxsettings.setColRecurrenceShow(m_settings.colRecurrenceShow());
    dxsettings.setOddMonth(m_settings.oddMonth());
    dxsettings.setEvenMonth(m_settings.evenMonth());
    dxsettings.setDisableColors(m_settings.disableColors());
    dxsettings.setEventsFont(m_settings.eventsFont());
    dxsettings.setDateLayout(m_settings.dateLayout());
    dxsettings.setDateFormat(m_settings.dateFormat());
    dxsettings.setTimeFormat(m_settings.timeFormat());
    dxsettings.setUseAMPM(m_settings.useAMPM());
    dxsettings.setReminderTime(m_settings.reminderTime());
    dxsettings.setReminderPresetTime(m_settings.reminderPresetTime());
    dxsettings.setReminderMinutes(m_settings.reminderMinutes());
    EndModal(wxID_CANCEL);
}

void OptionsDialog::OnApply(wxCommandEvent &/*event*/)
{
    saveOptions();
    m_reminders->applyPreference(m_resize->GetValue());
}

void OptionsDialog::saveOptions()
{
    if(m_startOnLogin) dxsettings.setStartOnLogin(m_startOnLogin->GetValue());
    if(m_startAtTray) dxsettings.setStartAtTray(m_startAtTray->GetValue());
    if(m_hideTrayIcon) dxsettings.setHideTrayIcon(m_hideTrayIcon->GetValue());
    if(m_notaskbar) dxsettings.setNotaskbar(m_notaskbar->GetValue());
    if(m_silentMode) dxsettings.setSilentMode(m_silentMode->GetValue());
    dxsettings.setEncrypt(m_encrypt->GetValue());
    dxsettings.setCloseDayDialog(m_closeDayDialog->GetValue());
    dxsettings.setRightclickNewevent(m_rightclickNewevent->GetValue());
    if(m_look0->GetValue()) dxsettings.setLook(0);
    if(m_look1->GetValue()) dxsettings.setLook(1);
    if(m_look2->GetValue()) dxsettings.setLook(2);
    if(m_look3->GetValue()) dxsettings.setLook(3);
    dxsettings.setResize(m_resize->GetValue());
    dxsettings.setShowNote(m_showNote->GetValue());
    dxsettings.setCalForeground(m_calForeground->GetColour());
    dxsettings.setCalBackground(m_calBackground->GetColour());
    dxsettings.setCalHolidayFg(m_calHolidayFg->GetColour());
    dxsettings.setCalHolidayBg(m_calHolidayBg->GetColour());
    dxsettings.setCalWeekendFg(m_calWeekendFg->GetColour());
    dxsettings.setCalWeekendBg(m_calWeekendBg->GetColour());
    dxsettings.setCalHighlightFg(m_calHighlightFg->GetColour());
    dxsettings.setCalHighlightBg(m_calHighlightBg->GetColour());
    dxsettings.setCalReminder(m_calReminder->GetColour());
    dxsettings.setCalRecurrence(m_calRecurrence->GetColour());
    dxsettings.setCalFont(dxutils::fontToString(m_calFont->GetSelectedFont()));
    if(m_calMiniFont) dxsettings.setCalMiniFont(dxutils::fontToString(m_calMiniFont->GetSelectedFont()));
    if(m_calendarToday) dxsettings.setCalendarToday(m_calendarToday->GetValue());
    dxsettings.setShowWeekNumber(m_showWeekNumber->GetValue());
    dxsettings.setColDateShow(m_colDateShow->GetValue());
    dxsettings.setColReminderShow(m_colReminderShow->GetValue());
    dxsettings.setColRecurrenceShow(m_colRecurrenceShow->GetValue());
    dxsettings.setOddMonth(m_oddMonth->GetColour());
    dxsettings.setEvenMonth(m_evenMonth->GetColour());
    dxsettings.setHighlightFg(m_highlightFg->GetColour());
    dxsettings.setHighlightBg1(m_highlightBg1->GetColour());
    dxsettings.setHighlightBg2(m_highlightBg2->GetColour());
    dxsettings.setHighlightBg3(m_highlightBg3->GetColour());
    dxsettings.setHighlightBg4(m_highlightBg4->GetColour());
    dxsettings.setHighlightBg5(m_highlightBg5->GetColour());
    dxsettings.setDisableColors(m_disableColors->GetValue());
    dxsettings.setEventsFont(dxutils::fontToString(m_eventsFont->GetSelectedFont()));
    dxsettings.setAlwaysNone(m_alwaysNone->GetValue());
    dxsettings.setAlwaysOncenotdelete(m_alwaysOncenotdelete->GetValue());
    dxsettings.setDateLayout(m_dateLayout->GetSelection());
    dxsettings.setDateFormat(selectionToDateFormat(m_dateFormat->GetSelection()));
    dxsettings.setTimeFormat(selectionToTimeFormat(m_timeFormat->GetSelection()));
    dxsettings.setUseAMPM(m_useAMPM->GetValue());
    if(m_reminderTime0->GetValue()) dxsettings.setReminderTime(0);
    if(m_reminderTime1->GetValue()) dxsettings.setReminderTime(1);
    if(m_reminderTime2->GetValue()) dxsettings.setReminderTime(2);
    dxsettings.setReminderPresetTime(m_reminderPresetTime->time());
    dxsettings.setReminderMinutes(m_reminderMinutes->GetValue());
    dxsettings.setDefaultReminder(m_defaultReminder->GetSelection());
    dxsettings.setDefaultRecurrence(dxutils::recurrenceSelectionToType(m_defaultRecurrence->GetSelection()));
}

void OptionsDialog::OnDefaultColors(wxCommandEvent &/*event*/)
{
    m_calForeground->SetColour(wxSystemSettings::GetColour(wxSYS_COLOUR_WINDOWTEXT));
    m_calBackground->SetColour(wxSystemSettings::GetColour(wxSYS_COLOUR_WINDOW));
    m_calHighlightFg->SetColour(wxSystemSettings::GetColour(wxSYS_COLOUR_HIGHLIGHTTEXT));
    m_calHighlightBg->SetColour(wxSystemSettings::GetColour(wxSYS_COLOUR_HIGHLIGHT));
    m_calHolidayFg->SetColour(wxColour(33, 105, 206));
    m_calHolidayBg->SetColour(wxSystemSettings::GetColour(wxSYS_COLOUR_WINDOW));
    m_calReminder->SetColour(wxColour(0,128,0));
    m_calRecurrence->SetColour(wxColour(255,165,0));
    m_calWeekendFg->SetColour(*wxRED);
    m_calWeekendBg->SetColour(wxSystemSettings::GetColour(wxSYS_COLOUR_WINDOW));
}

void OptionsDialog::OnDefaultEvents(wxCommandEvent &/*event*/)
{
    m_oddMonth->SetColour(wxSystemSettings::GetColour(wxSYS_COLOUR_WINDOW));
    m_evenMonth->SetColour(wxColour(200,200,200));
    m_highlightFg->SetColour(*wxWHITE);
    m_highlightBg1->SetColour(*wxRED);
    m_highlightBg2->SetColour(wxColour(0,128,0));
    m_highlightBg3->SetColour(*wxBLUE);
    m_highlightBg4->SetColour(wxColour(255,165,0));
    m_highlightBg5->SetColour(wxColour(173,216,230));
}

void OptionsDialog::OnDateText(wxKeyEvent &event)
{
    m_dateFormatPreview->SetLabel(dxutils::formatDate(m_date, m_dateFormatText->GetValue()));
    m_mainSizer->Layout();
    m_mainSizer->Fit(this);
    m_mainSizer->SetSizeHints(this);
    event.Skip();
}

void OptionsDialog::OnTimeText(wxKeyEvent &event)
{
    m_timeFormatPreview->SetLabel(dxutils::formatTime(m_date, m_timeFormatText->GetValue()));
    m_mainSizer->Layout();
    m_mainSizer->Fit(this);
    m_mainSizer->SetSizeHints(this);
    event.Skip();
}

int OptionsDialog::dateFormatToSelection(const wxString &format)
{
    if(format == "%Y-%m-%d") return 0;
    if(format == "%Y %B %d") return 1;
    if(format == "%m/%d/%Y") return 2;
    if(format == "%B %d, %Y") return 3;
    if(format == "%b %d, %Y") return 4;
    if(format == "%A, %B %d, %Y") return 5;
    if(format == "%a, %b %d, %Y") return 6;
    if(format == "%d/%m/%Y") return 7;
    if(format == "%d %B %Y") return 8;
    if(format == "%d %b %Y") return 9;
    if(format == "%A, %d %B %Y") return 10;
    if(format == "%a, %d %b %Y") return 11;
    if(format == "%x") return 12;
    return 13;
}

wxString OptionsDialog::selectionToDateFormat(int selection)
{
    switch (selection) {
    case 0: return "%Y-%m-%d";
    case 1: return "%Y %B %d";
    case 2: return "%m/%d/%Y";
    case 3: return "%B %d, %Y";
    case 4: return "%b %d, %Y";
    case 5: return "%A, %B %d, %Y";
    case 6: return "%a, %b %d, %Y";
    case 7: return "%d/%m/%Y";
    case 8: return "%d %B %Y";
    case 9: return "%d %b %Y";
    case 10: return "%A, %d %B %Y";
    case 11: return "%a, %d %b %Y";
    case 12: return "%x";
    default: return m_dateFormatText->GetValue();
    }
}

int OptionsDialog::timeFormatToSelection(const wxString &format)
{
    if(format == "%H:%M") return 0;
    if(format == "%H:%M:%S") return 1;
    if(format == "%l:%M %P") return 2;
    if(format == "%l:%M:%S %P") return 3;
    if(format == "%X") return 4;
    return 5;
}

wxString OptionsDialog::selectionToTimeFormat(int selection)
{
    switch (selection) {
    case 0: return "%H:%M";
    case 1: return "%H:%M:%S";
    case 2: return "%l:%M %P";
    case 3: return "%l:%M:%S %P";
    case 4: return "%X";
    default: return m_timeFormatText->GetValue();
    }
}

void OptionsDialog::OnChoiceDateFormat(wxCommandEvent &/*event*/)
{
    showHideDateFormat();
}

void OptionsDialog::OnChoiceTimeFormat(wxCommandEvent &/*event*/)
{
    showHideTimeFormat();
}

void OptionsDialog::OnFont(wxFontPickerEvent &/*event*/)
{
    m_mainSizer->Layout();
    m_mainSizer->Fit(this);
    m_mainSizer->SetSizeHints(this);
}

void OptionsDialog::showHideDateFormat()
{
    if(m_dateFormat->GetSelection() == 13)
    {
        m_dateFormatText->Show();
        m_dateFormatPreview->Show();
    }
    else
    {
        m_dateFormatText->Hide();
        m_dateFormatPreview->Hide();
    }
    m_datetimeSizer->Layout();
    m_datetimeSizer->Fit(m_datePage);
    m_datetimeSizer->SetSizeHints(m_datePage);
}

void OptionsDialog::showHideTimeFormat()
{
    if(m_timeFormat->GetSelection() == 5)
    {
        m_timeFormatText->Show();
        m_timeFormatPreview->Show();
    }
    else
    {
        m_timeFormatText->Hide();
        m_timeFormatPreview->Hide();
    }
    m_datetimeSizer->Layout();
    m_datetimeSizer->Fit(m_datePage);
    m_datetimeSizer->SetSizeHints(m_datePage);
}
