/* $NetBSD$ */

/*
 * Copyright (c) 2003 Dennis I. Chernoivanov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "paneld.h"

#include "cf-lex.h"
#include "cf-parse.h"

#define MAX_ID_LENGTH	255

#define get_char(c, cf) \
	do { \
		(cf)->cno++; \
		if ((cf)->lookahead != 0) { \
			(c) = (cf)->lookahead; \
			(cf)->lookahead = 0; \
		} else { \
			(c) = getc((cf)->f); \
		} \
	} while(0)

#define unget_char(c, cf) \
	do { \
		(cf)->cno--; \
		(cf)->lookahead = (c); \
	} while(0)

struct lex_data {
	FILE *f;
	const char *fnm;

	int cno;
	int lineno;
	char *id;

	int lookahead;
};

struct lex_data *cfile;

static int	scan_cmd(void);
static int	scan_num(void);
static int	scan_char(void);
static int	scan_string(void);
static int	scan_comment(void);
static int	scan_identifier(void);
static int	lookup_keyword(void);

int
scan_identifier() {
	int i;
	for (i = 0; i < MAX_ID_LENGTH; i++) {
		int c;

		get_char(c, cfile);
		if (!isalnum(c)) {
			cfile->id[i] = 0;
			unget_char(c, cfile);
/*			printf("id: '%s'\n", cfile->id);*/
			return TK_IDENTIFIER;
		}

		cfile->id[i] = c;
	}

	cfile->id[0] = 0;
/*	printf("id unknown\n");*/
	return TK_UNKNOWN;
}

int
scan_num() {
	int i;
	for (i = 0; i < MAX_ID_LENGTH; i++) {
		int c;

		get_char(c, cfile);
		if (!isdigit(c)) {
			cfile->id[i] = 0;
			unget_char(c, cfile);
			yylval.num = atoi(cfile->id);
			return TK_INTEGER;
		}

		cfile->id[i] = c;
	}

	cfile->id[0] = 0;
	return TK_UNKNOWN;
}

int
lookup_keyword() {
	int i;
	static struct {
		const char *nm;
		int tok;
	} keymap[] = {
		{ "id",         TK_MENUID   },
		{ "menu",	TK_MENU     },
		{ "xtbl",	TK_XTBL     },
		{ "xlate",	TK_XLATE    },
		{ "inc",        TK_INC      },
		{ "dec",	TK_DEC      },
		{ "next",	TK_NEXT     },
		{ "prev",	TK_PREV     },
		{ "enter",	TK_ENTER    },
		{ "title",	TK_TITLE    },
		{ "write",	TK_WRITE    },
		{ "read",	TK_READ     },
		{ "int",	TK_INT      },
		{ "hex",	TK_HEX      },
		{ "inet",	TK_INET     },
		{ "inet6",	TK_INET6    },
		{ "text",	TK_TEXT     },
		{ "resist",	TK_RESIST   },
		{ "repeat",	TK_REPEAT   },
		{ "keyrate",	TK_KEYRATE  },
		{ "banner",	TK_BANNER   },
		{ "start",	TK_START    },
		{ "devname",	TK_DEVNAME  },
		{ "devnode",	TK_DEVNODE  },
		{ "exit",	TK_EXIT     },
		{ "quit",	TK_QUIT     },
		{ "none",	TK_NONE     },
		{ "type",	TK_TYPE     },
		{ "leaf",	TK_LEAF     },
		{ "confirm",	TK_CONFIRM  },
		{ "select",	TK_SELECT   },
		{ "sequence",	TK_SEQUENCE },
		{ NULL,		0           }
	};

	for (i = 0; keymap[i].nm != NULL; i++) {
		if (!strcmp(cfile->id, keymap[i].nm))
			return keymap[i].tok;
	}

/*	printf("lookup keyword: %s\n", cfile->id);*/
	return TK_UNKNOWN;
}

int
scan_cmd() {
	int token = scan_identifier();
	if (token == TK_IDENTIFIER) {
		token = lookup_keyword();
	}
	return token;
}

int
scan_char()
{
	int c;
	int cend;

	get_char(c, cfile);
	get_char(cend, cfile);

	if (cend != '\'')
		yyerror("character expected");

	yylval.c = c;
	return TK_CHAR;
}

int
scan_string() {
	int i;
	for (i = 0; i < MAX_ID_LENGTH; i++) {
		int c;

		get_char(c, cfile);
/*		printf(".chr: '%c'\n", c);*/
		if (c == '"') {
			cfile->id[i] = 0;
			yylval.str = cfile->id;
/*			printf("str: '%s'\n", cfile->id);*/
			return TK_STRING;
		} else if (c == EOF) {
			break;
		}

		cfile->id[i] = c;
	}

	cfile->id[0] = 0;
/*	printf("str unknown\n");*/
	return TK_UNKNOWN;
}

int
scan_comment() {
	for (;;) {
		int c;
		get_char(c, cfile);
		if ((c == '\n') || (c == EOF)) {
			unget_char(c, cfile);
			break;
		}
	}
	return (-1);
}

int
yylex() {
	int token = -1;

	while (token < 0) {
		get_char(token, cfile);
		switch(token) {
		case '.':
			token = scan_cmd();
			break;

		case '"':
			token = scan_string();
			break;

		case '[':
		case ']':
		case '{':
		case '}':
		case '=':
		case ';':
			break;

		case '#':
			token = scan_comment();
			break;

		case '\n':
			cfile->cno = 0;
			cfile->lineno++;
			/* FALLTHROUGH */

		case ' ':
		case '\t':
			token = -1;
			break;

		case EOF:
			token = 0;
			break;

		case '\'':
			token = scan_char();
			break;

		default:
			if (isalpha(token)) {
				unget_char(token, cfile);
				token = scan_cmd();
				break;
			} else if (isdigit(token)) {
				unget_char(token, cfile);
				token = scan_num();
				break;
			}

			token = TK_UNKNOWN;
			break;
		}
	}

	return (token);
}

void
yyerror(const char *msg) {
	util_trace(LOG_ERR, "[%s:%d] %s", cfile->fnm, cfile->lineno, msg);
	/* NORETURN */
}

void
yylex_init(const char *fname)
{
	cfile = (struct lex_data*)cf_malloc(sizeof(struct lex_data));
	memset(cfile, 0, sizeof(struct lex_data));

	if ( (cfile->f = fopen(fname, "r")) == NULL)
		util_trace(LOG_ERR, "cannot open %s", fname);

	cfile->fnm = fname;
	cfile->lineno = 1;
	cfile->id = (char*)cf_malloc(MAX_ID_LENGTH);
}

void
yylex_fini()
{
	fclose(cfile->f);
	free(cfile->id);
	free(cfile);
}

void*
cf_malloc(size_t size)
{
	void *ptr = malloc(size);
	if (ptr == NULL)
		err(EXIT_FAILURE, NULL);
	return ptr;
}

char*
cf_strdup(const char *str)
{
	char *s = strdup(str);
	if (s == NULL)
		err(EXIT_FAILURE, NULL);
	return s;
}
