"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WORKFLOW_STEP_TYPE = exports.WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP = exports.WORKFLOW_STATE = exports.WORKFLOW_RESOURCE_TYPE = exports.MODEL_STATE = exports.MODEL_ALGORITHM = void 0;
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
/**
 ********** WORKFLOW TYPES/INTERFACES **********
 */
/**
 ********** MODAL SUB-FORM TYPES/INTERFACES **********
 We persist sub-forms in the form modals s.t. data is only
 saved back to the parent form if the user explicitly saves.

 We define the structure of the forms here.
 */
// Ingest docs modal
// Search request modal
// Configure template modal
// Configure expression modal
// Configure multi-expression modal
/**
 ********** WORKSPACE TYPES/INTERFACES **********
 */
/**
 * An input field for a component. Specifies enough configuration for the
 * UI node to render it properly (help text, links, etc.)
 */
/**
 * The base interface the components will implement.
 */
/**
 * We need to include some extra instance-specific data to the ReactFlow component
 * to perform extra functionality, such as deleting the node from the ReactFlowInstance.
 */
/**
 ********** USE CASE TEMPLATE TYPES/INTERFACES **********
 */
// A stateless template of a workflow
// An instance of a workflow based on a workflow template
/**
 ********** ML PLUGIN TYPES/INTERFACES **********
 */
// Based off of https://github.com/opensearch-project/ml-commons/blob/main/common/src/main/java/org/opensearch/ml/common/model/MLModelState.java
let MODEL_STATE = exports.MODEL_STATE = /*#__PURE__*/function (MODEL_STATE) {
  MODEL_STATE["REGISTERED"] = "Registered";
  MODEL_STATE["REGISTERING"] = "Registering";
  MODEL_STATE["DEPLOYING"] = "Deploying";
  MODEL_STATE["DEPLOYED"] = "Deployed";
  MODEL_STATE["PARTIALLY_DEPLOYED"] = "Partially deployed";
  MODEL_STATE["UNDEPLOYED"] = "Undeployed";
  MODEL_STATE["DEPLOY_FAILED"] = "Deploy failed";
  return MODEL_STATE;
}({}); // Based off of https://github.com/opensearch-project/ml-commons/blob/main/common/src/main/java/org/opensearch/ml/common/FunctionName.java
let MODEL_ALGORITHM = exports.MODEL_ALGORITHM = /*#__PURE__*/function (MODEL_ALGORITHM) {
  MODEL_ALGORITHM["LINEAR_REGRESSION"] = "Linear regression";
  MODEL_ALGORITHM["KMEANS"] = "K-means";
  MODEL_ALGORITHM["AD_LIBSVM"] = "AD LIBSVM";
  MODEL_ALGORITHM["SAMPLE_ALGO"] = "Sample algorithm";
  MODEL_ALGORITHM["LOCAL_SAMPLE_CALCULATOR"] = "Local sample calculator";
  MODEL_ALGORITHM["FIT_RCF"] = "Fit RCF";
  MODEL_ALGORITHM["BATCH_RCF"] = "Batch RCF";
  MODEL_ALGORITHM["ANOMALY_LOCALIZATION"] = "Anomaly localization";
  MODEL_ALGORITHM["RCF_SUMMARIZE"] = "RCF summarize";
  MODEL_ALGORITHM["LOGISTIC_REGRESSION"] = "Logistic regression";
  MODEL_ALGORITHM["TEXT_EMBEDDING"] = "Text embedding";
  MODEL_ALGORITHM["METRICS_CORRELATION"] = "Metrics correlation";
  MODEL_ALGORITHM["REMOTE"] = "Remote";
  MODEL_ALGORITHM["SPARSE_ENCODING"] = "Sparse encoding";
  MODEL_ALGORITHM["SPARSE_TOKENIZE"] = "Sparse tokenize";
  MODEL_ALGORITHM["TEXT_SIMILARITY"] = "Text similarity";
  MODEL_ALGORITHM["QUESTION_ANSWERING"] = "Question answering";
  MODEL_ALGORITHM["AGENT"] = "Agent";
  return MODEL_ALGORITHM;
}({}); // Based off of JSONSchema. For more info, see https://json-schema.org/understanding-json-schema/reference/type
// For rendering options, we extract the name (the key in the input/output obj) and combine into a single obj
/**
 ********** MISC TYPES/INTERFACES ************
 */
// Based off of https://github.com/opensearch-project/flow-framework/blob/main/src/main/java/org/opensearch/flowframework/model/State.java
let WORKFLOW_STATE = exports.WORKFLOW_STATE = /*#__PURE__*/function (WORKFLOW_STATE) {
  WORKFLOW_STATE["NOT_STARTED"] = "Not started";
  WORKFLOW_STATE["PROVISIONING"] = "Provisioning";
  WORKFLOW_STATE["FAILED"] = "Failed";
  WORKFLOW_STATE["COMPLETED"] = "Completed";
  return WORKFLOW_STATE;
}({});
// Based off of https://github.com/opensearch-project/flow-framework/blob/main/src/main/java/org/opensearch/flowframework/common/WorkflowResources.java
let WORKFLOW_RESOURCE_TYPE = exports.WORKFLOW_RESOURCE_TYPE = /*#__PURE__*/function (WORKFLOW_RESOURCE_TYPE) {
  WORKFLOW_RESOURCE_TYPE["PIPELINE_ID"] = "Ingest pipeline";
  WORKFLOW_RESOURCE_TYPE["INDEX_NAME"] = "Index";
  WORKFLOW_RESOURCE_TYPE["MODEL_ID"] = "Model";
  WORKFLOW_RESOURCE_TYPE["MODEL_GROUP_ID"] = "Model group";
  WORKFLOW_RESOURCE_TYPE["CONNECTOR_ID"] = "Connector";
  return WORKFLOW_RESOURCE_TYPE;
}({});
let WORKFLOW_STEP_TYPE = exports.WORKFLOW_STEP_TYPE = /*#__PURE__*/function (WORKFLOW_STEP_TYPE) {
  WORKFLOW_STEP_TYPE["CREATE_INGEST_PIPELINE_STEP_TYPE"] = "create_ingest_pipeline";
  WORKFLOW_STEP_TYPE["CREATE_SEARCH_PIPELINE_STEP_TYPE"] = "create_search_pipeline";
  WORKFLOW_STEP_TYPE["CREATE_INDEX_STEP_TYPE"] = "create_index";
  return WORKFLOW_STEP_TYPE;
}({}); // We cannot disambiguate ingest vs. search pipelines based on workflow resource type. To work around
// that, we maintain this map from workflow step type -> formatted type
let WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP = exports.WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP = /*#__PURE__*/function (WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP) {
  WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP["create_ingest_pipeline"] = "Ingest pipeline";
  WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP["create_search_pipeline"] = "Search pipeline";
  WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP["create_index"] = "Index";
  return WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP;
}({});
/**
 ********** OPENSEARCH TYPES/INTERFACES ************
 */
// from https://opensearch.org/docs/latest/ingest-pipelines/simulate-ingest/#example-specify-a-pipeline-in-the-path
// from https://opensearch.org/docs/latest/ingest-pipelines/simulate-ingest/#example-specify-a-pipeline-in-the-path
// from https://opensearch.org/docs/latest/ingest-pipelines/simulate-ingest/#example-specify-a-pipeline-in-the-path
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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