"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenSearchRoutesService = void 0;
exports.registerOpenSearchRoutes = registerOpenSearchRoutes;
var _configSchema = require("@osd/config-schema");
var _common = require("../../common");
var _helpers = require("./helpers");
var _helpers2 = require("../utils/helpers");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
/**
 * Server-side routes to process OpenSearch-related node API calls and execute the
 * corresponding API calls against the OpenSearch cluster.
 */
function registerOpenSearchRoutes(router, opensearchRoutesService) {
  router.get({
    path: `${_common.CAT_INDICES_NODE_API_PATH}/{pattern}`,
    validate: {
      params: _configSchema.schema.object({
        pattern: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.catIndices);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/catIndices/{pattern}`,
    validate: {
      params: _configSchema.schema.object({
        pattern: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.catIndices);
  router.get({
    path: `${_common.GET_MAPPINGS_NODE_API_PATH}/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getMappings);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/mappings/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getMappings);
  router.get({
    path: `${_common.GET_INDEX_NODE_API_PATH}/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getIndex);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/getIndex/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getIndex);
  router.post({
    path: `${_common.SEARCH_INDEX_NODE_API_PATH}/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.searchIndex);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/search/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.searchIndex);
  router.post({
    path: `${_common.SEARCH_INDEX_NODE_API_PATH}/{index}/{search_pipeline}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        search_pipeline: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.searchIndex);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/search/{index}/{search_pipeline}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        search_pipeline: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.searchIndex);
  router.put({
    path: `${_common.INGEST_NODE_API_PATH}/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.ingest);
  router.put({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/ingest/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.ingest);
  router.post({
    path: `${_common.BULK_NODE_API_PATH}/{pipeline}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.bulk);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/bulk/{pipeline}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.bulk);
  router.post({
    path: _common.BULK_NODE_API_PATH,
    validate: {
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.bulk);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/bulk`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.bulk);
  router.post({
    path: _common.SIMULATE_PIPELINE_NODE_API_PATH,
    validate: {
      body: _configSchema.schema.object({
        pipeline: _configSchema.schema.any(),
        docs: _configSchema.schema.any()
      })
    }
  }, opensearchRoutesService.simulatePipeline);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/simulatePipeline`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        pipeline: _configSchema.schema.any(),
        docs: _configSchema.schema.any()
      })
    }
  }, opensearchRoutesService.simulatePipeline);
  router.get({
    path: `${_common.INGEST_PIPELINE_NODE_API_PATH}/{pipeline_id}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getIngestPipeline);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/getIngestPipeline/{pipeline_id}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getIngestPipeline);
  router.get({
    path: `${_common.SEARCH_PIPELINE_NODE_API_PATH}/{pipeline_id}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getSearchPipeline);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/getSearchPipeline/{pipeline_id}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getSearchPipeline);
}
class OpenSearchRoutesService {
  constructor(client, dataSourceEnabled) {
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "dataSourceEnabled", void 0);
    _defineProperty(this, "catIndices", async (context, req, res) => {
      const {
        pattern
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('cat.indices', {
          index: pattern,
          format: 'json',
          h: 'health,index'
        });

        // re-formatting the index results to match Index
        const cleanedIndices = response.map(index => ({
          name: index.index,
          health: index.health
        }));
        return res.ok({
          body: cleanedIndices
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getMappings", async (context, req, res) => {
      const {
        index
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      try {
        var _response$index;
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('indices.getMapping', {
          index
        });

        // Response will be a dict with key being the index name. Attempt to
        // pull out the mappings. If any errors found (missing index, etc.), an error
        // will be thrown.
        const mappings = (_response$index = response[index]) === null || _response$index === void 0 ? void 0 : _response$index.mappings;
        return res.ok({
          body: mappings
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getIndex", async (context, req, res) => {
      const {
        index
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('indices.get', {
          index
        });
        // re-formatting the results to match IndexResponse
        const cleanedIndexDetails = Object.entries(response).map(([indexName, indexDetails]) => ({
          indexName,
          indexDetails
        }));
        return res.ok({
          body: cleanedIndexDetails
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "searchIndex", async (context, req, res) => {
      const {
        index,
        search_pipeline
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      const body = req.body;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('search', {
          index,
          body,
          search_pipeline
        });
        return res.ok({
          body: response
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "ingest", async (context, req, res) => {
      const {
        data_source_id = ''
      } = req.params;
      const {
        index
      } = req.params;
      const doc = req.body;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('index', {
          index,
          body: doc
        });
        return res.ok({
          body: response
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "bulk", async (context, req, res) => {
      const {
        data_source_id = ''
      } = req.params;
      const {
        pipeline
      } = req.params;
      const body = req.body;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('bulk', {
          body,
          pipeline
        });
        return res.ok({
          body: response
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "simulatePipeline", async (context, req, res) => {
      const {
        data_source_id = ''
      } = req.params;
      const {
        pipeline,
        docs
      } = req.body;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('ingest.simulate', {
          body: {
            pipeline,
            docs
          }
        });
        return res.ok({
          body: {
            docs: response.docs
          }
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getIngestPipeline", async (context, req, res) => {
      const {
        pipeline_id
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('ingest.getPipeline', {
          id: pipeline_id
        });
        // re-formatting the results to match IngestPipelineResponse
        const cleanedIngestPipelineDetails = Object.entries(response).map(([pipelineId, ingestPipelineDetails]) => ({
          pipelineId,
          ingestPipelineDetails
        }));
        return res.ok({
          body: cleanedIngestPipelineDetails
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getSearchPipeline", async (context, req, res) => {
      const {
        pipeline_id
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('coreClient.getSearchPipeline', {
          pipeline_id: pipeline_id
        });

        // re-formatting the results to match SearchPipelineResponse
        const cleanedSearchPipelineDetails = Object.entries(response).map(([pipelineId, searchPipelineDetails]) => ({
          pipelineId,
          searchPipelineDetails
        }));
        return res.ok({
          body: cleanedSearchPipelineDetails
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    this.client = client;
    this.dataSourceEnabled = dataSourceEnabled;
  }
}
exports.OpenSearchRoutesService = OpenSearchRoutesService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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