<?php

declare(strict_types=1);

/*
 * Copyright (c) 2017-2022 François Kooman <fkooman@tuxed.net>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

require_once dirname(__DIR__) . '/vendor/autoload.php';

use fkooman\SeCookie\ActiveSession;
use fkooman\SeCookie\MemcacheSessionStorage;

function same($a, $b): void
{
    if ($a !== $b) {
        throw new Exception(var_export($a, true) . ' does not match ' . var_export($b, true));
    }
}

$serverList = array_slice($argv, 1);
if (0 === count($serverList)) {
    $serverList = ['localhost:11211'];
}

echo '### Testing ' . implode(', ', $serverList) . PHP_EOL;

$dateTime = new DateTimeImmutable('2045-10-10T09:00:00+00:00', new DateTimeZone('UTC'));

try {
    // use all servers to write the session data to
    $sessionStorage = new MemcacheSessionStorage($serverList);
    $sessionStorage->store(
        new ActiveSession(
            'session_id',
            $dateTime,
            [
                'foo' => 'bar',
            ]
        )
    );

    // query all servers and get the first result
    $activeSession = $sessionStorage->retrieve('session_id');
    same('2045-10-10T09:00:00+00:00', $activeSession->expiresAt()->format(DateTimeImmutable::ATOM));
    same('bar', $activeSession->get('foo'));

    // loop over all servers individually to verify they have the session data
    // as well
    foreach ($serverList as $memcacheServer) {
        $tmpSessionStorage = new MemcacheSessionStorage([$memcacheServer]);
        $activeSession = $tmpSessionStorage->retrieve('session_id');
        same('bar', $activeSession->get('foo'));
    }

    $sessionStorage->destroy('session_id');
    same(null, $sessionStorage->retrieve('session_id'));

    // make sure session is gone everywhere
    foreach ($serverList as $memcacheServer) {
        $tmpSessionStorage = new MemcacheSessionStorage([$memcacheServer]);
        same(null, $tmpSessionStorage->retrieve('session_id'));
    }
    echo 'OK' . PHP_EOL;
} catch (Exception $e) {
    echo 'ERROR: ' . $e->getMessage() . PHP_EOL;
}
