/**********/
/* Macros */
/**********/

// Get the angles necessary to rotate an object from the
// +z axis, to parallel the vector from start to end
// Parameters:
//     start: The start point of the vector to align with
//     end:   The end point of the vector to align with

#macro get_angle (vstart, vend)
    #local tana = vend.x - vstart.x;
    #local tanb = vend.y - vstart.y;
    #if (vstart.x != vend.x)
        #local zangle = degrees(atan2(tana, tanb));
    #else
        #local zangle = 0;
    #end

    #local tana = vend.y - vstart.y;
    #local tanb = vend.z - vstart.z;
    #local xangle = degrees(atan2(tana, tanb));

    <-xangle, 0, -zangle>
#end

/*******************/
/* Clock Functions */
/*******************/

// Compute a sawtooth wave of motion based on the clock variable
// Parameters:
//     currclock: The original clock variable

#macro SawToothClock (currclock)
    #if (currclock <= 0.5)
        currclock*2
    #else
        (1 - currclock)*2
    #end
#end

// Compute a sine wave of motion based on the clock variable
// Parameters:
//     currclock: The original clock variable

#macro SineWaveClock (currclock)
    sin(3.14159265359*currclock)
#end

// Convert a sub-interval of the clock to an entire 0...1 range
// Parameters:
//     startval:  The starting value
//     endval:    The starting value
//     intclock:  The "time" variable

#macro subinterval (startval, endval, intclock)
    #local intlength = endval - startval;
    #local outclock = intclock - startval;
    outclock / intlength
#end

/**********************************/
/* "Physics of Throwing" formulas */
/**********************************/

// Get the end position of a rotated lever (the thrower's arm)
// Parameters:
//     motion_angle:  The angle the lever is rotated
//     motion_radius: The radius of the lever

#macro get_startpos (motion_angle, motion_radius)
    vrotate(motion_radius*z, motion_angle*x)
#end

// Get the end velocity of a rotated lever (the thrower's arm)
// Parameters:
//     motion_angle:  The total angle the lever is rotated
//     final_angle:   The angle of rotation at the end of its motion
//     motion_radius: The radius of the lever
//     time:          The time it takes to go through that angle (as
//                    fractions of the full run of the clock variable)

#macro get_startvelvector (motion_angle, final_angle, motion_radius, time)
    #local vector = vrotate(motion_radius*z, final_angle*x);
    #local tangent = vrotate(motion_radius*z, (final_angle-90)*x);
    #local tangent = vnormalize(tangent);
    
    #local speed = abs(motion_radius*(3.1415926/180)*motion_angle/time);
    
//    #render concat("vector = <", str(vector.x,5,2), ", ", str(vector.y,5,2))
//    #render concat(", ", str(vector.z,5,2), ">\n")
    
//    #render concat("tangent = <", str(tangent.x,5,2), ", ", str(tangent.y,5,2))
//    #render concat(", ", str(tangent.z,5,2), ">\n")

//    #render concat("speed = ", str(speed,5,2), "\n")
    
    speed*tangent
#end

#macro get_startvelvector2 (motion_angle, final_angle, motion_radius, time)
    #local vector = vrotate(motion_radius*z, final_angle*x);
    #local tangent = vrotate(motion_radius*z, (final_angle-90)*x);
    #local tangent = vnormalize(tangent);
    
    #local speed = abs(motion_radius*(3.1415926/180)*motion_angle/time);
    
//    #render concat("vector = <", str(vector.x,5,2), ", ", str(vector.y,5,2))
//    #render concat(", ", str(vector.z,5,2), ">\n")
    
//    #render concat("tangent = <", str(tangent.x,5,2), ", ", str(tangent.y,5,2))
//    #render concat(", ", str(tangent.z,5,2), ">\n")

//    #render concat("speed = ", str(speed,5,2), "\n")
    
    cylinder { 0, vector,  0.1 pigment { color Red  } }
    cylinder { 0, tangent, 0.1 pigment { color Blue } }
#end

// Interpolate between two values
// Parameters:
//     startval: The starting value
//     endval:   The starting value
//     intclock: The "time" variable

#macro interp_values (startval, endval, intclock)
    startval + intclock*(endval - startval)
#end

// Projectile motion
// Parameters:
//     startvel:  The starting velocity
//     currclock: The "time" variable

#macro projectile (startvel, currclock)
    #local projx = startvel.x*currclock;
    #local projy = startvel.y*currclock - 0.5*gravconst*currclock*currclock;
    #local projz = startvel.z*currclock;
    <projx, projy, projz>
#end

