// Persistence of Vision Ray Tracer Scene Description File
// File: MrRobot.inc
// Vers: 3.1
// Desc: Include file with all of my "Mr Robot" Macro's
// Date: 01/14/00
//
// COPYRIGHT: I SUBMIT TO THE STANDARD RAYTRACING COMPETITION COPYRIGHT.
// Copyright (c) Dan Lauer, 2000                               
//
// mailto:dlauer@optonline.net
//
// This set of macros were originally created for use in my "Robot Factory" Animation.
// That animation was created specifically for entry into the January 15, 2000 round of the
// Internet Ray Tracing Competition.  See http://www.irtc.org/.
//
// The code in this and the accompanying files may be used freely for non-commercial
// purposes without permission.  However, I would love to here from anyone with questions
// or comments.  Commercial uses other than those described in the STANDARD RAYTRACING
// COMPETITION COPYRIGHT require permission from the author.  Please contact me via e-mail.
//
//
// **************     Macros defined    *************************************************
//
// Head(head_power, head_rotation, eye_rotation, color_index)
//    head_power:     on or off / light in eyes on
//    head_rotation:  about z axis = scalar degrees / neg = right / pos = left
//    eye_rotation:   about z axis = scalar degrees / neg = right / pos = left
//    Note: Head is translate z * 1.7 when attached to body
//
//  Leg(length, leg_angle, ankle_angle, color_index)
//    length:         length of shin, range 2.1 - 3.3; normal = 2.8
//    leg_angle:      rotation at hip, scalar degrees, pos is forward
//    ankle_angle:    rotation at ankle, scalar degrees, pos is forward         
//    Note: Leg is translated -1.2 * z when attached to body
//
//  FrontPanel(power)
//    power:          on/off
//    Note: FrontPanel is translated -0.475*x when attached to body
//
//  RightArm(shoulder_vec, elbow_vec, length, hand_vec, thumb_angle, color_index)
//  LeftArm( shoulder_vec, elbow_vec, length, hand_vec, thumb_angle, color_index)
//    should_vec:     x = neg:in/pos:out(right)   neg:out/pos:in(left)
//                    y = pos:fwd / neg:back
//                    z = not used
//    elbow_vec:      x = neg:in/pos:out(right)   neg:out/pos:in(left)
//                    y = pos:fwd / neg:back
//                    z = not used
//    length:         (elbow to wrist) range=1.1-1.7  normal is 1.5
//    hand_vec:       x = neg:in/pos:out(right)   neg:out/pos:in(left)
//                    y = pos:fwd / neg:back
//                    z = rotation
//    thumb_angle:    0 closed, 60 open
//              
//  MrRobot( power, head_rotation, eye_rotation, leg_length, leg_angle, ankle_angle,
//           r_shoulder_vec, r_elbow_vec, r_length, r_hand_vec, r_thumb_angle,
//           l_shoulder_vec, l_elbow_vec, l_length, l_hand_vec, l_thumb_angle, color_index)
//
//

// ********  MATERIALS  *******
#include "robotz.inc"

#ifndef (LED_frame_rate)
#declare LED_frame_rate = 1;
#end

#declare RobotShell_mat = array[12] {
  RobotBlue,  RobotGreen, RobotGreen,  RobotGreen,
  RobotGreen, RobotGreen, RobotGreen,  RobotGreen,
  RobotGreen, RobotGreen, RobotGreen,  RobotGreen
}                                    

#declare CircuitBoardTex = material {   
  texture {
    marble
    texture_map {
      [0.0  pigment { color rgb <0.114567, 0.614567, 0.104167> }               
            finish  { ambient 0.1       phong 0.266167         }
      ]
      [0.6  pigment { color rgb <0.114567, 0.614567, 0.104167> }               
            finish  { ambient 0.1       phong 0.266167         }
      ]
      [0.65 pigment { color rgb <0.583333, 0.583333, 0.583333> }               
            finish  { ambient 0.1       reflection 0.302133    }
      ]
      [0.8  pigment { color rgb <0.583333, 0.583333, 0.583333> }               
            finish  { ambient 0.1       reflection 0.302133    }
      ]
      [0.81 pigment {
              marble
              color_map {
                [ 0.0     rgbft <0.927067, 0.0, 0.0, 0.0, 0.0> ]
                [ 1.0     rgbft <1.0, 1.0, 0.052067, 0.0, 0.0> ]
              }
            }               
            normal { marble , 0.5   }               
            finish { ambient 0.1  phong 0.4964 }
      ]
      [1.0  pigment {
              marble
              color_map {
                [ 0.0     rgbft <0.927067, 0.0, 0.0, 0.0, 0.0> ]
                [ 1.0     rgbft <1.0, 1.0, 0.052067, 0.0, 0.0> ]
              }
            }               
            normal { marble , 0.5   }               
            finish { ambient 0.1  phong 0.4964 }
      ]
    }
    turbulence 3.0
    triangle_wave
  }
}


//
// ********  REFERENCED OBJECTS  *******
//

#local RightShoulderCuff = object {
  cylinder { // RightShoulderCuff
    <0,0,1>, <0,0,0>, 1 
    scale <0.25, 0.25, 0.1>
    rotate -90.0*x
    translate  <0.0, 0.9, 0.7>
  }
}
#local MotorRing1 = object {
  torus { // MotorRing1
    0.15, 0.05  rotate -x*90
    rotate -90.0*x
    translate  0.05*y
  }
}
#local RightHipMotor = union {
  object { MotorRing1 }
  object { // MotorRing2->MotorRing1
    MotorRing1
    translate  0.1*y
  }
  object { // MotorRing3->MotorRing1
    MotorRing1
    translate  0.2*y
  }
  scale <1.0, 2.0, 1.0>
  translate  <0.0, 0.15, -1.2>
}
#local LeftHipMotorMount = object {
  superellipsoid { // LeftHipMotorMount
    <0.2, 0.2>
    scale <0.2, 0.3, 0.05>
    translate  <0.0, -0.45, -1.0>
  }
}
#declare CornerBoltBottomLeft = object {
  box { // CornerBoltBottomLeft
    <-1, -1, -1>, <1, 1, 1>
    scale <0.2, 0.05, 0.05>
    translate  <-0.25, -0.75, -0.85>
  }
}
#local LeftEyeSocket = object {
  sphere { // LeftEyeSocket
    <0,0,0>,1
    scale 0.1
    translate  <-0.4, -0.2, 0.1>
  }
}
#declare RightAnkle = object {
  cylinder { // RightAnkle
    <0,0,1>, <0,0,0>, 1 
    scale <0.15, 0.15, 0.1>
    rotate -90.0*x
    translate  <0.0, 0.15, 0.5>
  }
}
#declare RightAnkle_Support = object {
  box { // RightAnkle_Support
    <-1, -1, -1>, <1, 1, 1>
    scale <0.15, 0.05, 0.11>
    translate  <0.0, 0.2, 0.4>
  }
}
#declare Tire1 = object {
  cylinder { // Tire1
    <0,0,1>, <0,0,0>, 1 
    material {
      Rubber_Tire
    }
    scale 0.1
    rotate -90.0*x
    translate  <0.5, 0.5, 0.1>
  }
}
#declare Axle1 = object {
  cylinder { // Axle1
    <0,0,1>, <0,0,0>, 1 
    material {
      Chrome_Metal
    }
    scale <0.03, 0.03, 1.36>
    rotate -90.0*x
    translate  <0.5, -0.68, 0.1>
  }
}
#declare Shoulder = union {
  cylinder { // ShoulderShaft
    <0,0,1>, <0,0,0>, 1 
    scale <0.1, 0.1, 1.0>
    rotate 90.0*x
    translate  0.15*y
  }
  cylinder { // ShoulderPin
    <0,0,1>, <0,0,0>, 1 
    scale <0.05, 0.05, 0.4>
    rotate -90.0*y
    translate  0.2*x
  }
  material {
    Chrome_Metal
  }
}
#declare ArmSoulderRing = object {
  cylinder { // ArmSoulderRing
    <0,0,1>, <0,0,0>, 1 
    scale <0.15, 0.15, 0.05>
    rotate -90.0*y
    translate  -0.12*x
  }
}
#local ArmBone1 = object {
  box { // ArmBone1
    <-1, -1, -1>, <1, 1, 1>
    scale <0.025, 0.15, 0.45>
    translate  <-0.145, 0.0, 0.45>
  }
}
#local UpperArmCSG = union {
  box { // ArmBone3
    <-1, -1, -1>, <1, 1, 1>
    scale <0.12, 0.025, 0.6>
  }
  object { ArmBone1 }
  object { // ArmBone2->ArmBone1
    ArmBone1
    translate  0.29*x
  }
}
#declare UpperArmBone = intersection {
  cone { // UpperArmCone
    <0,0,0>, 0.07, <0,0,1>, 0.25
    scale <1.0, 1.0, 1.5>
    translate  -0.6*z
  }
  object { UpperArmCSG }
  translate  -0.9*z
}
#macro UpperArm(color_index) 
union {
  object { ArmSoulderRing }
  object { // ArmShoulderRing2->ArmSoulderRing
    ArmSoulderRing
    rotate -0.0*y
    translate  0.29*x
  }
  object { UpperArmBone }
  sphere { // ElbowBall
    <0,0,0>,1
    material {
      Chrome_Metal
    }
    scale 0.1
    translate  -1.5*z
  }
  box { // UpperArmGrabPoint
    <-1, -1, -1>, <1, 1, 1>
    scale <0.01, 0.1, 0.1>
    translate  <0.0, 0.15, -0.2>
  }
  material {
    RobotShell_mat[color_index]
  }
}
#end

#local ElbowRing = object {
  torus { // ElbowRing
    0.075, 0.015  rotate -x*90
    rotate -90.0*x
    translate  0.087*y
  }
}
#local ElbowRingWeld = object {
  box { // ElbowRingWeld
    <-1, -1, -1>, <1, 1, 1>
    scale <0.06, 0.009, 0.02>
    translate  <0.0, 0.084, -0.085>
  }
}
#declare LowerArmPiston = union {
  cylinder { // ArmPiston
    <0,0,1>, <0,0,0>, 1 
    scale <0.1, 0.1, 0.8>
    translate  -0.9*z
  }
  cylinder { // ArmPistonRing
    <0,0,1>, <0,0,0>, 1 
    scale <0.12, 0.12, 0.02>
    translate  -0.93*z
  }
  object { ElbowRing }
  object { ElbowRingWeld }
  object { // ElbowRingWeld2->ElbowRingWeld
    ElbowRingWeld
    rotate -0.0*x
    translate  -0.166*y
  }
  object { // ElbowRing2->ElbowRing
    ElbowRing
    translate  -0.174*y
  }
  box { // ArmGrabPoint
    <-1, -1, -1>, <1, 1, 1>
    scale <0.01, 0.1, 0.1>
    translate  <0.0, 0.1, -0.8>
  }
  material {
    Polished_Brass
  }
}
#declare LowerArmShaft = union {
  cylinder { // ArmShaft
    <0,0,1>, <0,0,0>, 1 
    scale <0.05, 0.05, 0.95>
  }
  sphere { // WristBall
    <0,0,0>,1
    scale 0.1
  }
  material {
    Chrome_Metal
  }
  translate  -1.5*z
}
#declare ThumbBone = difference {
  box { // ThumbBone_pos
    <-1, -1, -1>, <1, 1, 1>
    scale <0.04, 0.01, 0.175>
    translate  <-0.035, -0.01, -0.165>
  }
  box { // ThumbBone_neg
    <-1, -1, -1>, <1, 1, 1>
    scale <0.03, 0.03, 0.2>
    rotate -10.0*y
    translate  <-0.08, 0.0, -0.158232>
  }
}             

#macro Thumb(thumb_angle) 
union {
  box { // ThumbSurface
    <-1, -1, -1>, <1, 1, 1>
    material {
      Rubber_Tire
    }
    scale <0.007, 0.03, 0.15>
    translate  -0.2*z
  }
  cylinder { // ThumnBoneJoint
    <0,0,1>, <0,0,0>, 1 
    scale <0.04, 0.04, 0.027>
    rotate 90.0*x
  }
  object { ThumbBone }
  rotate thumb_angle*y
  translate  <-0.03, 0.0, -0.15>
}
#end

#local Wrist = difference {
  sphere { // Wrist_pos
    <0,0,0>,1
    scale 0.12
  }
  cone { // Wrist_neg
    <0,0,0>, 0.0, <0,0,1>, 1.0
    scale <0.3, 0.3, 0.15>
  }
}
#local HandBack = difference {
  box { // HandBack_pos
    <-1, -1, -1>, <1, 1, 1>
    scale <0.05, 0.02, 0.2>
    translate  <0.05, 0.0, -0.29>
  }
  box { // HandBack_neg
    <-1, -1, -1>, <1, 1, 1>
    scale <0.05, 0.05, 0.25>
    rotate 12.573558*y
    translate  <0.115, 0.0, -0.3>
  }
}
#declare HandStuff = union {
  object { Wrist }
  object { HandBack }
  cylinder { // ThumbJoint
    <0,0,1>, <0,0,0>, 1 
    scale <0.06, 0.06, 0.02>
    rotate -90.0*x
    translate  <-0.03, 0.0, -0.15>
  }
  box { // Palm
    <-1, -1, -1>, <1, 1, 1>
    material {
      Rubber_Tire
    }
    scale <0.01, 0.03, 0.15>
    translate  -0.35*z
  }
  cylinder { // ThumbJoint1
    <0,0,1>, <0,0,0>, 1 
    material {
      Chrome_Metal
    }
    scale <0.015, 0.015, 0.06>
    rotate -90.0*x
    translate  <-0.03, -0.03, -0.15>
  }
}
#declare LEDBezel = object {
  box { // LEDBezel
    <-1, -1, -1>, <1, 1, 1>
    material {
      Chrome_Metal
    }
    scale <0.054, 0.029, 0.01>
    rotate 90.0*y
    translate  -0.01*x
  }
}
#local TaperSweep002 = prism {
  conic_sweep
  linear_spline
  0.0,1.0
  5  // control points
  <-1.0, -1.0>,
  <1.0, -1.0>,
  <1.0, 1.0>,
  <-1.0, 1.0>,
  <-1.0, -1.0>
  rotate <90,0,90>
  translate  -2.0*z
}
#declare LED_Lens = union {
  box { // Cube032
    <-1, -1, -1>, <1, 1, 1>
  }
  object { TaperSweep002 }
  material {
    RedPlastic_off
  }
  scale <0.05, 0.025, 0.015>
  rotate 90.0*y
  translate  -0.015*x
}

//
// ********  OBJECTS  *******
//

#local TorsoShell_Pos = union {
  superellipsoid { // MainTorsoShell
    <0.1, 0.1>
    scale <0.5, 0.9, 1.0>
  }
  object { RightShoulderCuff }
  object { // LeftShoulderCuff->RightShoulderCuff
    RightShoulderCuff
    rotate 0.0*x
    translate  -1.9*y
  }
  cone { // Neck
    <0,0,0>, 0.2, <0,0,1>, 0.15
    scale <1.0, 1.0, 0.3>
    translate  1.0*z
  }
  box { // TorsoLowerGrabPoint
    <-1, -1, -1>, <1, 1, 1>
    scale <0.01, 0.1, 0.1>
    translate  <0.0, 0.9, -0.8>
  }
  box { // TorsoUpperGrabPoint
    <-1, -1, -1>, <1, 1, 1>
    scale <0.01, 0.1, 0.1>
    translate  <0.0, 0.9, 1.0>
  }
}
#local TorsoShellCutouts = union {
  box { // MainTorsoCutou
    <-1, -1, -1>, <1, 1, 1>
    scale <0.55, 0.8, 0.9>
    translate  -0.15*x
  }
  cylinder { // ShoulderSockets
    <0,0,1>, <0,0,0>, 1 
    scale <0.1, 0.1, 2.2>
    rotate -90.0*x
    translate  <0.0, -1.1, 0.7>
  }
  cylinder { // NeckCutout
    <0,0,1>, <0,0,0>, 1 
    scale <0.05, 0.06, 0.5>
    translate  0.85*z
  }
}
#declare TorsoShell = difference {
  object { TorsoShell_Pos }
  object { TorsoShellCutouts }
}
#declare TorsoMotors = union {
  object { RightHipMotor }
  object { // LeftHipMotor->RightHipMotor
    RightHipMotor
    scale -1.0
    rotate <-180.0, 0.0, -180.0>
  }
  object { // RightShoulderMotor->RightHipMotor
    RightHipMotor
    translate  <0.0, 0.1, 1.9>
  }
  object { // LeftShoulderMotor->RightHipMotor
    RightHipMotor
    translate  <0.0, -0.95, 1.9>
  }
  object { LeftHipMotorMount }
  object { // RightHipMotorMount->LeftHipMotorMount
    LeftHipMotorMount
    translate  0.9*y
  }
  material {
    MotorWinding
  }
}
#macro Torso(color_index)
union {
  object { TorsoShell }
  object { TorsoMotors }
  object { CornerBoltBottomLeft }
  object { // CornerBoltTopRight->CornerBoltBottomLeft
    CornerBoltBottomLeft
    translate  <0.0, 1.5, 1.7>
  }
  object { // CornetBoltTopLeft->CornerBoltBottomLeft
    CornerBoltBottomLeft
    rotate <-0.0, 0.0, 0.0>
    translate  <0.0, 0.0, 1.7>
  }
  object { // CornerBoltBottomRight->CornerBoltBottomLeft
    CornerBoltBottomLeft
    rotate <-0.0, 0.0, 0.0>
    translate  <0.0, 1.5, -0.0>
  }
  material {
    RobotShell_mat[color_index]
  }
}
#end

#declare HipJoint = union {
  cylinder { // HipJointShaft
    <0,0,1>, <0,0,0>, 1 
    scale <0.1, 0.1, 1.6>
    rotate -90.0*x
  }
  box { // HipJointGrabTab
    <-1, -1, -1>, <1, 1, 1>
    scale <0.01, 0.05, 0.09>
    translate  -0.05*y
  }
  material {
    Chrome_Metal
  }
  translate  <0.0, -0.8, -1.2>
}
#local EyeSockets = union {
  object { LeftEyeSocket }
  object { // RightEyeSocket->LeftEyeSocket
    LeftEyeSocket
    translate  0.4*y
  }
  cone { // NeckHole
    <0,0,0>, 0.0, <0,0,1>, 0.3
    scale 0.5
    rotate -180.0*x
  }
}
#declare Skull = difference {
  sphere { // Skull1
    <0,0,0>,1
    scale 0.5
  }
  object { EyeSockets }
  material {
    Porcilen
  }
}

#macro Eye(eye_power, eye_rotation)  // Define the EYE macro
union {
  sphere { // EyeGlass
    <0,0,0>,1
    #if (eye_power > 0)
      material {
        EyeOn
      }
    #else
      material {
        EyeOff
      }
    #end
    scale 0.08
  }
  sphere { // EyePupil
    <0,0,0>,1
    material {
      Pupil
    }
    scale 0.03
    translate  -0.06*x
  }
  rotate z*eye_rotation
}     
#end  //***** End the Eye macro

#declare HelmetCutouts = union {
  box { // HelmetBottomCutout
    <-1, -1, -1>, <1, 1, 1>
    scale <0.7, 0.7, 0.4>
    rotate 40.0*y
    translate  <-0.2, 0.0, -0.3>
  }
  cylinder { // HelmetEarCutouts
    <0,0,1>, <0,0,0>, 1 
    scale <0.15, 0.15, 1.2>
    rotate -90.0*x
    translate  -0.6*y
  }
}
#macro Helmet(color_index) 
difference {
  sphere { // HelmetShell
    <0,0,0>,1
    scale 0.55
  }
  object { HelmetCutouts }
  material {
    RobotShell_mat[color_index]
  }
}                                 
#end

//************* Macro for Head **************/
#macro Head(head_power, head_rotation, eye_rotation, color_index)
union {
  object { Skull }
  object { 
    Eye(head_power, eye_rotation)  //Left Eye
    translate  <-0.4, -0.2, 0.1>
  }
  object { 
    Eye(head_power, eye_rotation)  //Right Eye
    translate  <-0.4, 0.2, 0.1>
  }
  cone { // Nose
    <0,0,0>, 0.05, <0,0,1>, 0.0
    open
    material {
      Chrome_Metal
    }
    hollow
    scale <1.0, 1.0, 0.15>
    translate  <-0.5, 0.0, -0.15>
  }
  sphere { // Mouth
    <0,0,0>,1
    material {
      WireGrill
    }
    scale <0.03, 0.15, 0.075>
    rotate -30.0*y
    translate  <-0.433, 0.0, -0.25>
  }
  cylinder { // LeftEar
    <0,0,1>, <0,0,0>, 1 
    material {
      Chrome_Metal
    }
    scale <0.1, 0.08, 0.02>
    rotate -100.0*y
    translate  -0.5*y
  }
  cylinder { // RightEar
    <0,0,1>, <0,0,0>, 1 
    material {
      Chrome_Metal
    }
    scale <0.1, 0.08, 0.02>
    rotate -100.0*y
    translate  0.5*y
  }
  torus { // Lips
    0.15, 0.01  rotate -x*90
    material {
      Chrome_Metal
    }
    scale <0.5, 1.0, 1.0>
    rotate 60.0*y
    translate  <-0.433, 0.0, -0.25>
  }
  object { Helmet(color_index) }
  box { // HeadGrabPoint
    <-1, -1, -1>, <1, 1, 1>
    material {
      RobotShell_mat[color_index]
    }
    scale <0.02, 0.1, 0.1>
    translate  <0.0, 0.5, 0.3>
  }
  cylinder { // NeckPost
    <0,0,1>, <0,0,0>, 1 
    material {
      Chrome_Metal
    }
    scale <0.04, 0.04, 0.6>
    translate  -0.6*z
  }
  rotate z * head_rotation
  translate  1.7*z
}
#end

#local FootShell_Pos = intersection {
  superellipsoid { // MainFootShell
    <0.2, 0.2>
    scale <0.75, 0.75, 0.3>
  }
  box { // FootShellCube
    <-1, -1, -1>, <1, 1, 1>
    scale <0.8, 0.8, 0.15>
    translate  0.21*z
  }
}
#declare FootShell = difference {
  object { FootShell_Pos }
  box { // FootShellCutout
    <-1, -1, -1>, <1, 1, 1>
    scale <0.65, 0.65, 0.25>
  }
}
#macro FootCasing(color_index) 
union {
  object { FootShell }
  object { RightAnkle }
  object { RightAnkle_Support }
  object { // LeftAnkleSupport->RightAnkle Support
    RightAnkle_Support
    rotate 0.0*x
    translate  <0.0, -0.4, -0.0>
  }
  object { // LeftAnkle->RightAnkle
    RightAnkle
    rotate -0.0*x
    translate  -0.4*y
  }
  material {
    RobotShell_mat[color_index]
  }
}
#end

#macro Foot(color_index)
union {
  object { FootCasing(color_index) }
  object { Tire1 }
  object { // Tire2->Tire1
    Tire1
    rotate -0.0*x
    translate  <0.0, -1.0, 0.0>
  }
  object { // Tire3->Tire1
    Tire1
    rotate -0.0*x
    translate  <-1.0, -1.0, 0.0>
  }
  object { // Tire4->Tire1
    Tire1
    rotate -0.0*x
    translate  <-1.0, 0.0, 0.0>
  }
  object { Axle1 }
  cylinder { // Ankle
    <0,0,1>, <0,0,0>, 1 
    material {
      Chrome_Metal
    }
    scale <0.075, 0.075, 0.52>
    rotate 90.0*x
    translate  <0.0, 0.26, 0.5>
  }
  object { // Axle2->Axle1
    Axle1
    rotate <-0.0, -0.0, -0.0>
    translate  -1.0*x
  }
  translate  -0.5*z
}
#end

#macro LowerLeg(length, foot_vec, color_index) 
union {
  #local leg_length = min(max(3.5-length, 0.2), 1.4);   
  cylinder { // AnkleFitting
    <0,0,1>, <0,0,0>, 1 
    scale <0.175, 0.175, 0.28>
    rotate 90.0*x
    translate  0.14*y
  }
  cylinder { // Shin
    <0,0,1>, <0,0,0>, 1 
    scale <0.1, 0.1, 1.45>
  }
  object { 
    Foot(color_index)
    rotate <0, foot_vec.y ,0>
  }
  material {
    Chrome_Metal
  }                      
  rotate <0, 0, foot_vec.z>
  translate  leg_length*z
}
#end

#local HipLegFitting = difference {
  cylinder { // HipLegFitting_pos
    <0,0,1>, <0,0,0>, 1 
    scale <0.25, 0.25, 0.3>
    rotate 90.0*x
    translate  0.15*y
  }
  cylinder { // HipLegFitting_neg
    <0,0,1>, <0,0,0>, 1 
    scale <0.1, 0.1, 0.4>
    rotate 90.0*x
    translate  0.2*y
  }
  translate  1.6*z
}
#declare UpperLeg = union {
  cylinder { // ThighRing
    <0,0,1>, <0,0,0>, 1 
    scale <0.305, 0.305, 0.01>
    translate  0.18*z
  }
  cylinder { // Thigh
    <0,0,1>, <0,0,0>, 1 
    scale <0.3, 0.3, 1.1>
    translate  0.2*z
  }
  object { HipLegFitting }
  box { // HipLegWeld
    <-1, -1, -1>, <1, 1, 1>
    scale <0.2, 0.15, 0.1>
    translate  1.4*z
  }
  box { // LegGrabPoint
    <-1, -1, -1>, <1, 1, 1>
    scale <0.02, 0.1, 0.1>
    translate  <0.0, -0.3, 0.1>
  }
  material {
    Polished_Brass
  }
  translate  1.4*z
}
#macro Leg(length, leg_angle, ankle_angle, color_index)
union {                                
  object { LowerLeg(length, ankle_angle, color_index) }
  object { UpperLeg }
  translate  -3.0*z                       
  rotate leg_angle * y
  translate -1.2*z 
}
#end

#macro Hand(hand_vec,thumb_angle, color_index)
union {
  object { Thumb(thumb_angle) }
  object { HandStuff }
  material {
    RobotShell_mat[color_index]
  }   
  rotate hand_vec
  translate  -1.5*z
}     
#end

#macro LowerLowerArm(length, hand_vec, thumb_angle, color_index)
union {
  object { LowerArmShaft }
  object { Hand(hand_vec,thumb_angle, color_index) }  
  translate  (1.5-length)*z
}     
#end

#macro LowerRightArm(elbow_vec, length, hand_vec, thumb_angle, color_index) 
union {
  object { LowerLowerArm(length, hand_vec, thumb_angle, color_index)  }
  object { LowerArmPiston }
  rotate elbow_vec
  translate  -1.5*z
}     
#end

#macro LowerLeftArm(elbow_vec, length, hand_vec, thumb_angle, color_index) 
union {
  object { LowerLowerArm(length, hand_vec, thumb_angle, color_index)  }
  object { 
    LowerArmPiston
    rotate -180.0*z
  }
  rotate elbow_vec
  translate  -1.5*z
}
#end

#macro UpperRightArm(elbow_vec, length, hand_vec, thumb_angle, color_index)
union {
  object { UpperArm(color_index) }
  object { LowerRightArm(elbow_vec, length, hand_vec, thumb_angle, color_index) }
}     
#end

#macro UpperLeftArm(elbow_vec, length, hand_vec, thumb_angle, color_index)
union {
  object { 
    UpperArm(color_index)
    rotate -180.0*z
  }
  object { LowerLeftArm(elbow_vec, length, hand_vec, thumb_angle, color_index) }
}                                                                
#end

// ********* Right Arm Macro   ***************************************************
#macro RightArm(shoulder_vec, elbow_vec, length, hand_vec, thumb_angle, color_index)
union {
  object { 
    Shoulder
    rotate <0, shoulder_vec.y, 0>
  }
  object { 
    UpperRightArm(elbow_vec, length, hand_vec, thumb_angle, color_index)
    rotate <shoulder_vec.x, shoulder_vec.y, 0>    
  }
  translate  <0.0, 1.25, 0.7>
}
#end   // ******* end right Arm macro ***********************

// ********* Left Arm Macro   ***************************************************
#macro LeftArm(shoulder_vec, elbow_vec, length, hand_vec, thumb_angle, color_index)   
union {
  object {
    Shoulder
    scale <1.0, -1.0, 1.0>
    rotate <0, shoulder_vec.y, 0>
  }
  object { 
    UpperLeftArm(elbow_vec, length, hand_vec, thumb_angle, color_index)
    rotate <shoulder_vec.x, shoulder_vec.y, 0>
  }
  translate  <0.0, -1.25, 0.7>
}  
#end   // ******* end left Arm macro ***********************

#local TaperSweep001 = prism {
  conic_sweep
  linear_spline
  0.0,1.0
  9  // control points
  <-0.5, -0.8>,
  <-0.3, -1.0>,
  <0.3, -1.0>,
  <0.5, -0.8>,
  <0.5, 0.8>,
  <0.3, 1.0>,
  <-0.3, 1.0>,
  <-0.5, 0.8>,
  <-0.5, -0.8>
  rotate <90,0,90>
  scale <1.0, 1.0, 0.75>
  translate  -0.75*z
}
#declare ControlPanel = intersection {
  object { TaperSweep001 }
  box { // Cube031
    <-1, -1, -1>, <1, 1, 1>
    scale <1.1, 0.6, 0.0125>
  }
  material {
    LogoMap
  }
  scale 0.6
  rotate <-90.0, 0.0, -90.0>
  translate  <-0.025, 0.0, 0.4>
}
#declare CB_Post = object {
  cylinder { // CB_Post
    <0,0,1>, <0,0,0>, 1 
    material {
      Chrome_Metal
    }
    scale <0.02, 0.02, 0.6>
    rotate 90.0*y
    translate  <0.0, 0.5, 0.6>
  }
}
#declare CircuitBoard = object {
  box { // CircuitBoard
    <-1, -1, -1>, <1, 1, 1>
    material {
      CircuitBoardTex
    }
    scale <0.02, 0.6, 0.7>
    translate  0.25*x
  }
}


#macro LED(blink)
union {
  object { LEDBezel }
  object { 
    LED_Lens
    #if(blink > 0)
      material {
        RedPlastic_on
      }
    #else
      material {
        RedPlastic_off
      }
    #end    
  }
}
#end

#macro FrontPanel(power)  // ********* The Front Panel Macro  ***********  
#if (power > 0.0 & clock_delta != 0)
  #local frame_number = clock / clock_delta;
#else
  #local frame_number = 0;
#end  
#local led1_state = mod(int(frame_number/(LED_frame_rate * 1)), 2);
#local led2_state = mod(int(frame_number/(LED_frame_rate * 2)), 2);
#local led3_state = mod(int(frame_number/(LED_frame_rate * 4)), 2);
#local led4_state = mod(int(frame_number/(LED_frame_rate * 8)), 2);
union {
  box { // FrontPanelMain
    <-1, -1, -1>, <1, 1, 1>
    material {
      Dark_Brushed_Aluminum
    }
    scale <0.025, 0.8, 0.9>
  }
  object { ControlPanel }
  sphere { // PowerLamp
    <0,0,0>,1
    #if (power > 0.0)
      material {
        AmberPlasticLens_On
      }
    #else
      material {
        AmberPlastic_Off
      }
    #end    
    scale 0.05
    translate  <-0.025, 0.35, 0.5>
  }
  #if (power > 0.0)
    sphere { // Sphere014
      <0,0,0>,1
      material {
        Amber_aura
      }
      hollow
      scale 0.1
      translate  <-0.025, 0.35, 0.5>
    }
  #end
  torus { // Power Lamp Bezel
    0.05, 0.002  rotate -x*90
    material {
      Chrome_Metal
    }
    rotate -90.0*y
    translate  <-0.05, 0.35, 0.5>
  }
  object { 
    LED(led1_state)
    translate  <-0.025, 0.1, 0.5>    
  }
  object { 
    LED(led2_state)
    translate  <-0.025, -0.05, 0.5>    
  }
  object { 
    LED(led3_state)
    translate  <-0.025, -0.2, 0.5>    
  }
  object { 
    LED(led4_state)
    translate  <-0.025, -0.35, 0.5>    
  }
  object { CB_Post }
  object { CircuitBoard }
  object { // CircuitBoard1->CircuitBoard
    CircuitBoard
    translate  0.25*x
  }
  object { // CB_Post1->CB_Post
    CB_Post
    rotate <0.0, 0.0, -0.0>
    translate  <-0.0, -1.0, 0.0>
  }
  object { // CB_Post2->CB_Post
    CB_Post
    rotate <0.0, 0.0, -0.0>
    translate  <-0.0, -1.0, -1.2>
  }
  object { // CB_Post3->CB_Post
    CB_Post
    rotate <0.0, 0.0, -0.0>
    translate  <-0.0, 0.0, -1.2>
  }
  translate  -0.475*x
}

#end //******** End of Front Panel Macro **********************
#macro MrRobot( power, head_rotation, eye_rotation, leg_length, leg_angle, ankle_angle,
                r_shoulder_vec, r_elbow_vec, r_length, r_hand_vec, r_thumb_angle,
                l_shoulder_vec, l_elbow_vec, l_length, l_hand_vec, l_thumb_angle, color_index)

union { // Robot
  object { Torso(color_index) }
  object { HipJoint }
  object { 
    Head(power, head_rotation, eye_rotation, color_index)
    }
  object {
    Leg(leg_length, leg_angle, ankle_angle, color_index)
    }
  object { 
    RightArm(r_shoulder_vec, r_elbow_vec, r_length, r_hand_vec, r_thumb_angle, color_index)
  }
  object { 
    FrontPanel(power)
  }
  object { 
    LeftArm(l_shoulder_vec, l_elbow_vec, l_length, l_hand_vec, l_thumb_angle, color_index)
  }
}
#end